﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace kvdb {

    template<int N>
    class BoundedString
    {
    public:
        BoundedString() NN_NOEXCEPT
        {
            m_Buffer[0] = '\0';
        }

        explicit BoundedString(const char* str) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(util::Strnlen(str, N) < N);
            util::Strlcpy(m_Buffer, str, N);
        }

        const char* Get() const NN_NOEXCEPT
        {
            return m_Buffer;
        }

        operator const char*() const NN_NOEXCEPT
        {
            return Get();
        }

        size_t GetLength() const NN_NOEXCEPT
        {
            return util::Strnlen(m_Buffer, N);
        }

        BoundedString<N>& Assign(const char* str) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(util::Strnlen(str, N) < N);
            util::Strlcpy(m_Buffer, str, N);

            return *this;
        }

        BoundedString<N>& AssignFormat(const char* format, ...) NN_NOEXCEPT
        {
            std::va_list args;

            va_start(args, format);
            NN_ABORT_UNLESS(util::VSNPrintf(m_Buffer, N, format, args) < N);
            va_end(args);

            return *this;
        }

        BoundedString<N>& Append(char c) NN_NOEXCEPT
        {
            auto length = GetLength();
            NN_ABORT_UNLESS(length + 1 < N);
            m_Buffer[length] = c;
            m_Buffer(length + 1) = '\0';

            return *this;
        }

        BoundedString<N>& Append(const char* subStr) NN_NOEXCEPT
        {
            auto length = GetLength();
            NN_ABORT_UNLESS(length + util::Strnlen(subStr, N) < N);
            std::strncat(m_Buffer, subStr, N - length - 1);

            return *this;
        }

        BoundedString<N>& AppendFormat(const char* format, ...) NN_NOEXCEPT
        {
            auto length = GetLength();
            std::va_list args;

            va_start(args, format);
            NN_ABORT_UNLESS(util::VSNPrintf(&m_Buffer[length], N - length, format, args) < static_cast<int>(N - length));
            va_end(args);

            return *this;
        }

        bool Equals(const char* str, size_t offset = 0) const NN_NOEXCEPT
        {
            return std::strncmp(&m_Buffer[offset], str, N - offset) == 0;
        }

        bool EqualsPostfix(const char* postfix) const NN_NOEXCEPT
        {
            auto postLen = static_cast<size_t>(util::Strnlen(postfix, N));
            auto length = GetLength();
            if (postLen > length)
            {
                return false;
            }
            auto offset = length - postLen;

            return Equals(postfix, offset);
        }

        void GetSubString(char* buffer, size_t bufferSize, size_t offset, size_t length) const NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(offset + length <= GetLength());
            NN_ABORT_UNLESS(bufferSize > length);
            std::strncpy(buffer, &m_Buffer[offset], length);
            buffer[length] = '\0';
        }

        BoundedString<N> MakeSubString(size_t offset, size_t length) const NN_NOEXCEPT
        {
            BoundedString<N> ret;
            GetSubString(ret.m_Buffer, sizeof(ret.m_Buffer), offset, length);

            return ret;
        }

        static BoundedString<N> Make(const char* str) NN_NOEXCEPT
        {
            return BoundedString<N>(str);
        }

        static BoundedString<N> MakeFormat(const char* format, ...) NN_NOEXCEPT
        {
            std::va_list args;

            BoundedString<N> ret;
            va_start(args, format);
            ret.AssignFormatArgs(format, args);
            va_end(args);

            return ret;
        }

    private:
        BoundedString<N>& AssignFormatArgs(const char* format, std::va_list args) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(util::VSNPrintf(m_Buffer, N, format, args) < N);

            return *this;
        }

        char m_Buffer[N];
    };

}}
