﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ホスト・ターゲット通信デバッグ用 API
 */

#pragma once

#include <nn/nn_Common.h>

#include <nn/sf/sf_ObjectFactory.h>

#include <nn/htclow/detail/htclow_DebugTypes.h>

namespace nn { namespace htclow { namespace detail {

class IDebugServiceObject;

/**
 * @brief   htclow デバッグ用 API の初期化
 *
 * @details デバッグ用 API を実行するための htclow ライブラリの初期化を行います。
 *          通常用 API の初期化は、この関数とは別に実行する必要があります。
 */
void InitializeForDebugApi(nn::sf::SharedPointer<IDebugServiceObject>&& manager) NN_NOEXCEPT;

/**
 * @brief   サービスオブジェクトを指定して htclow デバッグ API を初期化
 *
 * @pre     デバッグ用 API の初期化を実行していない
 * @details デバッグ用 API を実行するための htclow ライブラリの初期化を行います。
 *          通常用 API の初期化は、この関数とは別に実行する必要があります。
 */
void InitializeForDebugApiWith(nn::sf::SharedPointer<IDebugServiceObject>&& manager) NN_NOEXCEPT;

/**
 * @brief   テストから htclow ライブラリにデータを送信
 *
 * @pre     デバッグ用 API の初期化を実行済
 * @details htclow ライブラリの DebugDriver の受信バッファにデータを書き込みます。
 *          テストから htclow ライブラリにデータを送信するのに用います。
 */
nn::Result SendFromTest(const void* pBuffer, size_t sendSize) NN_NOEXCEPT;

/**
 * @brief   テストが htclow ライブラリからデータを受信
 *
 * @pre     デバッグ用 API の初期化を実行済
 * @details htclow ライブラリの DebugDriver の送信バッファからデータを読み出します。
 *          テストが htclow ライブラリからデータを受信するのに用います。
 */
nn::Result ReceiveFromTest(void* pBuffer, size_t receiveSize) NN_NOEXCEPT;

/**
 * @brief   指定したドライバの使用を開始する
 *
 * @pre     - デバッグ用 API の初期化を実行済
 *          - いずれのドライバも使用していない
 * @post    driverType で指定したドライバの使用を開始
 *
 * @detail  driverType で指定したドライバの使用を開始します。
 *          動的通信路切替を実現するため、将来的にはデバッグ用 API から通常用 API に格上げ予定です。
 */
nn::Result OpenDriver(DriverType driverType) NN_NOEXCEPT;

/**
 * @brief   ドライバの使用を終了する
 *
 * @pre     デバッグ用 API の初期化を実行済
 *
 * @post    いずれのドライバも使用していない
 *
 * @detail  driverType で指定したドライバの使用を開始します。
 *          動的通信路切替を実現するため、将来的にはデバッグ用 API から通常用 API に格上げ予定です。
 */
void CloseDriver() NN_NOEXCEPT;

/**
 * @brief   デフォルトアロケータのメモリ使用量を取得する
 *
 * @pre     デバッグ用 API の初期化を実行済
 *
 * @detail  htclow 内部のデフォルトアロケータが使用しているメモリサイズを返します。
 */
size_t GetDefaultAllocatorUsage() NN_NOEXCEPT;

/**
 * @brief   デフォルトアロケータのメモリ使用量を出力する
 *
 * @pre     デバッグ用 API の初期化を実行済
 *
 * @detail  htclow 内部のデフォルトアロケータが使用しているメモリサイズをログ出力します。
 */
void PrintDefaultAllocatorUsage() NN_NOEXCEPT;

}}}
