﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/gc/detail/gc_Types.h>

#define NN_DETAIL_GC_SDK_REQUIRES(x, ...) \
    { \
        NN_SDK_REQUIRES(x, ##__VA_ARGS__); \
        if((x) == false) \
        { \
            return fs::ResultGameCardPreconditionViolation(); \
        } \
    }

#define NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(x, ...) \
    { \
        NN_SDK_REQUIRES(x, ##__VA_ARGS__); \
        if((x) == false) \
        { \
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ResultGameCardPreconditionViolation()); \
        } \
    }

namespace nn { namespace gc {
namespace detail {

template<typename T, T unitValue>
inline T CalcUnitCeiling(const T value) NN_NOEXCEPT
{
    T remainder = value % unitValue;
    return (value - remainder) + static_cast<T>(remainder != 0) * unitValue;
}

inline void MemcpyFromU32(char* outBuffer, u32 data) NN_NOEXCEPT
{
    char* localDataBuffer = reinterpret_cast<char*>(&data);
    std::memcpy(outBuffer, localDataBuffer, sizeof(u32));
}

inline void BigEndianCopy(u8* outBuffer, const u32 data, const size_t length) NN_NOEXCEPT
{
    const int bitNum = 8;
    for(size_t i = 0; i < length; i++)
    {
        outBuffer[i] = static_cast<u8>( (data >> (bitNum * (length - 1 - i))) & 0xFF );
    }
}

inline void BigEndianCopy3Byte(u8* outBuffer, const u32 data) NN_NOEXCEPT
{
    BigEndianCopy(outBuffer, data, 3);
}

inline void BigEndianCopyU32(u8* outBuffer, const u32 data) NN_NOEXCEPT
{
    BigEndianCopy(outBuffer, data, sizeof(u32));
}

int CompareAllMemory(const void* buffer1, const void* buffer2, const size_t bufferLength) NN_NOEXCEPT;

inline void CallGcCallback(GcCallbackStruct* callback) NN_NOEXCEPT
{
    if(callback->m_pCallbackFunction != nullptr)
    {
        callback->m_pCallbackFunction(callback->m_pParameter);
    }
}

inline void InitializeGcCallback(GcCallbackStruct* callback) NN_NOEXCEPT
{
    callback->m_pCallbackFunction = nullptr;
    callback->m_pParameter = nullptr;
}

// 0 埋め関数のコードコピー(memset による 0 埋め最適化対策)
// memset_s に置き換え予定
inline void SecureMemoryZero(void* addr, size_t size) NN_NOEXCEPT
{
    auto ptr = static_cast<volatile char*>(addr);
    for (size_t i = 0; i < size; ++i)
    {
        ptr[i] = 0;
    }
}

class Time
{
public:
    Time() : m_StartTime(0), m_EndTime(0), m_Ticks(0) {}
    ~Time() {}

    void Start()
    {
        m_StartTime = nn::os::GetSystemTick();
    }
    void Stop()
    {
        m_Ticks = GetElapsedTime();
    }
    bool IsInitialState()
    {
        return m_StartTime.GetInt64Value() == 0;
    }
    int64_t GetRecordedTime()
    {
        return m_Ticks;
    }
    int64_t GetElapsedTime()
    {
        m_EndTime = nn::os::GetSystemTick();
        return (m_EndTime - m_StartTime).ToTimeSpan().GetMicroSeconds();
    }

private:
    nn::os::Tick m_StartTime;
    nn::os::Tick m_EndTime;
    int64_t m_Ticks;
};

// TODO: 不必要コードはビルドオプションで切る

class QueueBuffer
{
    NN_DISALLOW_COPY(QueueBuffer);

private:
    static char* g_CommonDataBuffer;

public:
    char* dataBuffer;
    size_t dataBufferLength;
    size_t length;

public:
    QueueBuffer() NN_NOEXCEPT
    {
        Initialize();
    }

    QueueBuffer(const char* buffer1, const size_t buffer1Length, const char* buffer2, const size_t buffer2Length) NN_NOEXCEPT
    {
        Initialize();
        Append(buffer1, buffer1Length);
        Append(buffer2, buffer2Length);
    }

    QueueBuffer(char* workBuffer, const size_t bufferLength) NN_NOEXCEPT
    {
        Initialize();
        Append(workBuffer, bufferLength);
    }

    void Reset() NN_NOEXCEPT
    {
        length = 0;
    }

    void Append(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;

private:
    void Initialize() NN_NOEXCEPT
    {
        length = 0;
        dataBuffer = g_CommonDataBuffer;
        dataBufferLength = sizeof(g_CommonDataBuffer);
    }
};



template <typename T>
class RingBuffer
{
    NN_DISALLOW_COPY(RingBuffer);

NN_DETAIL_GC_PROTECTED:
    T* m_Buffer;
    size_t m_BufferLength;
    size_t m_AtIndex;
    size_t m_Length;

protected:
    RingBuffer() NN_NOEXCEPT
    {
        Init();
        m_Buffer = nullptr;
        m_BufferLength = 0;
    }

public:
    RingBuffer(T* dataBuffer, const size_t dataBufferLength) NN_NOEXCEPT;
    ~RingBuffer() NN_NOEXCEPT
    {
    }
    size_t GetBufferLength() NN_NOEXCEPT
    {
        return m_BufferLength;
    }
    void Enqueue(T data) NN_NOEXCEPT
    {
        Enqueue(&data, 1);
    }
    void Enqueue(T* dataBuffer, const size_t dataBufferLength) NN_NOEXCEPT;
    size_t Dequeue(T* pOutValue) NN_NOEXCEPT
    {
        return Dequeue(pOutValue, 1);
    }
    size_t Dequeue(T* outBuffer, size_t outBufferLength) NN_NOEXCEPT;
    size_t DeleteHead(const size_t length) NN_NOEXCEPT;
    size_t DeleteTail(const size_t length) NN_NOEXCEPT;

    void Clear() NN_NOEXCEPT
    {
        m_Length = 0;
    }
    size_t GetLength() NN_NOEXCEPT
    {
        return m_Length;
    }
    bool IsEmpty() NN_NOEXCEPT
    {
        return m_Length == 0;
    }
    void Print() NN_NOEXCEPT;

    // リングバッファが回りきって上書きされた可能性があるか
    bool IsFull() NN_NOEXCEPT
    {
        return m_Length == m_BufferLength;
    }

    size_t GetHead(T* outBuffer, const size_t outBufferLength) NN_NOEXCEPT;
    size_t GetTail(T* outBuffer, const size_t outBufferLength) NN_NOEXCEPT;

NN_DETAIL_GC_PROTECTED:
    void Init() NN_NOEXCEPT
    {
        m_Length = 0;
        m_AtIndex = 0;
    }
    size_t GetIndexWithOffset(const size_t offset)
    {
        return (m_AtIndex + NormalizeValue(offset)) % m_BufferLength;
    }
    size_t GetIndexWithInverseOffset(const size_t offset)
    {
        return (m_AtIndex + ConvertMinusValue(offset)) % m_BufferLength;
    }
    size_t GetStartIndex()
    {
        return GetIndexWithInverseOffset( m_Length );
    }
    size_t GetStartIndexWithOffset(const size_t offset)
    {
        return (m_AtIndex - m_Length + NormalizeValue(offset)) % m_BufferLength;
    }
    size_t GetStartIndexWithInverseOffset(const size_t offset)
    {
        return (m_AtIndex - m_Length + ConvertMinusValue(offset)) % m_BufferLength;
    }
    size_t ConvertMinusValue(const size_t value)
    {
        return m_BufferLength - NormalizeValue(value);
    }
    size_t NormalizeValue(const size_t value)
    {
        return (value % m_BufferLength);
    }
    size_t GetMinimumLength(const size_t length)
    {
        return m_Length < length ? m_Length : length;
    }
};


class CharRingBuffer : public RingBuffer<char>
{
    NN_DISALLOW_COPY(CharRingBuffer);

public:
    CharRingBuffer() NN_NOEXCEPT;

    void SetBuffer(char* dataBuffer, const size_t dataBufferLength) NN_NOEXCEPT
    {
        m_Buffer = dataBuffer;
        m_BufferLength = dataBufferLength;
    }
    void AddNullByteToLast() NN_NOEXCEPT
    {
        char c = 0;
        Enqueue(&c, 1);
    }
    void RemoveLastNullByte() NN_NOEXCEPT
    {
        if(m_Length > 0)
        {
            size_t index = GetIndexWithInverseOffset(1);
            if(m_Buffer[index] == 0)
            {
                m_Length--;
                m_AtIndex = index;
            }
        }
    }
    size_t DequeueLine(char* outBuffer, const size_t outBufferLength) NN_NOEXCEPT;
};

} } }
