﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstddef>
#include <nn/nn_BitTypes.h>
#include <nn/util/util_BitPack.h>

#ifdef NN_DETAIL_TEST_GC
#define NN_DETAIL_GC_PRIVATE public
#define NN_DETAIL_GC_PROTECTED public
#else
#define NN_DETAIL_GC_PRIVATE private
#define NN_DETAIL_GC_PROTECTED protected
#endif


namespace nn { namespace gc {

struct RmaInformation
{
    uint8_t pairNo[8];
    uint8_t otpTestForbidFlag;
    uint8_t reserved1[3];
    uint8_t otpTestArea1[2];
    uint8_t reserved2[2];
    uint8_t otpTestArea2[16];
    uint8_t otpTestArea3[32];
    uint8_t hardwareVersion[4];
    uint8_t reserved3[4];
    uint8_t flagAndVersion[8];
    uint8_t reserved4[432];
};

// *** detail ***
namespace detail {

typedef uint8_t u8;
typedef uint16_t u16;
typedef uint32_t u32;
typedef uint64_t u64;

#define U32SIZEOF(a)    (sizeof(a)/sizeof(u32))

typedef enum : u8
{
    AsicFirmwareType_Read,
    AsicFirmwareType_Write,
} AsicFirmwareType;

typedef enum : u8
{
    AsicOperationIndex_TRANS_FW                        = 0x01,
    AsicOperationIndex_RD_ASICPUBKEY                   = 0x02,
    AsicOperationIndex_WR_SOCPUBKEY                    = 0x03,
    AsicOperationIndex_RD_RND1                         = 0x04,
    AsicOperationIndex_WR_RND2                         = 0x05,
    AsicOperationIndex_RD_RND3                         = 0x06,
    AsicOperationIndex_WR_RESPONSE1                    = 0x07,
    AsicOperationIndex_WR_RND4                         = 0x08,
    AsicOperationIndex_RD_RESPONSE2                    = 0x09,
    AsicOperationIndex_CHG_SECURE                      = 0x0A,
    AsicOperationIndex_WRITE_REG                       = 0x0B,
    AsicOperationIndex_READ_REG                        = 0x0C,
    AsicOperationIndex_CARD_INIT_DEBUG                 = 0x0D,
    AsicOperationIndex_CARD_INIT                       = 0x0E,
    AsicOperationIndex_CARD_CHG_SECURE                 = 0x0F,
    AsicOperationIndex_CARD_CMDOUT                     = 0x10,
    AsicOperationIndex_CARD_BUS_ENABLE                 = 0x11,
    AsicOperationIndex_EXCHANGE_KEY                    = 0x12,
    AsicOperationIndex_RD_RMA                          = 0x13,
} AsicOperationIndex;

typedef enum : u8
{
    DataDirection_NoData,
    DataDirection_Write,
    DataDirection_Read
} DataDirection;

typedef enum : u8
{
    // ノーマルモードコマンド
    CardCommandId_NormalReadId1                        = 0x56,
    CardCommandId_NormalReadId2                        = 0x28,
    CardCommandId_NormalReadId3                        = 0xA5,
    CardCommandId_NormalReadPage                       = 0x5B,
    CardCommandId_NormalReadSelfRefresh                = 0x06,
    CardCommandId_NormalReadRefreshStatus              = 0xFA,

    // T1/T2 モードコマンド
    CardCommandId_SecureReadId1                        = 0x67,
    CardCommandId_SecureReadId2                        = 0xC4,
    CardCommandId_SecureReadId3                        = 0x30,
    CardCommandId_SecureReadUid                        = 0x0F,
    CardCommandId_SecureReadRefresh                    = 0xB8,
    CardCommandId_SecureReadPage                       = 0x21,
    CardCommandId_SecureWritePage                      = 0x83,
    CardCommandId_SecureWriteSetAccessPattern8         = 0xF1,
    CardCommandId_SecureWriteSetAccessPattern71        = 0x17,
    CardCommandId_SecureReadAccessPattern              = 0x59,
    CardCommandId_SecureWriteAccessPattern             = 0x8A,
    CardCommandId_SecureSetKey                         = 0x39,

    // デバッグモードコマンド
    CardCommandId_DebugReadId1                         = 0x10,
    CardCommandId_DebugReadId2                         = 0x11,
    CardCommandId_DebugReadId3                         = 0x12,
    CardCommandId_DebugReadUid                         = 0x13,
    CardCommandId_DebugReadCrc                         = 0x15,
    CardCommandId_DebugWritePage                       = 0x16,
    CardCommandId_DebugReadErase                       = 0x18,
    CardCommandId_DebugReadPara                        = 0x19,
    CardCommandId_DebugWritePara                       = 0x20,
    CardCommandId_DebugReadErcnt                       = 0x22,
} CardCommandId;

// MEMO: 互換性を強く意識して明示的に値を割り当てたほうがよい...？
typedef enum : u8
{
    GameCardMode_Initial,
    GameCardMode_Normal,
    GameCardMode_Secure,
    GameCardMode_Debug,
} GameCardMode;

typedef enum : u8
{
    AsicState_Initial,
    AsicState_Secure,
} AsicState;

typedef enum : u8
{
    GcDeviceStatusAsicError_None                       = 0x00,
    GcDeviceStatusAsicError_Boot                       = 0x01,
    GcDeviceStatusAsicError_BootFailure                = 0x02,
    GcDeviceStatusAsicError_Unknown                    = 0x03,
} GcDeviceStatusAsicError;

typedef enum
{
    BusPower_3_1V,
    BusPower_1_8V,
} BusPower;

typedef enum : u8
{
    DataPageSize_NONE                                  = 0x00,
    DataPageSize_4Byte                                 = 0x01,
    DataPageSize_64Byte                                = 0x02,
    DataPageSize_512Byte                               = 0x03,
} DataPageSize;

typedef enum : u8
{
    DataTransDirectionForCard_Read                     = 0x00,
    DataTransDirectionForCard_Write                    = 0x01
} DataTransDirectionForCard;

typedef enum : u8
{
    MakerCodeForCardId1_MegaChips                      = 0xc2,
    MakerCodeForCardId1_Lapis                          = 0xae
} MakerCodeForCardId1;

typedef enum : u8
{
    MemoryCapacity_1GB                                 = 0xFA,
    MemoryCapacity_2GB                                 = 0xF8,
    MemoryCapacity_4GB                                 = 0xF0,
    MemoryCapacity_8GB                                 = 0xE0,
    MemoryCapacity_16GB                                = 0xE1,
    MemoryCapacity_32GB                                = 0xE2,
} MemoryCapacity;

typedef enum : u8
{
    MemoryType_T1RomFast                               = 0x01,
    MemoryType_T2RomFast                               = 0x02,
    MemoryType_T1NandFast                              = 0x09,
    MemoryType_T2NandFast                              = 0x0A,
    MemoryType_T1RomLate                               = 0x21,
    MemoryType_T2RomLate                               = 0x22,
    MemoryType_T1NandLate                              = 0x29,
    MemoryType_T2NandLate                              = 0x2A,
} MemoryType;

typedef enum : u8
{
    MakerCodeForCardUid_MegaChips                      = 0x02,
    MakerCodeForCardUid_Lapis                          = 0x01
} MakerCodeForCardUid;

typedef enum : u8
{
    CardTypeForUid_WritableProd                        = 0xFF,
    CardTypeForUid_WritableDev                         = 0xFE,
} CardTypeForUid;

typedef enum : u8
{
    SelfRefreshResponse_NoNeeded                       = 0x00,
    SelfRefreshResponse_Started                        = 0x01,
} SelfRefreshResponse;

typedef enum : u8
{
    SelfRefreshProgress_Success                        = 0x00,
    SelfRefreshProgress_Executing                      = 0x01,
    SelfRefreshProgress_Failure                        = 0x03,
    SelfRefreshProgress_NoExecute                      = 0xFF,
} SelfRefreshProgress;

typedef enum : u8
{
    CardErrorState_NoError                             = 0x00,
    CardErrorState_Error                               = 0x01,
} CardErrorState;

typedef enum : u8
{
    CardSecurityNumber_0                               = 0x00,
    CardSecurityNumber_1                               = 0x01,
    CardSecurityNumber_2                               = 0x02,
    CardSecurityNumber_3                               = 0x03,
    CardSecurityNumber_4                               = 0x04,
} CardSecurityNumber;

typedef enum : u8
{
    CardType_Rom                                       = 0x00,
    CardType_Writable_Dev_T1                           = 0x01,
    CardType_Writable_Prod_T1                          = 0x02,
    CardType_Writable_Dev_T2                           = 0x03,
    CardType_Writable_Prod_T2                          = 0x04,
} CardType;

typedef enum : u8
{
    DevCardRomSize_128MB                               = 0x00,
    DevCardRomSize_256MB                               = 0x01,
    DevCardRomSize_512MB                               = 0x02,
    DevCardRomSize_1GB                                 = 0x03,
    DevCardRomSize_2GB                                 = 0x04,
    DevCardRomSize_4GB                                 = 0x05,
    DevCardRomSize_8GB                                 = 0x06,
    DevCardRomSize_16GB                                = 0x07,
    DevCardRomSize_32GB                                = 0x08,
    DevCardRomSize_64GB                                = 0x09,
} DevCardRomSize;

typedef enum : u8
{
    DevCardNandSize_128MB                              = 0x00,
    DevCardNandSize_256MB                              = 0x01,
    DevCardNandSize_512MB                              = 0x02,
    DevCardNandSize_1GB                                = 0x03,
    DevCardNandSize_2GB                                = 0x04,
    DevCardNandSize_4GB                                = 0x05,
    DevCardNandSize_8GB                                = 0x06,
    DevCardNandSize_16GB                               = 0x07,
    DevCardNandSize_32GB                               = 0x08,
    DevCardNandSize_64GB                               = 0x09,
} DevCardNandSize;

typedef enum : u8
{
    SelectSecurityMode_T1 = 0x01,
    SelectSecurityMode_T2 = 0x02,
} SelectSecurityMode;

struct CardId1
{
    MakerCodeForCardId1  makerCode;
    MemoryCapacity       memoryCapacity;
    u8                   reserve;
    MemoryType           memoryType;
};

struct CardId2
{
    CardSecurityNumber   cardSecurityNumber;
    CardType             cardType;
    u8                   reserve[2];
};

struct CardId3
{
    u8                   reserve[4];
};

struct CardUid
{
    MakerCodeForCardUid  makerCode;
    u8                   version;
    CardTypeForUid       cardType;
    u8                   uniqueData[9];
    u32                  random;
    u8                   platformFlag;// 0x00 決め打ち
    u8                   reserve[11];
    CardId1              cardId1;
    u8                   mac[32];
};

struct SelfRefreshStatus
{
    SelfRefreshProgress  selfRefreshProgress;
    u8                   totalSuccessNum; // 0 - 0x0f
    CardErrorState       errorState;//過去一度でもエラーありなら 1
    u8                   reserve;//0
};

struct CardHeader
{
    u32 magicCode;
    u32 romAreaStartPageAddress;
    u32 backupAreaStartPageAddress;
    nn::util::BitPack8 keyIndex;
    u8 romSize;
    u8 version;
    u8 flags;
    u8 packageId[8];
    u32 validDataEndAddress;
    u8 reserved2[4];
    u8 iv[0x10];
    uint64_t partitionFsHeaderAddress;
    uint64_t partitionFsHeaderSize;
    u8 partitionFsHeaderHash[0x20];
    u8 initialDataHash[0x20];
    u32 selSec;
    u32 selT1Key;
    u32 selKey;
    u32 limArea;
    // 以下は元々暗号化されている領域
    u32 fwVersion[2];
    u32 accCtrl1;
    u32 wait1TimeRead;
    u32 wait2TimeRead;
    u32 wait1TimeWrite;
    u32 wait2TimeWrite;
    u32 fwMode;
    u32 cupVersionReserve;
    u8 reserved4[4];
    u8 uppHashForLotCheck[8];   // これはドライバでは使用しない（バイナリ取扱い時の運用のための保存場所）
    u64 cupId;
    u8 reserved5[0x38];
};

struct CardHeaderMap_KeyIndex
{
    typedef nn::util::BitPack8::Field<0, 4, u8> kekIndex;
    typedef nn::util::BitPack8::Field<4, 4, u8> titleKeyDecIndex;
};

static const int CardDeviceIdLength = 16;

// 0x200 + 0x200 padding : 512 + 512 bytes
struct T1CardCertificate
{
    u8 signature[0x100];
    u8 magicCode[4];
    u32 version;
    u8 kekIndex;
    u8 flags[7];
    u8 t1CardDeviceId[CardDeviceIdLength];
    u8 iv[16];
    u8 hwKey[16];
    u8 reserve[0xC0];
    // * ----------- *
    u8 padding[0x200];
};

struct T2CardCertificate
{
    u8 reserve[1024];
};

struct CardSecurityInformation
{
    u32 memoryInterfaceMode;
    u32 asicStatus;
    CardId1 id1;
    CardId2 id2;
    CardUid uid;
    u8 reserved[432];
    union
    {
        u8 cardCerificate[1024];
        T1CardCertificate t1CardCertificate;
        T2CardCertificate t2CardCertificate;
    };
    union
    {
        u8 initialDataArray[512];
        struct
        {
            u8 payload[60];
            u8 reserved[452];
        } initialData;
    };
};

struct CardInitReceiveData
{
    u32 cupVersion;
    CardId1 cardId1;
};

struct DevCardErrorCount
{
    u8                   errorCount[24];
    u8                   reserve[40];
};

struct DevCardParameter
{
    CardId1              cardId1;// ここで指定した ID1 の値になる。romSize に紐づいて変更が必要
    CardId2              cardId2;
    CardId3              cardId3;
    u32                  romAreaStartAddr; // Secure Area の開始アドレス。 32KB 単位。
    u32                  backupAreaStartAddr;// 未使用
    u8                   reserveAreaStartAddr[3];// Unused Area の開始アドレス。 8MB 単位。
    DevCardRomSize       romSize;// ここで指定した容量のカードの振る舞いをする
    u8                   waitCycle1ForRead[2];
    u8                   waitCycle2ForRead[2];
    u8                   speedChangeEmurateWaitCycle1FrecencyForRead;
    u8                   speedChangeEmurateWaitCycle1ForRead[3];
    u8                   speedChangeEmurateWaitCycle2FrecencyForRead;
    u8                   speedChangeEmurateWaitCycle2ForRead[3];
    u8                   firstReadPageWaitCycleForRead[3];
    u8                   waitCycle1ForWrite[2];
    u8                   waitCycle2ForWrite[3];
    u8                   speedChangeEmurateWaitCycle1FrecencyForWrite;
    u8                   speedChangeEmurateWaitCycle1ForWrite[3];
    u8                   speedChangeEmurateWaitCycle2FrecencyForWrite;
    u8                   speedChangeEmurateWaitCycle2ForWrite[3];
    u8                   waitCycle1ForSetAccessPattern[2];
    u8                   waitCycle2ForSetAccessPattern[3];
    u8                   waitCycleForRefresh[3];
    u8                   waitCycleForSetKey[3];
    u8                   waitCycleForIrdInit[3];
    u8                   waitCycleForIsetInit1[3];
    u8                   waitCycleForISetGen[3];
    u8                   waitCycleForIsetInit2[3];
    DevCardNandSize      nandSize;// Writable カード自体の容量。Read Only。
    u8                   reserve[436];
};

struct RefreshResponse
{
    u8                   randomValue[32];
    u8                   reserve[32];
};

enum AsicWork
{
    AsicWork_None,
    AsicWork_Activate,
    AsicWork_Deactivate,
    AsicWork_SetCardToSecureMode,
    AsicWork_PutToSleep,
    AsicWork_Awaken,
    AsicWork_Read,
    AsicWork_GetCardStatus,
    AsicWork_GetDeviceId,
    AsicWork_GetDeviceCertificate,
    AsicWork_GetCardImageHash,
    AsicWork_GetCardIdSet,
    AsicWork_GetCardHeader,
    AsicWork_IsCardActivationValid,
    AsicWork_GetErrorInfo,
};

typedef struct
{
    char* outCounterBuffer;
    char* outDataBuffer;
    size_t counterBufferLength;
    size_t dataBufferSize;
    uint32_t pageAddress;
    uint32_t pageCount;
} ReadInfo;

typedef struct
{
    AsicWork workIndex;
    int lockerIndex;
    bool hasLocker;
    ReadInfo* pReadInfo;
} AsicWorkInfo;

typedef void (*GcCallbackFunctionPointer)(void* pParameter);

typedef struct
{
    GcCallbackFunctionPointer m_pCallbackFunction;
    void* m_pParameter;

} GcCallbackStruct;

typedef void (*UtilOutputCallbackFunctionPointer)(char* buffer, size_t bufferLength);

} } }

