﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gc/detail/gc_Types.h>

namespace nn { namespace gc {
namespace writer
{
    class GcWriter;
}

namespace detail {

class StateMachine
{
    NN_DISALLOW_COPY(StateMachine);

private:
    AsicFirmwareType m_AsicFirmwareType;
    AsicState m_AsicState;      // ASIC の状態：AsicHandler が把握している限りで常に正しい状態が入る
    GameCardMode m_GameCardMode;  // GC の状態：AsicHandler が把握している限りで常に正しい状態が入る

    bool m_IsForceReset;
    bool m_IsCommonKeyReady;
    bool m_IsKeyUpdateReady;

private:
    StateMachine() NN_NOEXCEPT;

    // 基本的に変数への書き込みは StateMachine ではなく StateMachineFor*** 内に書く
public:
    ~StateMachine() NN_NOEXCEPT;
    static StateMachine& GetInstance() NN_NOEXCEPT;

    GameCardMode GetGameCardMode() NN_NOEXCEPT;
    AsicState GetAsicState() NN_NOEXCEPT;

    bool IsCommonKeyReady() NN_NOEXCEPT
    {
        return m_IsCommonKeyReady;
    }
    bool IsKeyUpdateReady() NN_NOEXCEPT
    {
        return m_IsKeyUpdateReady;
    }
    bool IsForceReset() NN_NOEXCEPT
    {
        return m_IsForceReset;
    }
    // ** FW 系
    bool IsReaderFwSet() NN_NOEXCEPT
    {
        return m_AsicFirmwareType == AsicFirmwareType_Read;
    }
    bool IsWriterFwSet() NN_NOEXCEPT
    {
        return m_AsicFirmwareType == AsicFirmwareType_Write;
    }
    // ** 状態系
    bool IsCardInserted() NN_NOEXCEPT;  // GPIO 経由で直接見る
    bool IsCardActivated() NN_NOEXCEPT
    {
        return (m_GameCardMode != GameCardMode_Initial);
    }
    bool IsCardSecure() NN_NOEXCEPT
    {
        return (m_GameCardMode == GameCardMode_Secure);
    }
    bool IsAsicSecure() NN_NOEXCEPT
    {
        return m_AsicState == AsicState_Secure;
    }
    bool IsCardDebug() NN_NOEXCEPT
    {
        return m_GameCardMode == GameCardMode_Debug;
    }


public:
    // NOTE: 複数スレッドによる State 管理は危険性が高いので、サブクラスを使って特定クラスへのアクセス制限する運用をとる

    // AsicHandler スレッドから操作できるもの
    class StateMachineForAsicHandler
    {
        NN_DISALLOW_COPY(StateMachineForAsicHandler);
        friend class AsicHandler;
        friend class AsicHandlerCore;
        friend class AsicHandlerProcedure;
        friend class writer::GcWriter;
        friend class StateMachine;

    private:
        StateMachine* m_pBase;

    public:
        explicit StateMachineForAsicHandler(StateMachine* base) NN_NOEXCEPT;

    // フレンドクラスに呼んでもらうもの
    NN_DETAIL_GC_PROTECTED:
        void ResetState() NN_NOEXCEPT
        {
            m_pBase->m_IsForceReset = false;
            m_pBase->m_IsCommonKeyReady = false;
            m_pBase->m_IsKeyUpdateReady = false;
            m_pBase->m_AsicState = AsicState_Initial;
            m_pBase->m_GameCardMode = GameCardMode_Initial;
        }
        void SetCommonKeyReady(bool isKeyReady) NN_NOEXCEPT
        {
            m_pBase->m_IsCommonKeyReady = isKeyReady;
        }
        void SetKeyUpdateReady(bool isKeyReady) NN_NOEXCEPT
        {
            m_pBase->m_IsKeyUpdateReady = isKeyReady;
        }
        void SetForceResetFlag() NN_NOEXCEPT
        {
            m_pBase->m_IsForceReset = true;
        }
        void SelectAsicFirmware(AsicFirmwareType asicFirmwareType) NN_NOEXCEPT;

        nn::Result TransitStateWithSecureAsic() NN_NOEXCEPT;
        nn::Result TransitStateWithResetAsic() NN_NOEXCEPT;

        nn::Result TransitStateWithActivateGameCard() NN_NOEXCEPT;
        nn::Result TransitStateWithSecureGameCard() NN_NOEXCEPT;
        nn::Result TransitStateWithDebugGameCard() NN_NOEXCEPT;
        nn::Result TransitStateWithInvalidateGameCard() NN_NOEXCEPT;

    // 外から呼んでほしくないもの
    private:
        nn::Result ProcessTransitState(GameCardMode fromState, GameCardMode toState) NN_NOEXCEPT;
    };

private:
    // アクセス制限を施した子クラス
    StateMachineForAsicHandler* stateThread;

public:
    static StateMachineForAsicHandler& GetInstanceForAsicHandler() NN_NOEXCEPT;
};

} } }
