﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstring>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Util.h>


namespace nn { namespace gc {
namespace detail {


void PrintDebugArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
bool IsGcThread() NN_NOEXCEPT;

#ifdef GC_DETAIL_LOG_ENABLE

class InstanceLogger
{
private:
    int m_LineNumber;
    char m_FunctionNameBuffer[256];
    char m_FileNameBuffer[256];
    char m_TargetInWord[64];
    char m_TargetOutWord[64];
    char m_TargetWord[128];
    static int g_FunctionIndent[2];
    static const int IndentCoef = 2;

public:
    InstanceLogger(const char* functionName, const int lineNumber, const char* fileName) NN_NOEXCEPT
    {
        Reset(functionName, lineNumber, fileName);
        PrintIn();
    }

    InstanceLogger(const char* functionName, const int lineNumber, const char* fileName, const char* targetInWord, const char* targetOutWord, const char* targetWord) NN_NOEXCEPT;

    ~InstanceLogger() NN_NOEXCEPT
    {
        PrintOut();
    }

    void Reset(const char* functionName, const int lineNumber, const char* fileName) NN_NOEXCEPT;

    bool IsTargetWordSpecified() NN_NOEXCEPT
    {
        return m_TargetWord[0] == 0;
    }

    static int GetThreadIndent() NN_NOEXCEPT
    {
        return IndentCoef * g_FunctionIndent[(int)(nn::gc::detail::IsGcThread())];
    }

    void PrintIn() NN_NOEXCEPT;
    void PrintOut() NN_NOEXCEPT;
};
#endif

#ifdef GC_DETAIL_LOG_TO_MEMORY_ENABLE
class MemoryLogger
{
public:
    static const int RingBufferForLogSize = 4 * 1024 * 1024;
    static const int NumberOfRingBufferForLog = 2;
    CharRingBuffer m_RingBuffer[NumberOfRingBufferForLog];

private:
    uint32_t m_LogCounter;
    uint32_t m_LogStartPosition;
    mutable nn::os::Mutex m_LogCounterMutex;
    mutable nn::os::Mutex m_LogMutex;

    class LogRemainFlag
    {
    public:
        // ログがリングバッファに残っているかのフラグ
        bool remainFlag;
        uint32_t nextId;

    public:
        LogRemainFlag() NN_NOEXCEPT;
        void SetNext(CharRingBuffer* ring) NN_NOEXCEPT;
    };

public:
    static MemoryLogger& GetInstance() NN_NOEXCEPT;
    void AddLog(char* buffer, size_t bufferLength) NN_NOEXCEPT;
    void OutputLog(UtilOutputCallbackFunctionPointer callback) NN_NOEXCEPT;
    void OutputLogToConsole() NN_NOEXCEPT
    {
        OutputLog(OutputToConsole);
    }
    void ClearLog() NN_NOEXCEPT;

private:
    MemoryLogger() NN_NOEXCEPT;
    uint32_t GetCounter() NN_NOEXCEPT;
    int GetBufferIndex() NN_NOEXCEPT
    {
        return nn::gc::detail::IsGcThread() ? 0 : 1;
    }
    void AddLogImpl(char* buffer, size_t bufferLength) NN_NOEXCEPT;
    bool IsRingBufferForLogEmpty() NN_NOEXCEPT;
    static void OutputToConsole(char* buffer, size_t bufferLength) NN_NOEXCEPT;

private:
    static uint32_t GetLogId(CharRingBuffer* ring) NN_NOEXCEPT;
};
#endif

// *** マクロ
#define __file__ (strrchr(__FILE__, '\\') + 1)

// 出力先マクロ
#ifdef GC_DETAIL_LOG_TO_MEMORY_ENABLE
#define GC_NN_SDK_LOG(...) { \
    char _sdk_log_buffer[2 * 1024]; \
    size_t _sdk_log_length = nn::util::SNPrintf(_sdk_log_buffer, sizeof(_sdk_log_buffer), __VA_ARGS__); \
    nn::gc::detail::MemoryLogger::GetInstance().AddLog(_sdk_log_buffer, _sdk_log_length); \
}
#else
#ifdef NN_BUILD_CONFIG_OS_WIN
#define GC_NN_SDK_LOG(...) printf(__VA_ARGS__)
#else
#define GC_NN_SDK_LOG(...) NN_SDK_LOG(__VA_ARGS__)
#endif
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#define GC_DETAIL_TEST_BREAK_RESULT return nn::ResultSuccess()
#define GC_DETAIL_TEST_BREAK_ZERO return 0
#define GC_DETAIL_TEST_BREAK return
#else
#define GC_DETAIL_TEST_BREAK_RESULT static_cast<void>(0)
#define GC_DETAIL_TEST_BREAK_ZERO static_cast<void>(0)
#define GC_DETAIL_TEST_BREAK static_cast<void>(0)
#endif

#if defined(GC_DETAIL_LOG_FUNCTION_INDENT_ENABLE) && defined(GC_DETAIL_LOG_ENABLE)
#define NN_DETAIL_GC_GET_LOG_INDENT nn::gc::detail::InstanceLogger::GetThreadIndent()
#else
#define NN_DETAIL_GC_GET_LOG_INDENT static_cast<void>(0)
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#define NN_DETAIL_INTERNAL_LOG(fmt, ...)  printf("[%s] %*s", nn::gc::detail::IsGcThread() ? "gc-t" : " gc ", NN_DETAIL_GC_GET_LOG_INDENT, " "); printf(fmt, ##__VA_ARGS__)
#define NN_DETAIL_INTERNAL_DEBUGE_LOG(fmt, ...)  printf("[%s] %*s", nn::gc::detail::IsGcThread() ? " gc-t" : " gc ", NN_DETAIL_GC_GET_LOG_INDENT, " "); printf(fmt, ##__VA_ARGS__)
#else
extern nn::os::Tick g_Tick;
#define NN_DETAIL_INTERNAL_LOG(fmt, ...)   { \
    auto _current_tick = nn::os::GetSystemTick(); \
    int64_t emt = (_current_tick - nn::gc::detail::g_Tick).ToTimeSpan().GetMicroSeconds(); \
    GC_NN_SDK_LOG("[%s]{%2" PRId64 "ms} %*s" fmt, nn::gc::detail::IsGcThread() ? "gc-t" : " gc ", emt / 1000, NN_DETAIL_GC_GET_LOG_INDENT, nn::gc::detail::IsGcThread() ? "= " : " ", ##__VA_ARGS__); \
    nn::gc::detail::g_Tick = _current_tick; \
}
#define NN_DETAIL_INTERNAL_DEBUGE_LOG(fmt, ...)   { \
    auto _current_tick = nn::os::GetSystemTick(); \
    int64_t emt = (_current_tick - nn::gc::detail::g_Tick).ToTimeSpan().GetMicroSeconds(); \
    GC_NN_SDK_LOG("[%s]{%2" PRId64 "ms} %*s%s():%d - %s / " fmt, nn::gc::detail::IsGcThread() ? "gc-t" : " gc ", emt / 1000, NN_DETAIL_GC_GET_LOG_INDENT, nn::gc::detail::IsGcThread() ? "= " : " ", __FUNCTION__, __LINE__, __file__, ##__VA_ARGS__); \
    nn::gc::detail::g_Tick = _current_tick; \
}
#endif


// 出力マクロ（define により有効・無効を切り替え）
#ifdef GC_LOG_ENABLE
#define NN_DETAIL_GC_LOG(...)   NN_DETAIL_INTERNAL_LOG(__VA_ARGS__)
#else
#define NN_DETAIL_GC_LOG(...)   static_cast<void>(0)
#endif

#ifdef GC_DETAIL_LOG_ENABLE
#define NN_DETAIL_GC_DETAIL_LOG(...)   NN_DETAIL_INTERNAL_LOG( __VA_ARGS__)
#else
#define NN_DETAIL_GC_DETAIL_LOG(...)   static_cast<void>(0)
#endif

#ifdef GC_DETAIL_DEBUG_LOG_ENABLE
#define NN_DETAIL_GC_DEBUG_LOG(...)   NN_DETAIL_INTERNAL_DEBUGE_LOG( __VA_ARGS__ )
#else
#define NN_DETAIL_GC_DEBUG_LOG(...)   static_cast<void>(0)
#endif

#ifdef GC_DETAIL_ERR_LOG_ENABLE
#define NN_DETAIL_GC_ERR_LOG(...)   NN_SDK_LOG("Error: [gc] " __VA_ARGS__)
#else
#define NN_DETAIL_GC_ERR_LOG(...)   static_cast<void>(0)
#endif

#ifdef GC_DETAIL_WARNING_LOG_ENABLE
#define NN_DETAIL_GC_WARNING_LOG(...)   NN_SDK_LOG("Warning: [gc] " __VA_ARGS__)
#else
#define NN_DETAIL_GC_WARNING_LOG(...)   static_cast<void>(0)
#endif

#ifdef GC_DETAIL_IO_LOG_ENABLE
#define NN_DETAIL_GC_IO_LOG(...)   NN_DETAIL_INTERNAL_LOG("(IO) " __VA_ARGS__)
#else
#define NN_DETAIL_GC_IO_LOG(...)   static_cast<void>(0)
#endif

#ifdef GC_DETAIL_CRYPTO_LOG_ENABLE
#define NN_DETAIL_GC_CRYPTO_LOG(...)   NN_DETAIL_INTERNAL_LOG("(Crypto) " __VA_ARGS__)
#else
#define NN_DETAIL_GC_CRYPTO_LOG(...)   static_cast<void>(0)
#endif

#ifdef GC_DEBUG_MEMDUMP_ENABLE
#define NN_DETAIL_GC_DEBUG_MEMDUMP(buffer, bufferLength, ...)   NN_DETAIL_INTERNAL_LOG("(dump) " __VA_ARGS__); \
    nn::gc::detail::PrintDebugArray(buffer, bufferLength)
#define NN_DETAIL_GC_DEBUG_MEMDUMP_QB(queueBuffer, ...)   NN_DETAIL_INTERNAL_LOG("(dump) " __VA_ARGS__); \
    nn::gc::detail::PrintDebugArray(queueBuffer.dataBuffer, queueBuffer.length)
#else
#define NN_DETAIL_GC_DEBUG_MEMDUMP(...)   static_cast<void>(0)
#define NN_DETAIL_GC_DEBUG_MEMDUMP_QB(...)   static_cast<void>(0)
#endif

#ifdef GC_DETAIL_LOG_ENABLE
#define NN_DETAIL_GC_THREAD_DEBUG_LOG(...) nn::gc::detail::InstanceLogger _logger(__FUNCTION__, __LINE__, __file__)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_MUTEX(target) nn::gc::detail::InstanceLogger _wordLogger(__FUNCTION__, __LINE__, __file__, "lock_wait", "unlock", target)
#else
#define NN_DETAIL_GC_THREAD_DEBUG_LOG(...) static_cast<void>(0)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_MUTEX(target) static_cast<void>(0)
#endif
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE  NN_DETAIL_GC_DETAIL_LOG("%s() - %s:%d\n", __FUNCTION__, __file__, __LINE__)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_S(str)  NN_DETAIL_GC_DETAIL_LOG(str " %s() - %s:%d\n", __FUNCTION__, __file__, __LINE__)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_SS(fmt, ...)  NN_DETAIL_GC_DETAIL_LOG(fmt " %s() - %s:%d\n", ##__VA_ARGS__, __FUNCTION__, __file__, __LINE__)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_MUTEX_START(target)  NN_DETAIL_GC_THREAD_DEBUG_LOG_SS("@{lock}: <%s>:", target)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_WAIT(target) NN_DETAIL_GC_THREAD_DEBUG_LOG_SS("@{event_wait}: <%s>:", target)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_WAIT_END(target) NN_DETAIL_GC_THREAD_DEBUG_LOG_SS("@{event_wait_end}: <%s>:", target)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_TRY_WAIT(target) NN_DETAIL_GC_THREAD_DEBUG_LOG_SS("@{event_try_wait}: <%s>:", target)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_SIGNAL(target) NN_DETAIL_GC_THREAD_DEBUG_LOG_SS("@{event_signal}: <%s>:", target)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_CLEAR(target) NN_DETAIL_GC_THREAD_DEBUG_LOG_SS("@{event_clear}: <%s>:", target)
#define NN_DETAIL_GC_SIGNAL_EVENT(e)  nn::os::SignalEvent((e)); \
    NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_SIGNAL(#e)
#define NN_DETAIL_GC_CLEAR_EVENT(e)  nn::os::ClearEvent((e)); \
    NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_CLEAR(#e)
#define NN_DETAIL_GC_WAIT_EVENT(e)  NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_WAIT(#e); \
    nn::os::WaitEvent((e)); \
    NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_WAIT_END(#e)
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_LOG(log)  NN_DETAIL_GC_THREAD_DEBUG_LOG_S("@{log}: \"\"" log "\"\"")
#define NN_DETAIL_GC_THREAD_DEBUG_LOG_LOGS(fmt, ...)  NN_DETAIL_GC_THREAD_DEBUG_LOG_SS("@{log}: \"\"" fmt "\"\"", ##__VA_ARGS__)

// *** RESULT

#define NN_DETAIL_GC_RESULT_THROW_IF(condition, r) \
    if (condition) \
    { \
        NN_RESULT_THROW(r); \
    }

#define NN_DETAIL_GC_RESULT_DO_LOG_RETHROW(r, result) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_DETAIL_GC_DETAIL_LOG("! Failure (Module:%d, Description:%d), escalating ...\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription()); \
            NN_RESULT_THROW(result); \
        } \
    }

#define NN_DETAIL_GC_RESULT_LOG(r) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_DETAIL_GC_DETAIL_LOG("! Failure (Module:%d, Description:%d), escalating ...\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription()); \
        } \
    }

#define NN_DETAIL_GC_RESULT_ERR_LOG(r) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_DETAIL_GC_DETAIL_LOG("! Failure (Module:%d, Description:%d), escalating ...\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription()); \
        } \
    }

#define NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(r, ...) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_DETAIL_GC_ERR_LOG(__VA_ARGS__); \
        } \
    }

#define NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(r, ...) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_DETAIL_GC_ERR_LOG(__VA_ARGS__); \
            NN_RESULT_THROW(r); \
        } \
    }

#define NN_DETAIL_GC_RESULT_DO_ERR_LOG_RETHROW_MESSAGE(r, result, ...) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_DETAIL_GC_ERR_LOG(__VA_ARGS__); \
            NN_RESULT_THROW(result); \
        } \
    }


} } }
