﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/gpio/gpio.h>

namespace nn { namespace gc {
namespace detail {

class DeviceDetector;

// GPIO, I2C 周りの全般的な IO を扱うクラス
// 注意：このクラスは AsicHandler からのみ参照する
class GeneralIo
{
    NN_DISALLOW_COPY(GeneralIo);

    friend class DeviceDetector;

private:
    nn::gpio::GpioPadSession m_GcResetPadSession;
    nn::gpio::GpioPadSession m_GcDetectPadSession;
    nn::os::SystemEventType m_DetectEvent;

    static const int GcPowerOnWaitUsec = 1500;
    static const int GcPowerOffWaitUsec = 2000;
    static const int GcPostResetWaitTimeUsec = 150; // 最低限必要なのは 50us

private:
    GeneralIo() NN_NOEXCEPT;

public:
    // Singleton パターン
    static GeneralIo& GetInstance() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    // GPIO
    nn::Result SetGpioResetPin(const bool isOn) NN_NOEXCEPT;
    // I2C
    nn::Result SetGcPower(const bool isOn) NN_NOEXCEPT;
    nn::Result SetGcPowerImpl(const bool isOn, const BusPower power) NN_NOEXCEPT;

    nn::Result ReleaseAsicReset() NN_NOEXCEPT;
    nn::Result HoldAsicReset() NN_NOEXCEPT;
    nn::Result SetGcPowerOn() NN_NOEXCEPT
    {
        return SetGcPower(true);
    }
    nn::Result SetGcPowerOff() NN_NOEXCEPT
    {
        return SetGcPower(false);
    }

    // DeviceDetector class 向けの API
NN_DETAIL_GC_PROTECTED:
    nn::os::SystemEventType* GetDetectEvent() NN_NOEXCEPT
    {
        return &m_DetectEvent;
    }

    // 挿入状態：true, 抜去状態：false
    bool GetGpioDetectPin() NN_NOEXCEPT;
    void ClearGpioDetectEvent() NN_NOEXCEPT;
    bool IsGpioDetectWakeEventActive() NN_NOEXCEPT;

// TODO: TEMP: 製品化時に消す
NN_DETAIL_GC_PRIVATE:
    bool m_IsDetectPinOnForTest;
    void SetDetectPinStateForTest(bool isOn) NN_NOEXCEPT;

    nn::Result WaitUntilDetectCardRemove(const bool isOldPlatform) NN_NOEXCEPT
    {
        return WaitGpioDetectPin(false, isOldPlatform);
    }
    nn::Result WaitUntilDetectCardInsert(const bool isOldPlatform) NN_NOEXCEPT
    {
        return WaitGpioDetectPin(true, isOldPlatform);
    }
    nn::Result WaitGpioDetectPin(const bool isOn, const bool isOldPlatform) NN_NOEXCEPT;
};

} } }
