﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_Util.h>

namespace nn { namespace gc {
namespace detail {

// ** ASIC とのセッション構築で利用する固有情報を保持するクラス **
class EmbeddedDataHolder
{
    NN_DISALLOW_COPY(EmbeddedDataHolder);
    friend class GcCrypto;

public:
    // TODO: 正しい場所に保管する
    static char g_FwReadBuffer[GcAsicFirmwareSize];
    static char g_FwWriterBuffer[GcAsicFirmwareSize];
    static char* g_pFwUserBuffer;

    // 較正情報パーティションにある SoC 証明書
    static char g_EmmcEmbeddedSocCertificate[GcCertificateSize];

NN_DETAIL_GC_PRIVATE:
    static bool g_IsDev;
    // *** PROD 向けに差し替え対象となるドライバ抱え込みの鍵・固有情報類ここから ***
    // HMAC 鍵（ドライバ抱え込みに関する議論：HWSEC-127）
    static char g_LibraryEmbeddedHmacKeyForCv[GcCrypto::GcHmacKeyLength];
    // CV 用固定値
    static char g_LibraryEmbeddedCvConstantValue[GcCrypto::GcCvConstLength];
    // *** PROD 向けに差し替え対象となるドライバ抱え込みの鍵・固有情報類ここまで ***

    // 較正情報パーティションにある SoC 秘密鍵
    static char g_EmmcEmbeddedSocRsaKeyModulus[GcCrypto::GcRsaKeyLength];

    // CA 公開鍵
    static char g_LibraryEmbeddedCa1Modulus[GcCrypto::GcCa1PublicModulusLength];
    static char g_LibraryEmbeddedCa1PublicExponent[GcCrypto::GcCa1PublicExponentLength];

public:
    static nn::Result SetLibraryEmbeddedKeys() NN_NOEXCEPT;
    static nn::Result SetEmmcEmbeddedKeys(const char* encryptedKeyBuffer, const size_t encryptedKeyBufferSize,
                                    const char* socCertBuffer, const size_t socCertBufferSize) NN_NOEXCEPT;

NN_DETAIL_GC_PRIVATE:
    static bool IsDev() NN_NOEXCEPT
    {
        return g_IsDev;
    }
    static void SetSocRsaKeyModulus(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
    static void SetSocCertificate(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
    static nn::Result DecryptGcMessageSecure(
                                    size_t*     pOutResultSize,
                                    void*       pResultBuffer,
                                    size_t      resultBufferSize,
                                    const void* pCipher,
                                    size_t      cipherSize) NN_NOEXCEPT;
    static nn::Result EncryptGcMessageSecure(
                                    char* outBuffer, const size_t outBufferLength,
                                    const char* inBuffer, const size_t inBufferLength,
                                    char* keyModulus, const size_t keyModulusLength,
                                    char* keyExponent, const size_t keyExponentLength) NN_NOEXCEPT;
    static nn::Result GetHmacForGenerateRandomValues(
                                    char* pOutResultBuffer, const size_t resultBufferLength,
                                    const char* randomValue1Buffer, const size_t randomValue1BufferLength,
                                    const char* randomValue2Buffer, const size_t randomValue2BufferLength) NN_NOEXCEPT;


#ifdef NN_GC_ENABLE_OVERWRITE_EMBEDDED_DATA
    // TEMP: テスト用に HMAC key 等をセット：他の値は public なので直接 memcpy してもらう
    static void SetHmacKeyForCvForDebug(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
    static void SetHmacKeyForAesForDebug(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
    static void SetCvConstantValueForDebug(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
    static void SetRsaOaepLabel(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
    static void SetCa1Modulus(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
    static void SetCa1PublicExponent(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;
#endif

private:
    // ドライバ直持ちの鍵類の構造体
    // NOTE: 鍵類を変更する場合は IV も変える事！
    struct ConcatenatedGcLibraryEmbeddedKeys
    {
        char encHmacKeyForCv[GcCrypto::GcHmacKeyLength];
        char encHmacKeyForKeyAndIv[GcCrypto::GcHmacKeyLength];
        char encCvConstantValue[GcCrypto::GcCvConstLength];
        char encRsaOaepLabelHash[GcCrypto::GcSha256Length];
    };
    EmbeddedDataHolder(){};
    static const ConcatenatedGcLibraryEmbeddedKeys g_LibraryLocalEmbeddedGcLibKeys[2];
    static void SelectAsicFirmware(bool isDev) NN_NOEXCEPT;
    static nn::Result DecryptoEmbeddedKeys(ConcatenatedGcLibraryEmbeddedKeys* pOutKeys, size_t outKeySize) NN_NOEXCEPT;

};


} } }
