﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Define.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/gc/detail/gc_ThreadInterface.h>

namespace nn { namespace gc {
namespace detail {

struct DeviceDetectorErrorInfo
{
    bool isInserted;
    char reserved[1];
    uint16_t insertionCount;
    uint16_t removalCount;
};

// カード挿抜を監視するスレッドのクラス
class DeviceDetector : public ThreadInterface
{
    NN_DISALLOW_COPY(DeviceDetector);

private:
    bool m_IsInserted;
    bool m_IsPaused;
    bool m_IsRemovedDuringSleep;
    DeviceDetectorErrorInfo m_DeviceDetectorErrorInfo;

    // 割り込みイベント
    nn::os::SystemEventType* m_pGpioDetectRawEvent;
    nn::os::EventType m_GpioDetectFilteredEvent;
    nn::os::EventType m_PauseRequestEvent;
    nn::os::EventType m_ResumeRequestEvent;
    nn::os::EventType m_ResumeCompleteEvent;

    // 待ち受け
    nn::os::MultiWaitHolderType m_GpioDetectRawEventHolder;
    nn::os::MultiWaitHolderType m_PauseRequestEventHolder;

    // カード抜けの際のコールバック
    GcCallbackStruct m_CardRemovalEventCallback;

    // スレッド設定
    static const size_t ThreadStackSize = 16 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[ ThreadStackSize ];

public:
    static const int GcDetectDebounceTimeMsec = 128;

public:
    // NOTE: 先に GeneralIo の初期化が必須
    static DeviceDetector& GetInstance() NN_NOEXCEPT;
    void InitializeThread() NN_NOEXCEPT;
    nn::os::EventType* GetDetectEvent() NN_NOEXCEPT
    {
        return &m_GpioDetectFilteredEvent;
    }
    bool GetGpioDetectPin() NN_NOEXCEPT;
    void ClearGpioDetectEvent() NN_NOEXCEPT;

    // スレッドの一時停止（ブロッキング API）
    void Pause() NN_NOEXCEPT;
    bool Resume() NN_NOEXCEPT;
    void ResetPauseState() NN_NOEXCEPT;

    // スリープを制御する GcCore のみこの API を呼ぶ
    void ManualSignalGpioDetectEventForGcCore() NN_NOEXCEPT;

    // カード抜去時のコールバック登録
    void RegisterCardRemovalEventCallback(const GcCallbackStruct* pCardRemovalEventCallback) NN_NOEXCEPT;
    void UnregisterCardRemovalEventCallback() NN_NOEXCEPT;

    void GetDeviceDetectorErrorInfo(DeviceDetectorErrorInfo* pOutDeviceDetectorErrorInfo) NN_NOEXCEPT;

private:
    DeviceDetector() NN_NOEXCEPT;
    virtual void InitializeWaitEvents() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeWaitEvents() NN_NOEXCEPT NN_OVERRIDE;
    virtual void RunThreadFunctionImpl() NN_NOEXCEPT NN_OVERRIDE;
    void WaitForResume() NN_NOEXCEPT;
    void SetDetectStateAndSignalEvent(bool isInserted) NN_NOEXCEPT;
    // 状態が変わっていたらシグナルする関数：メインの処理外で挿抜ハンドリングする場合
    void CheckDetectStateChangeAndSignalEvent() NN_NOEXCEPT;
};

} } }
