﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_Util.h>

namespace nn { namespace gc {
namespace detail {

// u64 では表せない数以上をカウントアップする多倍長整数カウンタのクラス：unsigned 型のみ対応
// 型ContainerT × CounterUnitLength のサイズを持つ
template <size_t CounterUnitLength, typename ContainerT, ContainerT ContainerT_MaxValue>
class MultiByteCounter
{
    NN_DISALLOW_COPY(MultiByteCounter);

private:
    // 切り上げサイズを確保
    ContainerT m_InnerCountValue[ (CounterUnitLength + sizeof(ContainerT) - 1) / sizeof(ContainerT) ];

public:
    // 外向けにバイト配列の値を保持
    char* const countValue;

public:
    MultiByteCounter() NN_NOEXCEPT : countValue(reinterpret_cast<char*>(m_InnerCountValue))
    {
        m_IsInverseEndian = false;
        ResetCounter();
    }
    void ResetCounter() NN_NOEXCEPT
    {
        memset(this->countValue, 0, sizeof(m_InnerCountValue));
    }
    // バイト配列を使用
    void ResetCounterWithInitialValue(const char* initialValueBuffer, const size_t bufferLength) NN_NOEXCEPT
    {
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == sizeof(m_InnerCountValue));
        memcpy(this->countValue, initialValueBuffer, bufferLength);
    }
    void UseInverseCounter() NN_NOEXCEPT
    {
        m_IsInverseEndian = true;
    }
    void Increment() NN_NOEXCEPT
    {
        CountUp(1);
    }
    void CountUp(const ContainerT value) NN_NOEXCEPT
    {
        CountUpImpl(value);
    }
    static ContainerT GetInverseEndianValue(const ContainerT value) NN_NOEXCEPT
    {
        const int containerSize = sizeof(ContainerT);
        ContainerT retVal = 0;
        for(int jj = 0; jj < containerSize; jj++)
        {
            const int shiftCount = jj * 8;
            const int invShiftCount = (sizeof(ContainerT) - 1 - jj) * 8;
            retVal += ((value >> invShiftCount) & 0xFF) << shiftCount;
        }
        return retVal;
    }
    ContainerT GetInverseEndianValueIf(const ContainerT value) NN_NOEXCEPT
    {
        if(m_IsInverseEndian)
        {
            return GetInverseEndianValue(value);
        }
        return value;
    }

private:
    bool m_IsInverseEndian;

    // カウントアップでカウンタの上限を超えた際に true となる
    void CountUpImpl(const ContainerT value) NN_NOEXCEPT
    {
        ContainerT addValue = value;
        const int counterIndexEndValue = sizeof(m_InnerCountValue) / sizeof(ContainerT) - 1;
        int counterInitialIndex = 0;
        int counterCountUpValue = 1;

        // エンディアン処理
        if ( m_IsInverseEndian )
        {
            counterInitialIndex = counterIndexEndValue;
            counterCountUpValue = -1;
        }

        for(int i = counterInitialIndex; 0 <= i && i <= counterIndexEndValue; i += counterCountUpValue)
        {
            // コンテナへの出し入れ時に（必要なら）エンディアン処理を行う
            ContainerT temp = GetInverseEndianValueIf(m_InnerCountValue[i]);

            // "もし加算で桁があふれないなら" （オーバーフロー回避）
            if ( temp <= (ContainerT_MaxValue - addValue) )
            {
                // 加算して終了
                m_InnerCountValue[i] = GetInverseEndianValueIf(temp + addValue);

                // TODO: ResultFailureCounterOverflow()
                return;
            }
            // "もし加算で桁があふれるなら"、加算して次の桁へ（オーバーフロー回避）
            volatile ContainerT lackValue = (ContainerT_MaxValue - temp) + 1;
            m_InnerCountValue[i] = GetInverseEndianValueIf(addValue - lackValue);
            addValue = 1;
        }
        return;
    }
};

// 純粋なカウントアップ用カウンタと、初期値を与えることでそこからのカウント値を保持するカウンタの２つを持つクラス
template <size_t CounterUnitLength, typename ContainerT, ContainerT ContainerT_MaxValue>
class MultiByteSetCounter
{
    NN_DISALLOW_COPY(MultiByteSetCounter);

protected:
    MultiByteCounter<CounterUnitLength, ContainerT, ContainerT_MaxValue> valueCounter;
    MultiByteCounter<CounterUnitLength, ContainerT, ContainerT_MaxValue> deltaCounter;

public:
    const size_t counterLength;

public:
    MultiByteSetCounter() NN_NOEXCEPT : counterLength(CounterUnitLength)
    {
        valueCounter.ResetCounter();
        deltaCounter.ResetCounter();
    }
    void UseInverseCounter() NN_NOEXCEPT
    {
        valueCounter.UseInverseCounter();
    }
    void CountUp(const ContainerT value) NN_NOEXCEPT
    {
        valueCounter.CountUp(value);
        deltaCounter.CountUp(value);
    }
    void Increment() NN_NOEXCEPT
    {
        CountUp(1);
    }
    void ResetCounter() NN_NOEXCEPT
    {
        valueCounter.ResetCounter();
        deltaCounter.ResetCounter();
    }
    // バイト配列を使用
    void ResetCounterWithInitialValue(const char* initialValueBuffer, const size_t bufferLength) NN_NOEXCEPT
    {
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(counterLength == bufferLength);
        ResetCounter();
        valueCounter.ResetCounterWithInitialValue(initialValueBuffer, bufferLength);
    }
    // バイト配列を使用
    void OutputCounterValue(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
    {
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(counterLength <= bufferLength);
        memcpy(outBuffer, valueCounter.countValue, bufferLength);
    }
};

static const size_t GcCounter128bitByteNum = (128 / 8);
typedef uint64_t GcCounter128bitContainerType;
#define GcCounter128bitContainerMaxSize  UINT64_MAX

// 128ビット（=16バイト）カウンタのクラス
class Counter128Bit : public MultiByteSetCounter<GcCounter128bitByteNum, GcCounter128bitContainerType, GcCounter128bitContainerMaxSize>
{
    NN_DISALLOW_COPY(Counter128Bit);

protected:
    char* const deltaCounterArray;

public:
    char* const valueCounterArray;

public:
    Counter128Bit() NN_NOEXCEPT :
        deltaCounterArray(deltaCounter.countValue),
        valueCounterArray(valueCounter.countValue){}

    // NOTE: 期待する結果を得るには GcCounter128bitContainerType が 64bit 以下である必要がある
    bool IsCounterValueOver64bit() NN_NOEXCEPT
    {
        // bit64 が格納されているインデックスより上位の値が 0 より大きければ 64bit カウンタが周り切っている
        for(size_t i = (sizeof(uint64_t) / sizeof(GcCounter128bitContainerType)); i<counterLength; i++)
        {
            if ( this->deltaCounterArray[i] > 0 )
            {
                return true;
            }
        }
        return false;
    }
};

} } }
