﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <vector>
#include <nn/fssystem/fs_IndirectStorage.h>
#include <nn/fssystem/fs_AesCtrCounterExtendedStorage.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>

namespace nn { namespace fssystem { namespace utilTool {

// パッチの再配置情報です。
class RelocationEntry
{
public:
    typedef std::pair<int64_t, int64_t> Range;

public:
    RelocationEntry() NN_NOEXCEPT
        : m_SrcOffset(0)
        , m_DstOffset(0)
        , m_Size(0)
        , m_Detail(BinaryMatchDetail_Unknown)
        , m_DetailIndex(0)
    {
    }

    RelocationEntry(
        int64_t srcOffset,
        int64_t dstOffset,
        int64_t size,
        BinaryMatchDetail detail,
        int detailIndex
    ) NN_NOEXCEPT
        : m_SrcOffset(srcOffset)
        , m_DstOffset(dstOffset)
        , m_Size(size)
        , m_Detail(detail)
        , m_DetailIndex(detailIndex)
    {
        NN_SDK_ASSERT_RANGE(detail, BinaryMatchDetail_Match, BinaryMatchDetail_Max);
        NN_SDK_ASSERT(IsValid());
    }

    // テスト向け
    RelocationEntry(
        int64_t srcOffset,
        int64_t dstOffset,
        int64_t size,
        bool isMatched
    ) NN_NOEXCEPT
        : m_SrcOffset(srcOffset)
        , m_DstOffset(dstOffset)
        , m_Size(size)
        , m_Detail(isMatched ? BinaryMatchDetail_Match : BinaryMatchDetail_RightJustify)
        , m_DetailIndex(0)
    {
        NN_SDK_ASSERT(IsValid());
    }

    void SwapOffset() NN_NOEXCEPT
    {
        std::swap(m_SrcOffset, m_DstOffset);
    }

    int64_t GetSrcOffset() const NN_NOEXCEPT
    {
        return m_SrcOffset;
    }

    int64_t GetDstOffset() const NN_NOEXCEPT
    {
        return m_DstOffset;
    }

    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    BinaryMatchDetail GetDetail() const NN_NOEXCEPT
    {
        return m_Detail;
    }

    int GetDetailIndex() const NN_NOEXCEPT
    {
        return m_DetailIndex;
    }

    bool IsMatched() const NN_NOEXCEPT
    {
        return m_Detail == BinaryMatchDetail_Match;
    }

    bool IsPadding() const NN_NOEXCEPT
    {
        return m_Detail == BinaryMatchDetail_Padding;
    }

    bool IsValid() const NN_NOEXCEPT
    {
        return 0 <= m_SrcOffset && 0 <= m_Size && (IsPadding() || 0 <= m_DstOffset);
    }

    int64_t ToDstOffset(int64_t srcOffset) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(srcOffset, m_SrcOffset, m_SrcOffset + m_Size);

        return ConvertOffset(srcOffset);
    }

    const Range ToDstRange(const Range& srcRange) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(
            srcRange.first <= srcRange.second &&
            m_SrcOffset <= srcRange.first &&
            srcRange.second <= m_SrcOffset + m_Size);

        return std::make_pair(ConvertOffset(srcRange.first), ConvertOffset(srcRange.second));
    }

private:
    int64_t ConvertOffset(int64_t srcOffset) const NN_NOEXCEPT
    {
        return srcOffset - m_SrcOffset + m_DstOffset;
    }

private:
    int64_t m_SrcOffset;
    int64_t m_DstOffset;
    int64_t m_Size;
    BinaryMatchDetail m_Detail;
    int m_DetailIndex;
};

// パッチの再配置の結果を格納/検索するクラス
class RelocationTable
{
public:
    typedef RelocationEntry Entry;
    typedef BinaryMatchProgress Progress;
    typedef Entry::Range Range;
    typedef std::pair<bool, Range> PaddingRange;
    typedef BinaryMatch Match;
    typedef Match::Result MatchResult;
    typedef Match::RegionArray RegionArray;
    typedef Match::RegionInfo RegionInfo;

    typedef std::vector<Entry>::const_iterator const_iterator;

    class BuildInfo
    {
    public:
        BuildInfo(
            int64_t oldOffset,
            int64_t newOffset,
            size_t shiftSize,
            size_t blockSize,
            size_t regionSize,
            int64_t matchSize
        ) NN_NOEXCEPT
            : m_OldStorageOffset(oldOffset)
            , m_OldStorageSize(0)
            , m_NewStorageOffset(newOffset)
            , m_NewStorageSize(0)
            , m_ShiftSize(shiftSize)
            , m_BlockSize(blockSize)
            , m_RegionSize(regionSize)
            , m_MatchSize(matchSize)
            , m_OverlapOffset(0)
            , m_OverlapSize(0)
            , m_ExpandableSize(0)
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(oldOffset, 0);
            NN_SDK_REQUIRES_GREATER_EQUAL(newOffset, 0);
            NN_SDK_REQUIRES(sizeof(int64_t) <= blockSize && util::ispow2(blockSize));
            NN_SDK_REQUIRES(0 < shiftSize && shiftSize <= blockSize && util::ispow2(shiftSize));
            NN_SDK_REQUIRES(blockSize < regionSize && util::ispow2(regionSize));
            NN_SDK_REQUIRES_GREATER_EQUAL(matchSize, 0);
        }

        Result Initialize(fs::SubStorage oldStorage, fs::SubStorage newStorage) NN_NOEXCEPT
        {
            NN_RESULT_DO(oldStorage.GetSize(&m_OldStorageSize));
            NN_RESULT_DO(newStorage.GetSize(&m_NewStorageSize));

            m_OldStorage = oldStorage;
            m_NewStorage = newStorage;

            const auto oldStorageBegin = m_OldStorageOffset;
            const auto newStorageBegin = m_NewStorageOffset;
            const auto oldStorageEnd = m_OldStorageOffset + m_OldStorageSize;
            const auto newStorageEnd = m_NewStorageOffset + m_NewStorageSize;

            const auto overlapBegin = std::max(oldStorageBegin, newStorageBegin);
            const auto overlapEnd = std::min(oldStorageEnd, newStorageEnd);
            const auto overlapSize = overlapEnd - overlapBegin;

            m_OverlapOffset = overlapBegin;
            m_OverlapSize = overlapSize;

            const int64_t ExpandableSizeMin =  10 * 1024 * 1024;
            const int64_t ExpandableSizeMax = 100 * 1024 * 1024;

            m_ExpandableSize = std::min(
                ExpandableSizeMax,
                std::max(ExpandableSizeMin, util::align_down(m_NewStorageSize / 10, m_BlockSize))
            );

            NN_RESULT_SUCCESS;
        }

        size_t GetShiftSize() const NN_NOEXCEPT
        {
            return m_ShiftSize;
        }

        size_t GetBlockSize() const NN_NOEXCEPT
        {
            return m_BlockSize;
        }

        size_t GetRegionSize() const NN_NOEXCEPT
        {
            return m_RegionSize;
        }

        int64_t GetMatchSize() const NN_NOEXCEPT
        {
            return m_MatchSize;
        }

        fs::SubStorage& GetOldStorage() const NN_NOEXCEPT
        {
            return m_OldStorage;
        }

        fs::SubStorage& GetNewStorage() const NN_NOEXCEPT
        {
            return m_NewStorage;
        }

        int64_t GetOldStorageSize() const NN_NOEXCEPT
        {
            return m_OldStorageSize;
        }

        int64_t GetNewStorageSize() const NN_NOEXCEPT
        {
            return m_NewStorageSize;
        }

        int64_t GetExpandableSize() const NN_NOEXCEPT
        {
            return m_ExpandableSize;
        }

        int64_t GetAdjustOldOffset() const NN_NOEXCEPT
        {
            return m_OverlapOffset - m_NewStorageOffset;
        }

        int64_t GetLimitOldOffset() const NN_NOEXCEPT
        {
            return m_OverlapSize + m_ExpandableSize;
        }

        int64_t GetLimitNewOffset() const NN_NOEXCEPT
        {
            return m_NewStorageOffset + m_NewStorageSize - GetAdjustOldOffset();
        }

        const std::pair<int64_t, int64_t> GetOverlapStorageRange() const NN_NOEXCEPT
        {
            const auto storageOffset = m_OverlapOffset - m_OldStorageOffset;
            const auto storageSize = std::min(
                m_OldStorageSize, storageOffset + m_OverlapSize + m_ExpandableSize) - storageOffset;

            return std::make_pair(storageOffset, storageSize);
        }

        bool IsOverlappedStorage() const NN_NOEXCEPT
        {
            return 0 < m_OverlapSize;
        }

    public:
        const int64_t m_OldStorageOffset;
        int64_t m_OldStorageSize;
        mutable fs::SubStorage m_OldStorage;
        const int64_t m_NewStorageOffset;
        int64_t m_NewStorageSize;
        mutable fs::SubStorage m_NewStorage;
        const size_t m_ShiftSize;
        const size_t m_BlockSize;
        const size_t m_RegionSize;
        const int64_t m_MatchSize;
        int64_t m_OverlapOffset;
        int64_t m_OverlapSize;
        int64_t m_ExpandableSize;
    };

public:
    RelocationTable() NN_NOEXCEPT;

    RelocationTable(const RelocationTable& other) NN_NOEXCEPT
        : m_RegionInfo()
        , m_Entries(other.m_Entries)
        , m_IsCommitted(other.m_IsCommitted)
    {
    }

    void Add(const Entry& entry) NN_NOEXCEPT;

    // リージョンハッシュを設定
    void SetRegion(const RegionArray& regions) NN_NOEXCEPT
    {
        m_RegionInfo.first = regions;
    }

    // リージョンハッシュの情報を取得
    RegionInfo GetRegionInfo() NN_NOEXCEPT
    {
        return std::move(m_RegionInfo);
    }

    Result Build(std::shared_ptr<Progress>* pOutProgress, const BuildInfo& info) NN_NOEXCEPT;

    Result Commit() NN_NOEXCEPT;

    Result Verify() const NN_NOEXCEPT;

    // 指定した範囲にテーブルを適用する
    Result ApplyTo(std::vector<Range>* pOutRanges, const Range& range) const NN_NOEXCEPT;

    // 指定した範囲にテーブルを適用する
    Result ApplyTo(std::vector<PaddingRange>* pOutRanges, const Range& range) const NN_NOEXCEPT;

    // 指定したデータにテーブルを適用する
    Result ApplyTo(
               fs::IStorage* pDstStorage,
               fs::IStorage* pSrcStorage,
               void* buffer,
               size_t bufferSize
           ) const NN_NOEXCEPT;

    // src と dst を入れ替えたテーブルを作成する
    Result Invert(
        RelocationTable* pOutTable,
        int64_t* pOutSrcSize,
        int64_t baseSrcOffset,
        int64_t baseDstOffset
    ) const NN_NOEXCEPT;

    const_iterator Find(int64_t offset) const NN_NOEXCEPT;

    bool IsValidOffset(int64_t offset) const NN_NOEXCEPT
    {
        return 0 < size() &&
               front().GetSrcOffset() <= offset &&
               offset < back().GetSrcOffset() + back().GetSize();
    }

    bool IsValidRange(const Range& range) const NN_NOEXCEPT
    {
        return 0 < size() &&
               front().GetSrcOffset() <= range.first &&
               range.second <= back().GetSrcOffset() + back().GetSize();
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsCommitted);
        return m_Entries.begin();
    }

    const_iterator end() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsCommitted);
        return m_Entries.end();
    }

    const Entry& front() const NN_NOEXCEPT
    {
        return m_Entries.front();
    }

    const Entry& back() const NN_NOEXCEPT
    {
        return m_Entries.back();
    }

    size_t size() const NN_NOEXCEPT
    {
        return m_Entries.size();
    }

    bool empty() const NN_NOEXCEPT
    {
        return size() <= 0;
    }

private:
    void BuildImpl(std::vector<MatchResult>&& results, const BuildInfo& info, BinaryMatchProgress* pProgress) NN_NOEXCEPT;

    void InsertEntry(
             const std::vector<MatchResult>& results,
             size_t partitionIndex
         ) NN_NOEXCEPT;

private:
    RegionInfo m_RegionInfo;
    std::vector<Entry> m_Entries;
    bool m_IsCommitted;
};

// 世代が同じエントリーをまとめてアライメントの調整をする
Result OptimizeAesCtrCounterExtendedEntry(
           std::vector<AesCtrCounterExtendedStorage::Entry>* pEntries,
           int64_t endOffset,
           int generation
       ) NN_NOEXCEPT;

// AesCtrEx のテーブルの検証を行う
Result VerifyAesCtrCounterExtendedTable(
           MemoryResource* pAllocator,
           int64_t currentOffset,
           const void* pCurrentHeader,
           size_t currentHeaderSize,
           fs::IStorage* pCurrentTableStorage,
           int64_t currentIndirectTableOffset,
           int64_t currentIndirectTableSize,
           int64_t previousOffset,
           const void* pPreviousHeader,
           size_t previousHeaderSize,
           fs::IStorage* pPreviousTableStorage,
           uint32_t generationForUpdatedRegion,
           const RelocationTable& invertedRelocatinTable
       ) NN_NOEXCEPT;

}}}
