﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssystem/utilTool/fs_SparseStorageBuilder.h>

namespace nn { namespace fssystem {

class NcaReader;
class NcaFsHeaderReader;

namespace utilTool {

// SparseStorage を外部ファイルとして扱うクラスです。
class NcaSparseStorage
{
public:
    typedef SparseStorageBuilder::Range Range;
    typedef SparseStorageBuilder::RangeArray RangeArray;

    // NcaFsHeader 部分から必要な部分を抽出したデータ
    struct Partition
    {
        static const int CountMax = NcaHeader::FsCountMax;

        uint32_t startSector;
        uint32_t endSector;
        NcaSparseInfo sparseInfo;
        char _reserved[40];

        bool operator==(const Partition& other) const NN_NOEXCEPT
        {
            return (this->startSector == other.startSector) && (this->endSector == other.endSector);
        }
    };

    // NcaHeader 部分から必要な部分を抽出したデータ
    struct Header
    {
        static const uint32_t Signature = NN_UTIL_CREATE_SIGNATURE_4('N','S','F','0');

        uint32_t signature;
        uint32_t sdkAddonVersion;
        uint64_t programId;
        char _reserved1[16];
        Partition partition[Partition::CountMax];
        char _reserved2[96];

        bool operator==(const Header& other) const NN_NOEXCEPT
        {
            return (this->sdkAddonVersion == other.sdkAddonVersion)
                && (this->programId == other.programId)
                && (this->partition[0] == other.partition[0])
                && (this->partition[1] == other.partition[1])
                && (this->partition[2] == other.partition[2])
                && (this->partition[3] == other.partition[3]);
        }
    };
    NN_STATIC_ASSERT(std::is_pod<Header>::value);
    NN_STATIC_ASSERT(sizeof(Header) == 512);

public:
    // コンストラクタ
    NcaSparseStorage() NN_NOEXCEPT;

    // コンストラクタ
    explicit NcaSparseStorage(const NcaReader& reader) NN_NOEXCEPT;

    // パーティションの設定
    Result SetPartition(MemoryResource* pAllocator, SparseStorageBuilder* pBuilder, const NcaFsHeaderReader& header) NN_NOEXCEPT;

    // 外部ファイルを読み込む
    Result Load(const void* buffer, size_t size) NN_NOEXCEPT;

    // NcaReader に対応するデータか確認
    bool Verify(const NcaReader& reader) const NN_NOEXCEPT
    {
        NcaSparseStorage other(reader);

        return other.m_Header == m_Header;
    }

    // 外部ファイルのデータを取得
    void Read(int64_t offset, void* buffer, size_t size) const NN_NOEXCEPT;

    // 外部ファイルのサイズを取得
    int64_t GetSize() const NN_NOEXCEPT;

    // NcaSparseInfo を取得
    const NcaSparseInfo& GetSparseInfo(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, Partition::CountMax);

        return m_Header.partition[index].sparseInfo;
    }

    // SparseStorage のテーブルデータを取得
    void* GetSparseData(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, Partition::CountMax);

        return m_Buffers[index].get();
    }

private:
    Header m_Header;
    std::unique_ptr<char[]> m_Buffers[Partition::CountMax];
};

}}}
