﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_IndirectStorage.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>
#include <nn/fssystem/utilTool/fs_BucketTreeRange.h>

namespace nn { namespace fssystem { namespace utilTool {

/**
 * @brief   IndirectStorage で扱うデータを構築するクラスです。
 */
class IndirectStorageBuilder
{
    NN_DISALLOW_COPY(IndirectStorageBuilder);

public:
    typedef MemoryResource IAllocator;

    /**
     * @brief   ストレージの比較しない領域を表します。
     */
    typedef BinaryExcludeRange ExcludeRange;

public:
    /**
     * @brief   コンストラクタです。
     */
    IndirectStorageBuilder() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    ~IndirectStorageBuilder() NN_NOEXCEPT;

    /**
     * @brief   初期化をします。
     *
     * @param[in]   pOldStorage     古いデータを指すストレージのポインタ
     * @param[in]   pNewStorage     新しいデータを指すストレージのポインタ
     * @param[in]   blockSize       ハッシュ化するデータのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pOldStorage != nullptr
     *      - pNewStorage != nullptr
     *      - pOldStorage != pNewStorage
     *      - 8 <= blockSize
     *      - ispow2(blockSize) != false
     *      - IsInitialize() == false
     */
    Result Initialize(
               fs::IStorage* pOldStorage,
               fs::IStorage* pNewStorage,
               size_t blockSize
           ) NN_NOEXCEPT;

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   初期化済みかどうかを取得します。
     *
     * @return  初期化済みかどうかを返します。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_pOldStorage != nullptr;
    }

    /**
     * @brief   古いデータを指すストレージで比較しない領域を指定します。
     *
     * @param[in]   rangeArray  指定領域の配列の先頭アドレス
     * @param[in]   rangeCount  指定領域の数
     *
     * @pre
     *      rangeArray != nullptr
     *      0 < rangeCount
     *
     * rangeArray は、Build() の実行が完了するまで保持してください。
     */
    void SetExcludeRangeForOldStorage(
             const ExcludeRange* rangeArray,
             int rangeCount
         ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(rangeArray);
        NN_SDK_REQUIRES_GREATER(rangeCount, 0);

        m_ExcludeRangeOldArray = rangeArray;
        m_ExcludeRangeOldCount = rangeCount;
    }

    /**
     * @brief   古いデータを指すストレージで比較しない領域を指定します。
     *
     * @param[in]   rangeArray  指定領域の配列
     *
     * rangeArray は、Build() の実行が完了するまで保持してください。
     */
    template< size_t N >
    void SetExcludeRangeForOldStorage(const ExcludeRange (&rangeArray)[N]) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(0 < N);

        m_ExcludeRangeOldArray = rangeArray;
        m_ExcludeRangeOldCount = N;
    }

    /**
     * @brief   新しいデータを指すストレージで比較しない領域を指定します。
     *
     * @param[in]   rangeArray  指定領域の配列の先頭アドレス
     * @param[in]   rangeCount  指定領域の数
     *
     * @pre
     *      rangeArray != nullptr
     *      0 < rangeCount
     *
     * rangeArray は、Build() の実行が完了するまで保持してください。
     */
    void SetExcludeRangeForNewStorage(
             const ExcludeRange* rangeArray,
             int rangeCount
         ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(rangeArray);
        NN_SDK_REQUIRES_GREATER(rangeCount, 0);

        m_ExcludeRangeNewArray = rangeArray;
        m_ExcludeRangeNewCount = rangeCount;
    }

    /**
     * @brief   新しいデータを指すストレージで比較しない領域を指定します。
     *
     * @param[in]   rangeArray  指定領域の配列
     *
     * rangeArray は、Build() の実行が完了するまで保持してください。
     */
    template< size_t N >
    void SetExcludeRangeForNewStorage(const ExcludeRange (&rangeArray)[N]) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(0 < N);

        m_ExcludeRangeNewArray = rangeArray;
        m_ExcludeRangeNewCount = N;
    }

    /**
     * @brief   BinaryMatch のヒント情報を設定します。
     *
     * @param[in]   hintArray   ヒント情報の配列の先頭アドレス
     * @param[in]   hintCount   ヒント情報の数
     */
    void SetBinaryMatchHint(const BinaryMatchHint* hintArray, int hintCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(hintArray);
        NN_SDK_REQUIRES_GREATER(hintCount, 0);

        m_BinaryMatchHint.assign(hintArray, hintArray + hintCount);
    }

    /**
     * @brief   リージョンハッシュの情報を設定します。
     *
     * @param[in]   region  リージョンハッシュの情報
     */
    void SetBinaryRegion(const BinaryRegionArray& region) NN_NOEXCEPT
    {
        m_BinaryRegion.first = region;
    }

    /**
     * @brief   リージョンハッシュの情報を取得します。
     *
     * @return  リージョンハッシュの情報
     */
    const BinaryRegionArray& GetBinaryRegion() const NN_NOEXCEPT
    {
        return m_BinaryRegion.first;
    }

    /**
     * @brief   ストレージを解析してテーブルを作成します。
     *
     * @param[in]   blockSize   ハッシュ化するデータのサイズ
     * @param[in]   regionSize  比較するデータの最小単位
     * @param[in]   matchSize   マッチするデータの最小単位
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - 8 <= blockSize
     *      - ispow2(blockSize) != false
     *      - blockSize < regionSize
     *      - ispow2(regionSize) != false
     *      - IsInitialized() != false
     *      - IsBuilt() == false
     */
    Result Build(size_t blockSize, size_t regionSize, int64_t matchSize) NN_NOEXCEPT
    {
        return Build(blockSize, blockSize, regionSize, matchSize, std::numeric_limits<int64_t>::max());
    }

    /**
     * @brief   探索幅を制限した上でストレージを解析してテーブルを作成します。
     *
     * @param[in]   shiftSize   ハッシュ化するデータの移動量
     * @param[in]   blockSize   ハッシュ化するデータのサイズ
     * @param[in]   regionSize  比較するデータの最小単位
     * @param[in]   matchSize   マッチするデータの最小単位
     * @param[in]   windowSize  比較するデータ領域の最大幅
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - 0 < shiftSize
     *      - shiftSize <= blockSize
     *      - ispow2(shiftSize) != false
     *      - 8 <= blockSize
     *      - ispow2(blockSize) != false
     *      - blockSize < regionSize
     *      - ispow2(regionSize) != false
     *      - regionSize < windowSize
     *      - IsInitialized() != false
     *      - IsBuilt() == false
     */
    Result Build(
               size_t shiftSize,
               size_t blockSize,
               size_t regionSize,
               int64_t matchSize,
               int64_t windowSize
           ) NN_NOEXCEPT;

    /**
     * @brief   生成済みのテーブルを注入します。// オーサリングツール向け
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   headerStorage   テーブルヘッダのストレージ
     * @param[in]   nodeStorage     テーブルノードのストレージ
     * @param[in]   entryStorage    テーブルエントリのストレージ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - headerStorage, nodeStorage, entryStorage に正しい情報が書き込まれている
     *      - IsInitialized() != false
     *      - IsBuilt() == false
     */
    Result Build(
               IAllocator* pAllocator,
               fs::SubStorage headerStorage,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage
           ) NN_NOEXCEPT;

    /**
     * @brief   テーブル作成済みかどうかを取得します。
     *
     * @return  テーブル作成済みかどうかを返します。
     */
    bool IsBuilt() const NN_NOEXCEPT
    {
        return m_Match != nullptr;
    }

    /**
     * @brief   テーブルを書き出します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   headerStorage   テーブルヘッダのストレージ
     * @param[in]   nodeStorage     テーブルノードのストレージ
     * @param[in]   entryStorage    テーブルエントリのストレージ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - IsBuilt() != false
     */
    Result WriteTable(
               IAllocator* pAllocator,
               fs::SubStorage headerStorage,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage
           ) NN_NOEXCEPT;

    /**
     * @brief   テーブルを元に必要なデータを書き出します。
     *
     * @param[in]   pDataStorage    データを書き出すストレージのポインタ
     * @param[in]   buffer          データの読み書きに使用するバッファの先頭アドレス
     * @param[in]   bufferSize      データの読み書きに使用するバッファのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pDataStorage != nullptr
     *      - buffer != nullptr
     *      - 0 < bufferSize
     *      - IsBuilt() != false
     */
    Result WriteData(fs::IStorage* pDataStorage, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /**
     * @brief   テーブルを元に必要なデータを読み込みます。
     *
     * @param[in]   offset  読み込むデータのオフセット
     * @param[in]   buffer  データを読み込むバッファのアドレス
     * @param[in]   size    読み込むデータのサイズ
     * @param[in]   pGroundStorage テーブルから参照されない領域があった場合に読み込まれるストレージ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - 0 <= offset
     *      - size == 0 || buffer != nullptr
     *      - IsBuilt() != false
     *
     * offset,size で指定された領域が [0, QueryDataStorageSize()) を超えている場合、
     * その領域は 0 埋めします。
     * テーブルに登録がない領域の読み込み要求があった場合、pGroundStorage の同領域を読み込みます。
     * このときに pGroundStorage == nullptr の場合、当該領域は 0 埋めになります。
     */
    Result ReadData(int64_t offset, void* buffer, size_t size, fs::IStorage* pGroundStorage) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_SortedMatches.ReadData(m_pNewStorage, m_DataStorageSize, offset, buffer, size, pGroundStorage));
        NN_RESULT_SUCCESS;
    }
    Result ReadData(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        return ReadData(offset, buffer, size, nullptr);
    }

    /**
     * @brief   Build() の進捗を取得します。
     */
    std::shared_ptr<BinaryMatchProgress> GetProgress() const NN_NOEXCEPT
    {
        return m_pProgress;
    }

    /**
     * @brief   テーブルでヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    int64_t QueryTableHeaderStorageSize() const NN_NOEXCEPT
    {
        return IndirectStorage::QueryHeaderStorageSize();
    }

    /**
     * @brief   テーブルでノード部分のストレージのサイズを取得します。
     *
     * @return  ノード部分のストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableNodeStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return IndirectStorage::QueryNodeStorageSize(m_MatchCount);
    }

    /**
     * @brief   テーブルでエントリ部分のストレージサイズを取得します。
     *
     * @return  エントリ部分のストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableEntryStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return IndirectStorage::QueryEntryStorageSize(m_MatchCount);
    }

    /**
     * @brief   テーブルのストレージサイズを取得します。
     *
     * @return  テーブルのストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableStorageSize() const NN_NOEXCEPT
    {
        return QueryTableHeaderStorageSize() +
               QueryTableNodeStorageSize() +
               QueryTableEntryStorageSize();
    }

    /**
     * @brief   データのストレージサイズを取得します。
     *
     * @return  データのストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryDataStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return m_DataStorageSize;
    }

    /**
     * @brief   IndirectStorage のビルド情報を出力します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   pTableStorage   テーブル情報を書き出すストレージのポインタ
     * @param[in]   pNodeStorage    ノード情報を書き出すストレージのポインタ
     * @param[in]   pEntryStorage   エントリ情報を書き出すストレージのポインタ
     * @param[in]   pDataStorage    データ情報を書き出すストレージのポインタ
     * @param[in]   startOffset     IndirectStorage の開始オフセット
     *
     * @pre
     *      - pAllocator != nullptr
     *      - pTableStorage != nullptr
     *      - pNodeStorage != nullptr
     *      - pEntryStorage != nullptr
     *      - pDataStorage != nullptr
     *      - 0 <= startOffset
     */
    Result OutputBuildLog(
               IAllocator* pAllocator,
               fs::IStorage* pTableStorage,
               fs::IStorage* pNodeStorage,
               fs::IStorage* pEntryStorage,
               fs::IStorage* pDataStorage,
               int64_t startOffset
           ) const NN_NOEXCEPT;

    /**
     * @brief   IndirectStorage のビルド情報を出力します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   pTableStorage   テーブル情報を書き出すストレージのポインタ
     * @param[in]   pNodeStorage    ノード情報を書き出すストレージのポインタ
     * @param[in]   pEntryStorage   エントリ情報を書き出すストレージのポインタ
     * @param[in]   pDataStorage    データ情報を書き出すストレージのポインタ
     * @param[in]   indirectStorage 書き出し済みの IndirectStorage のテーブル情報
     * @param[in]   entryCount      IndirectStorage のエントリ数
     * @param[in]   startOffset     IndirectStorage の開始オフセット
     * @param[in]   endOffset       IndirectStorage の終了オフセット
     *
     * @pre
     *      - pAllocator != nullptr
     *      - pTableStorage != nullptr
     *      - pNodeStorage != nullptr
     *      - pEntryStorage != nullptr
     *      - pDataStorage != nullptr
     *      - 0 < entryCount
     *      - 0 <= startOffset
     *      - startOffset < endOffset
     */
    static Result OutputBuildLog(
                      IAllocator* pAllocator,
                      fs::IStorage* pTableStorage,
                      fs::IStorage* pNodeStorage,
                      fs::IStorage* pEntryStorage,
                      fs::IStorage* pDataStorage,
                      fs::SubStorage indirectStorage,
                      int entryCount,
                      int64_t startOffset,
                      int64_t endOffset
                  ) NN_NOEXCEPT;

    /**
     * @brief   Build 時に渡した Storage と等価な別の Storage に置き換えます。
     *          Build 時にキャッシュ Storage を渡した場合に、元の Storage に戻すために使用します。
     *
     * @param[in]   pOldStorage     古いデータを指すストレージのポインタ
     * @param[in]   pNewStorage     新しいデータを指すストレージのポインタ
     *
     * @pre
     *      - pOldStorage != nullptr
     *      - pNewStorage != nullptr
     */
    void SetDataStorages(fs::IStorage* pOldStorage, fs::IStorage* pNewStorage) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOldStorage);
        NN_SDK_ASSERT_NOT_NULL(pNewStorage);

        m_pOldStorage = pOldStorage;
        m_pNewStorage = pNewStorage;
    }

private:
    typedef BinaryMatchResult MatchResult;
    typedef BinaryMatch::RegionInfo RegionInfo;

    //! リージョンのマッチ情報です。
    struct MatchEntry
    {
        int64_t virtualOffset;
        int64_t physicalOffset;
        int storageIndex;
    };
    NN_STATIC_ASSERT(std::is_pod<MatchEntry>::value);

    //! ReadData で利用するマッチ情報です。
    typedef BucketTreeRange MatchEntryWithExtentSize;

private:
    /**
     * @brief   エントリを設定します。
     *
     * @param[in]   entryIndex  エントリのインデックス
     * @param[in]   entry       設定するエントリ情報
     * @param[in]   entrySize   エントリのサイズ
     *
     * @return      追加したエントリの個数
     */
    int SetEntry(int entryIndex, const MatchResult& entry, int64_t entrySize) NN_NOEXCEPT;

    // ReadData で利用するマッチ情報を初期化します。
    Result SetSortedMatches(bool isStorageSizeUpdateRequired) NN_NOEXCEPT;

private:
    fs::IStorage* m_pOldStorage;
    fs::IStorage* m_pNewStorage;
    int64_t m_OldStorageSize;
    int64_t m_NewStorageSize;
    int64_t m_DataStorageSize;
    const ExcludeRange* m_ExcludeRangeOldArray;
    int m_ExcludeRangeOldCount;
    const ExcludeRange* m_ExcludeRangeNewArray;
    int m_ExcludeRangeNewCount;
    std::vector<BinaryMatchHint> m_BinaryMatchHint;
    RegionInfo m_BinaryRegion;
    std::unique_ptr<MatchEntry[]> m_Match;
    int m_MatchCount;
    std::shared_ptr<BinaryMatchProgress> m_pProgress;

    // ReadData 用
    BucketTreeRangeArray m_SortedMatches;

    friend class IndirectStorageBuilderTest;
};

}}}
