﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/fs/fs_IStorage.h>

namespace nn { namespace fssystem { namespace utilTool {

struct BucketTreeRange
{
    int64_t virtualOffset;
    int64_t physicalOffset;
    int64_t size;
};
NN_STATIC_ASSERT(std::is_pod<BucketTreeRange>::value);

class BucketTreeRangeArray
{
    NN_DISALLOW_COPY(BucketTreeRangeArray);

public:
    typedef BucketTreeRange value_type;
    typedef BucketTreeRange* pointer;
    typedef const BucketTreeRange* const_pointer;
    typedef BucketTreeRange& reference;
    typedef const BucketTreeRange& const_reference;
    typedef int size_type;
    typedef ptrdiff_t difference_type;
    typedef BucketTreeRange* iterator;
    typedef const BucketTreeRange* const_iterator;

public:
    BucketTreeRangeArray() NN_NOEXCEPT
        : m_Capacity(0)
        , m_Count(0)
        , m_Array()
    {
    }

    explicit BucketTreeRangeArray(int capacity) NN_NOEXCEPT
        : m_Capacity(capacity)
        , m_Count(0)
        , m_Array(new BucketTreeRange[capacity])
    {
    }

    NN_IMPLICIT BucketTreeRangeArray(BucketTreeRangeArray&& other) NN_NOEXCEPT
        : m_Capacity(other.m_Capacity)
        , m_Count(other.m_Count)
        , m_Array(std::move(other.m_Array))
    {
        other.m_Capacity = 0;
        other.m_Count = 0;
    }

    BucketTreeRangeArray& operator=(BucketTreeRangeArray&& other) NN_NOEXCEPT
    {
        if( this != &other )
        {
            m_Capacity = other.m_Capacity;
            m_Count = other.m_Count;
            m_Array = std::move(other.m_Array);

            other.m_Capacity = 0;
            other.m_Count = 0;
        }
        return *this;
    }

    reference operator[](int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, m_Count);

        return m_Array[index];
    }

    const_reference operator[](int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, m_Count);

        return m_Array[index];
    }

    void clear() NN_NOEXCEPT
    {
        m_Capacity = 0;
        m_Count = 0;
        m_Array.reset();
    }

    void resize(size_type size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_MINMAX(size, 0, m_Capacity);

        m_Count = size;
    }

    void push_back(const_reference range) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(m_Count, m_Capacity);

        m_Array[m_Count] = range;
        ++m_Count;
    }

    void emplace_back(int64_t virtualOffset, int64_t physicalOffset, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(m_Count, m_Capacity);

        m_Array[m_Count].virtualOffset = virtualOffset;
        m_Array[m_Count].physicalOffset = physicalOffset;
        m_Array[m_Count].size = size;
        ++m_Count;
    }

    reference front() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_Count, 0);
        return m_Array[0];
    }

    const_reference front() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_Count, 0);
        return m_Array[0];
    }

    reference back() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_Count, 0);
        return m_Array[m_Count - 1];
    }

    const_reference back() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_Count, 0);
        return m_Array[m_Count - 1];
    }

    iterator begin() NN_NOEXCEPT
    {
        return m_Array.get();
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        return m_Array.get();
    }

    const_iterator cbegin() const NN_NOEXCEPT
    {
        return begin();
    }

    iterator end() NN_NOEXCEPT
    {
        return m_Array.get() + m_Count;
    }

    const_iterator end() const NN_NOEXCEPT
    {
        return m_Array.get() + m_Count;
    }

    const_iterator cend() const NN_NOEXCEPT
    {
        return end();
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_Count == 0;
    }

    size_type size() const NN_NOEXCEPT
    {
        return m_Count;
    }

    pointer data() const NN_NOEXCEPT
    {
        return m_Array.get();
    }

    bool IsValid() const NN_NOEXCEPT
    {
        return m_Array != nullptr;
    }

    Result ReadData(
        fs::IStorage* pDataStorage,
        int64_t dataSize,
        int64_t readOffset,
        void* readBuffer,
        size_t readSize,
        fs::IStorage* pGroundStorage
    ) const NN_NOEXCEPT;

private:
    int m_Capacity;
    int m_Count;
    std::unique_ptr<BucketTreeRange[]> m_Array;
};

}}}
