﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>

namespace nn { namespace fssystem { namespace utilTool {

// 同じリージョンハッシュを持つ RegionSet の範囲を表す
class BinaryMatch::RegionRange
{
public:
    RegionRange() NN_NOEXCEPT
        : m_pArray(nullptr)
        , m_Count(0)
    {
    }

    RegionRange(Region** pArray, int count) NN_NOEXCEPT
        : m_pArray(pArray)
        , m_Count(count)
    {
        NN_SDK_REQUIRES_NOT_NULL(pArray);
        NN_SDK_REQUIRES_GREATER(count, 0);
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_Count == 0;
    }

    int size() const NN_NOEXCEPT
    {
        return m_Count;
    }

    Region** begin() const NN_NOEXCEPT
    {
        return m_pArray;
    }

    Region** end() const NN_NOEXCEPT
    {
        return m_pArray + m_Count;
    }

private:
    Region** m_pArray;
    int m_Count;
};

// リージョンハッシュを検索するためのコンテナ
class BinaryMatch::RegionSet
{
public:
    static bool IsValid(const std::unique_ptr<RegionSet>& ptr) NN_NOEXCEPT
    {
        return (ptr != nullptr) && ptr->IsValid();
    }

public:
    RegionSet(int regionCount, int hashCount) NN_NOEXCEPT
        : m_pRegions(new Region*[regionCount])
        , m_RegionCount(regionCount)
        , m_HashCount(hashCount)
        , m_DuplicateCountMax(0)
        , m_IndexTable()
    {
        NN_SDK_ASSERT_GREATER(regionCount, 0);
        NN_SDK_ASSERT_GREATER(hashCount, 0);
    }

    void Initialize(Region* pRegion) NN_NOEXCEPT;

    const RegionRange Find(const RegionHash& regionHash) const NN_NOEXCEPT;

    bool IsValid() const NN_NOEXCEPT
    {
        return m_pRegions != nullptr;
    }

    int GetDuplicateCountMax() const NN_NOEXCEPT
    {
        return m_DuplicateCountMax;
    }

private:
    typedef uint16_t BinType;
    NN_STATIC_ASSERT((std::is_same<decltype(std::numeric_limits<BinType>::max() + 2), int>::value));
    NN_STATIC_ASSERT(std::numeric_limits<BinType>::max() == 0xFFFF);

private:
    std::unique_ptr<Region*[]> m_pRegions;
    const int m_RegionCount;
    const int m_HashCount;
    int m_DuplicateCountMax;
    int m_IndexTable[0xFFFF + 2];
};

// リージョンハッシュを 1 ブロックずつ移動させるためのコンテナ
class BinaryMatch::HashQueue
{
public:
    typedef RegionHash::BlockHash BlockHash;

public:
    static size_t QuerySize(int blockHashCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(blockHashCount, 0);

        return sizeof(HashQueue) +
               (sizeof(BlockHash) * blockHashCount) +
               util::align_up(sizeof(bool) * blockHashCount, sizeof(void*));
    }

    static HashQueue* Create(char* buffer, size_t bufferSize, int blockHashCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);
        NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, QuerySize(blockHashCount));
        NN_SDK_REQUIRES_GREATER(blockHashCount, 0);
        NN_UNUSED(bufferSize);

        return new(buffer) HashQueue(buffer + sizeof(HashQueue), blockHashCount);
    }

    static void Destroy(HashQueue* pQueue) NN_NOEXCEPT
    {
        pQueue->~HashQueue();
        NN_UNUSED(pQueue);
    }

public:
    void Reset(const char* buffer, size_t hashSize) NN_NOEXCEPT;

    void Push(const char* buffer, size_t hashSize) NN_NOEXCEPT;

    const RegionHash& GetRegionHash() const NN_NOEXCEPT
    {
        return m_RegionHash;
    }

    bool IsEqualToPrevious() const NN_NOEXCEPT
    {
        return m_EqualCount == m_BlockHashCount;
    }

private:
    HashQueue(char* buffer, int blockHashCount) NN_NOEXCEPT
        : m_pBlockHash(reinterpret_cast<BlockHash*>(buffer))
        , m_pEqual(reinterpret_cast<bool*>(buffer + sizeof(BlockHash) * blockHashCount))
        , m_BlockHashCount(blockHashCount)
        , m_EqualCount(0)
        , m_Index(0)
        , m_Padding(0)
    {
        // NOTE: Create() からしか呼ばれないので事前検証とバッファサイズは省略
    }

private:
    RegionHash m_RegionHash;
    BlockHash* const m_pBlockHash;
    bool* const m_pEqual;
    const int m_BlockHashCount;
    int m_EqualCount;
    int m_Index;
    const int m_Padding;
};

// HashQueue の配列を管理するクラス
class BinaryMatch::HashQueueArray
{
public:
    typedef BinaryMatch::HashQueue HashQueue;

public:
    HashQueueArray(int hashQueueCount, int blockHashCount) NN_NOEXCEPT;

    ~HashQueueArray() NN_NOEXCEPT;

    HashQueue& operator[](int index) NN_NOEXCEPT
    {
        return *Get(index);
    }

    const HashQueue& operator[](int index) const NN_NOEXCEPT
    {
        return *Get(index);
    }

    bool IsValid() const NN_NOEXCEPT
    {
        return m_Memory != nullptr;
    }

private:
    HashQueue* Get(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, m_QueueCount);
        NN_SDK_REQUIRES_NOT_NULL(m_pBegin);

        return reinterpret_cast<HashQueue*>(m_pBegin + m_QueueSize * index);
    }

private:
    std::unique_ptr<char> m_Memory;
    char* m_pBegin;
    const size_t m_QueueSize;
    const int m_QueueCount;
};

// マッチ候補を表すクラス
class BinaryMatch::Candidate
{
public:
    Candidate() NN_NOEXCEPT
        : m_pBase(nullptr)
        , m_pMatch(nullptr)
        , m_MatchIndex(0)
        , m_BaseOffset(0)
        , m_OldOffset(0)
        , m_NewOffset(0)
        , m_Size(0)
    {
    }

    Candidate(const Region& region, int64_t oldOffset, int64_t newOffset, int64_t matchSize) NN_NOEXCEPT
        : m_pBase(&region)
        , m_pMatch(&region)
        , m_MatchIndex(0)
        , m_BaseOffset(oldOffset)
        , m_OldOffset(oldOffset)
        , m_NewOffset(newOffset)
        , m_Size(matchSize)
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(oldOffset, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(newOffset, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(matchSize, 0);
        NN_SDK_REQUIRES_NOT_NULL(m_pBase);
    }

    // サイズの昇順、オフセットの降順になるようにする（std::max_element() で使用）
    bool operator<(const Candidate& other) const NN_NOEXCEPT
    {
        if( m_Size == other.m_Size )
        {
            return m_OldOffset > other.m_OldOffset;
        }
        return m_Size < other.m_Size;
    }

    // 候補から除外することをマーク
    void Exclude() NN_NOEXCEPT
    {
        m_pMatch = nullptr;
    }

    // 比較を始めたリージョンに戻す
    void ResetRegion(int64_t regionSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(regionSize, 0);
        NN_SDK_REQUIRES_EQUAL(m_pBase->size % regionSize, 0); // m_pBase->size は regionSize の整数倍
        NN_SDK_REQUIRES_RANGE(m_OldOffset, m_pBase->offset, m_pBase->offset + m_pBase->size);

        m_pMatch = m_pBase;
        // まとめたリージョンハッシュの末尾側に一致していたことにした場合は 0 < m_MatchIndex になる
        m_MatchIndex = static_cast<int>((m_OldOffset - m_pBase->offset) / regionSize);
    }

    // regionSize 戻した位置のリージョンハッシュに移動
    const Region& MovePreviousRegion(int64_t regionSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(regionSize, 0);
        NN_SDK_REQUIRES_EQUAL(m_pBase->size % regionSize, 0);
        NN_SDK_REQUIRES_NOT_NULL(m_pMatch);

        if( m_MatchIndex <= 0 )
        {
            --m_pMatch;
            m_MatchIndex = static_cast<int>(m_pMatch->size / regionSize);
        }
        --m_MatchIndex;
        return *m_pMatch;
    }

    // regionSize 進めた位置のリージョンハッシュに移動
    const Region& MoveNextRegion(int64_t regionSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(regionSize, 0);
        NN_SDK_REQUIRES_EQUAL(m_pBase->size % regionSize, 0);
        NN_SDK_REQUIRES_NOT_NULL(m_pMatch);

        ++m_MatchIndex;
        if( m_pMatch->size <= m_MatchIndex * regionSize )
        {
            ++m_pMatch;
            m_MatchIndex = 0;
        }
        return *m_pMatch;
    }

    // まとめたリージョンハッシュの末尾側に一致していたことにして、その次のリージョンハッシュに強制移動
    const Region& SlideNextRegion() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_pBase, m_pMatch);
        NN_SDK_REQUIRES_GREATER(m_pBase->size, m_Size);
        NN_SDK_REQUIRES_EQUAL(m_pBase->offset, m_OldOffset);

        ++m_pMatch;
        m_MatchIndex = 0;
        m_BaseOffset = m_pMatch->offset - m_Size; // m_Size は変えないで m_OldOffset を調整
        m_OldOffset = m_BaseOffset;

        return *m_pMatch;
    }

    // 前のデータの部分一致を調査
    size_t ComparePreviousData(const char* oldBuffer, const char* newBuffer, size_t compareSize, size_t alignment) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(oldBuffer);
        NN_SDK_REQUIRES_NOT_NULL(newBuffer);
        NN_SDK_REQUIRES_GREATER(compareSize, static_cast<size_t>(0));
        NN_SDK_REQUIRES_GREATER(alignment, static_cast<size_t>(0));

        std::reverse_iterator<const char*> oldBegin(oldBuffer + compareSize);
        std::reverse_iterator<const char*> oldEnd(oldBuffer);
        std::reverse_iterator<const char*> newBegin(newBuffer + compareSize);

        const auto positions = std::mismatch(oldBegin, oldEnd, newBegin);
        const auto matchSize = nn::util::align_down(positions.first - oldBegin, alignment);
        NN_SDK_ASSERT_MINMAX(matchSize, 0, static_cast<ptrdiff_t>(compareSize));

        // 一部が一致しているならそのことを記録する
        ExpandNegative(matchSize);

        return static_cast<size_t>(matchSize);
    }

    // 後のデータの部分一致を調査
    size_t CompareNextData(const char* oldBuffer, const char* newBuffer, size_t compareSize, size_t alignment) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(oldBuffer);
        NN_SDK_REQUIRES_NOT_NULL(newBuffer);
        NN_SDK_REQUIRES_GREATER(compareSize, static_cast<size_t>(0));
        NN_SDK_REQUIRES_GREATER(alignment, static_cast<size_t>(0));

        const auto positions = std::mismatch(oldBuffer, oldBuffer + compareSize, newBuffer);
        const auto matchSize = nn::util::align_down(positions.first - oldBuffer, alignment);
        NN_SDK_ASSERT_MINMAX(matchSize, 0, static_cast<ptrdiff_t>(compareSize));

        // 一部が一致しているならそのことを記録する
        ExpandPositive(matchSize);

        return static_cast<size_t>(matchSize);
    }

    // 一致した領域をマイナス方向に拡張する
    void ExpandNegative(int64_t matchSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(matchSize, 0);

        m_OldOffset -= matchSize;
        m_NewOffset -= matchSize;
        m_Size += matchSize;
    }

    // 一致した領域をプラス方向に拡張する
    void ExpandPositive(int64_t matchSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(matchSize, 0);

        m_Size += matchSize;
    }

    // 候補から除外されたかどうか
    bool IsExcluded() const NN_NOEXCEPT
    {
        return m_pMatch == nullptr;
    }

    // まとめたリージョンハッシュの末尾側と比較するする必要があるかどうか
    bool IsSlideCheckNeeded() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pMatch);

        return (m_pBase == m_pMatch) && (m_pMatch->offset < m_OldOffset + m_Size);
    }

    // 比較を始めたリージョンを取得します。
    const Region& GetBaseRegion() const NN_NOEXCEPT
    {
        return *m_pBase;
    }

    // 一致した領域のサイズを取得
    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    // 一致した領域の開始オフセット（旧データ側）を取得
    int64_t GetOffset() const NN_NOEXCEPT
    {
        return m_OldOffset;
    }

    // 一致した領域の終了オフセット（旧データ側）を取得
    int64_t GetEndOffset() const NN_NOEXCEPT
    {
        return m_OldOffset + m_Size;
    }

    // 比較を始めたオフセットを取得します。
    int64_t GetBaseOffset() const NN_NOEXCEPT
    {
        return m_BaseOffset;
    }

    // 一致した領域の結果を取得
    const Result GetResult() const NN_NOEXCEPT
    {
        return Result::MakeMatch(m_NewOffset, m_OldOffset, m_Size);
    }

private:
    const Region* m_pBase;
    const Region* m_pMatch;
    int m_MatchIndex;
    int64_t m_BaseOffset;
    int64_t m_OldOffset;
    int64_t m_NewOffset;
    int64_t m_Size;
};

// マッチ候補のリストを扱うラッパークラス
class BinaryMatch::CandidateHolder
{
public:
    typedef std::vector<Candidate> Container;
    typedef Container::iterator iterator;
    typedef Container::const_iterator const_iterator;

public:
    explicit CandidateHolder(std::vector<Candidate>* pItems) NN_NOEXCEPT
        : m_pItems(pItems)
        , m_ItemCount(pItems->size())
    {
        NN_SDK_REQUIRES_NOT_NULL(pItems);
    }

    void Update(const Candidate& item) NN_NOEXCEPT
    {
        if( item.IsExcluded() )
        {
            NN_SDK_ASSERT_GREATER(m_ItemCount, static_cast<size_t>(0));
            --m_ItemCount;
        }
    }

    iterator begin() NN_NOEXCEPT
    {
        return m_pItems->begin();
    }

    iterator end() NN_NOEXCEPT
    {
        return m_pItems->end();
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        return m_pItems->begin();
    }

    const_iterator end() const NN_NOEXCEPT
    {
        return m_pItems->end();
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_ItemCount == 0;
    }

private:
    std::vector<Candidate>* m_pItems;
    size_t m_ItemCount;
};

// マッチしていない領域を比較対象として扱うクラス
class BinaryMatch::CompareRange
{
public:
    // コンストラクタ
    CompareRange(const std::vector<Result>& results, int64_t compareSize) NN_NOEXCEPT
        : m_Results(results)
        , m_ResultCount(results.size())
        , m_ResultIndex(0)
        , m_CompareSize(compareSize)
        , m_CurOffset(0)
        , m_EndOffset(0)
    {
        MoveNext(0);
    }

    // 比較開始位置をずらす
    void Advance(size_t size) NN_NOEXCEPT
    {
        m_CurOffset += size;
    }

    // 指定したオフセット以降でマッチしていない領域を指すように移動する
    void MoveNext(int64_t offset) NN_NOEXCEPT;

    // 比較可能かどうかを取得
    bool IsValid() const NN_NOEXCEPT
    {
        return m_ResultIndex < m_ResultCount;
    }

    // 比較できるサイズを取得
    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_EndOffset - m_CurOffset;
    }

    // 比較範囲の開始オフセットを取得
    int64_t GetOffset() const NN_NOEXCEPT
    {
        return m_CurOffset;
    }

    // 比較範囲の終了オフセットを取得
    int64_t GetEndOffset() const NN_NOEXCEPT
    {
        return m_EndOffset;
    }

private:
    const std::vector<Result>& m_Results;
    const size_t m_ResultCount;
    size_t m_ResultIndex;
    const int64_t m_CompareSize;
    int64_t m_CurOffset;
    int64_t m_EndOffset;
};

}}}
