﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fssystem/save/fs_ISaveFile.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief IFile、Storage間を呼びかえるクラスです。
*/
class StorageFile : public nn::fssystem::save::IFile
{
    NN_DISALLOW_COPY(StorageFile);

public:
    /**
    * @brief        コンストラクタ
    */
    StorageFile() NN_NOEXCEPT
    : m_pStorage(nullptr)
    {
    }

    /**
    * @brief        ストレージをファイルとしてマウントします。
    *
    * @param[in]    pStorage    下位ファイル
    */
    void Initialize(fs::IStorage *pStorage) NN_NOEXCEPT
    {
        m_pStorage = pStorage;
    }

    /**
    * @brief        下位ストレージへの参照を解放します。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_pStorage = nullptr;
    }

    /**
    * @brief        ストレージの総バイトサイズを取得します。
    *
    * @param[out]   outValue    総バイトサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->GetSize(outValue);
    }

    /**
    * @brief        ストレージの総バイトサイズを設定します。
    *
    * @param[out]   outValue    総バイトサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->SetSize(size);
    }

    /**
    * @brief        ストレージの offset から size バイト読み込み buffer にコピーします。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result ReadBytes(
                       int64_t offset,
                       void *buffer,
                       size_t size
                   )  NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->Read(offset, buffer, size);
    }

    /**
    * @brief        ストレージの offset 以降に buffer を size バイト分コピーします。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result WriteBytes(
                       int64_t offset,
                       const void *buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->Write(offset, buffer, size);
    }

    /**
    * @brief        ストレージに対してフラッシュ要求を出します。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->Flush();
    }

    /**
    * @brief        範囲指定処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲指定処理の種類
    * @param[in]    offset          範囲指定処理開始位置
    * @param[in]    size            範囲指定処理を行うデータサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

private:
    fs::IStorage* m_pStorage;            //!< 下位ストレージ
};

}}}

