﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fssystem/fs_PathTool.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>
#include <nn/fssystem/save/fs_SaveFileStorage.h>
#include <nn/fssystem/save/fs_IExclusiveFileSystem.h>

class FsSaveDataFileSystemCoreTest;

namespace nn { namespace fssystem { namespace save {

class SaveDataFile;
class SaveDataDirectory;
class IInternalStorageFileSystemVisitor;

/**
* @brief セーブデータアーカイブクラスです。
*
*        セーブデータ領域に対して、ファイルやディレクトリ操作を行うことができます。
*/
class SaveDataFileSystemCore : public IExclusiveFileSystem
{
    NN_DISALLOW_COPY(SaveDataFileSystemCore);
public:
    static const int  InternalStorageFileCount = 3; // WithZeroFree を含まない
    static const char InternalStorageFileNameAllocationTableControlArea[];
    static const char InternalStorageFileNameAllocationTableMeta[];
    static const char InternalStorageFileNameAllocationTableData[];
    static const char InternalStorageFileNameAllocationTableDataWithZeroFree[];

public:
    //!< ファイルシステム管理ヘッダー
    struct FileSystemHeader
    {
        uint32_t                            magic;                  //! 初期化チェック用コード
        uint32_t                            version;                //! アーカイバーのバージョン
        int64_t                             countTotalBlocks;       //! 全ブロック数
        int64_t                             sizeBlocks;             //! 1ブロックのサイズ
        FileSystemControlArea::ControlArea  fileSystemControlArea;  //! ファイルシステム管理領域
    };
    NN_STATIC_ASSERT(std::is_pod<FileSystemHeader>::value);

public:
    /**
    * @brief        セーブデータ ファイルシステムをフォーマットするのに必要なデータサイズを計算します。
    *
    * @param[in]    sizeBlock       ブロックサイズ
    * @param[in]    countDataBlock  実データ用ブロック数
    *
    * @return       ファイルシステムの実データ領域に必要なサイズ
    *
    * @details      本関数を呼び出しているソースファイルは、
    *               nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h をインクルードしなければなりません。
    */
    static inline int64_t QueryDataSize(
                          size_t sizeBlock,
                          size_t countDataBlock
                      ) NN_NOEXCEPT
    {
        return static_cast<int64_t>(sizeBlock) * countDataBlock;
    }

    /**
    * @brief        可変長メタデータ領域をフォーマット/拡張するのに必要なデータサイズを計算します。
    *
    * @param[in]    countDirectoryEntry ディレクトリエントリー数
    * @param[in]    countFileEntry      ファイルエントリー数
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    countDataBlock      実データ用ブロック数
    *
    * @return       ファイルシステムの可変長なメタデータのサイズ
    *
    * @details      本関数を呼び出しているソースファイルは、
    *               nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h をインクルードしなければなりません。
    */
    static inline int64_t QueryMetaSize(
                              size_t sizeBlock,
                              size_t countDataBlock
                          ) NN_NOEXCEPT
    {
        int64_t sizeFixed
            = QueryAllocationTableStorageSize(
                  static_cast<uint32_t>(countDataBlock)
              );
        return (sizeFixed + sizeBlock - 1) & (~(sizeBlock - 1));
    }

    /**
    * @brief        セーブデータ領域をフォーマットします。
    *
    * @param[in]    storageControlArea  管理領域として使用するストレージ
    * @param[in]    storageMeta         メタデータ領域として使用するストレージ
    * @param[in]    storageData         実データ領域として使用するストレージ
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    pBufferManager      バッファマネージャ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      fs::SubStorage storageControlArea,
                      fs::SubStorage storageMeta,
                      fs::SubStorage storageData,
                      uint32_t sizeBlock,
                      fssystem::IBufferManager* pBufferManager
                  ) NN_NOEXCEPT;

    /**
    * @brief        管理領域を拡張します。
    *
    * @param[in]    storageControlArea  管理領域として使用するストレージ
    * @param[in]    sizeDataNew         新しい実データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result ExpandControlArea(
                      fs::SubStorage storageControlArea,
                      int64_t sizeDataNew
                  ) NN_NOEXCEPT;

    /**
    * @brief        可変長メタデータ領域を拡張します。
    *
    * @param[in]    storageMeta     メタデータ領域として使用するストレージ
    * @param[in]    sizeBlock       ブロックサイズ
    * @param[in]    sizeDataOld     元々の実データ領域のサイズ
    * @param[in]    sizeDataNew     新しい実データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result ExpandMeta(
                      fs::SubStorage storageMeta,
                      uint32_t sizeBlock,
                      int64_t sizeDataOld,
                      int64_t sizeDataNew
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタ
    */
    SaveDataFileSystemCore() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~SaveDataFileSystemCore() NN_NOEXCEPT;

    /**
    * @brief        セーブデータ ファイルシステムをマウントします。
    *
    * @param[in]    storageControlArea  管理領域として使用するストレージ
    * @param[in]    storageMeta         メタデータ領域として使用するストレージ
    * @param[in]    storageData         実データ領域として使用するストレージ
    * @param[in]    pBufferManager      バッファマネージャ
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
               fs::SubStorage storageControlArea,
               fs::SubStorage storageMeta,
               fs::SubStorage storageData,
               fssystem::IBufferManager* pBufferManager
           ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータアーカイブをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ファイルを作成します。
    *
    * @param[in]    path    ファイルパス
    * @param[in]    size    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateFileImpl(
                       const Path& path,
                       int64_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    path ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateDirectoryImpl(
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;
    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outValue    ファイルオブジェクト
    * @param[in]    path        ファイルパス
    * @param[in]    mode        オープンモード
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ファイルオブジェクト用にメモリが確保されます。
    *               確保されたメモリは @ref CloseFile を用いて解放する必要があります。
    */
    virtual Result OpenFileImpl(
                       ExclusiveFile** outValue,
                       const Path& path,
                       int mode
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outValue    ディレクトリオブジェクト
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ディレクトリオブジェクト用にメモリが確保されます。
    *               確保されたメモリは @ref CloseDirectory を用いて解放する必要があります。
    */
    virtual Result OpenDirectoryImpl(
                       ExclusiveDirectory** outValue,
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルオブジェクト用リソースを解放します。
    *
    * @param[in]    pFile   ファイルオブジェクト
    */
    virtual void CloseFileImpl(
                     ExclusiveFile* pFile
                 ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリオブジェクト用リソースを解放します。
    *
    * @param[in]    pDirectory  ディレクトリオブジェクト
    */
    virtual void CloseDirectoryImpl(
                     ExclusiveDirectory* pDirectory
                 ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを削除します。
    *
    * @param[in]    path    ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteFileImpl(
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectoryImpl(
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルをリネームします。
    *
    * @param[in]    oldPath 変更対象のファイルパス
    * @param[in]    newPath 変更後のファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameFileImpl(
                       const Path& oldPath,
                       const Path& newPath
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリをリネームします。
    *
    * @param[in]    oldPath 変更対象のディレクトリパス
    * @param[in]    newPath 変更後のディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameDirectoryImpl(
                       const Path& oldPath,
                       const Path& newPath
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したファイルが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ファイルが存在しているかどうか
    * @param[in]    path        ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasFileImpl(
                       bool* outValue,
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したディレクトリが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ディレクトリが存在しているかどうか
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasDirectoryImpl(
                       bool* outValue,
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        空き領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    空き領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetFreeBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(m_CountBody != 0);

        NN_SDK_ASSERT_NOT_NULL(outValue);
        uint32_t count;
        NN_RESULT_DO(CalcFreeListLength(&count));
        *outValue = static_cast<int64_t>(count) * m_SizeBlock;
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        実データ領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    実データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetDataAreaBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES(m_CountBody != 0);

        NN_SDK_ASSERT_NOT_NULL(outValue);
        *outValue = static_cast<int64_t>(m_CountBody) * m_SizeBlock;
        NN_RESULT_SUCCESS;
    }

    //! セーブデータ内部ストレージアクセス用の Vistor を受け入れます。
    nn::Result AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT;

private:
    /**
    * @brief        アロケーションテーブルに必要なサイズを取得します。
    *
    * @param[in]    blockCount  ブロック数
    *
    * @return       アロケーションテーブルに必要なサイズ
    */
    static int64_t QueryAllocationTableStorageSize(uint32_t blockCount) NN_NOEXCEPT
    {
        return AllocationTable::QuerySize(blockCount);
    }

    /**
    * @brief セーブデータ領域をフォーマットします。
    *
    * @param[in]    pFileSystemHeader   管理領域
    * @param[in]    storageControlArea  管理領域として使用するストレージ
    * @param[in]    storageMeta         メタデータ領域として使用するストレージ
    * @param[in]    storageData         実データ領域として使用するストレージ
    * @param[in]    sizeBlock           実データ領域のブロックサイズ
    * @param[in]    countBlocks         実データ領域に割り当てるブロック数
    * @param[in]    pBufferManager      バッファマネージャ
    *
    * @return       関数の処理結果を返します。
    */
    static Result FormatDbmLayer(
                      fs::SubStorage storageControlArea,
                      fs::SubStorage storageMeta,
                      fs::SubStorage storageData,
                      uint32_t sizeBlock,
                      uint32_t countBlocks,
                      fssystem::IBufferManager* pBufferManager
                  ) NN_NOEXCEPT;

    /**
    * @brief        アロケーションテーブルの空きブロック数を取得します。
    *
    * @param[out]   outCount    アロケーションテーブルの空きブロック数
    *
    * @return       関数の処理結果を返します。
    */
    Result CalcFreeListLength(uint32_t* outCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outCount);
        return m_AllocationTable.CalcFreeListLength(outCount);
    }

    /**
    * @brief パスの正規化を行います。
    *
    * @param[out]   outBuf  正規化後のパス
    * @param[in]    path    パスオブジェクト
    *
    * @return 関数の処理結果を返します。
    */
    static Result GetStringPath(
                      Path::Char* outBuf,
                      const Path& path
                  ) NN_NOEXCEPT;

    /**
    * @brief        ファイルパスからファイル ID を取得します。
    *
    * @param[out]   outId   ファイル ID
    * @param[in]    path    フルパス
    * @param[in]    isFile  ファイルであれば true、ディレクトリであれば false
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetFileIdFromPath(
                       int64_t *outId,
                       const Path& path,
                       bool isFile
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ID からエントリとディレクトリが親子関係にあるか調べる
    *
    * @param[out]   outIsSubEntry       checkEntryId が baseDirectoryId の子か否か
    * @param[in]    baseDirectoryId     ディレクトリの ID
    * @param[in]    checkEntryId        エントリの ID
    * @param[in]    isFile              ファイルであれば true、ディレクトリであれば false
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ID からエントリとディレクトリが親子関係にあるか調べる
    */
    virtual Result CheckSubEntry(
            bool* outIsSubEntry,
            StorageIndex baseDirectoryId,
            StorageIndex checkEntryId,
            bool isFile
        ) const NN_NOEXCEPT NN_OVERRIDE;

private:
    fs::SubStorage m_StorageControlArea;                    //! 管理領域ストレージ
    fs::SubStorage m_StorageMeta;                           //! メタデータを管理するストレージ
    FileSystemControlArea m_ControlArea;                    //! 管理領域
    FileSystemObject m_FileSystem;                          //! ファイルシステム
    AllocationTable m_AllocationTable;                      //! ファイルアロケーションテーブル
    BufferedAllocationTableStorage m_StorageDirectoryEntry; //! ディレクトリエントリ用ストレージ
    BufferedAllocationTableStorage m_StorageFileEntry;      //! ファイルエントリ用ストレージ
    fs::SubStorage m_StorageData;                           //! 実データを管理するストレージ
    int64_t m_OffsetBody;                                   //! 実データ領域へのオフセット
    uint32_t m_CountBody;                                   //! 実データ領域でのブロック数
    int64_t m_SizeBlock;                                    //! ブロックサイズ

private:
    // セーブデータディレクトリ/ファイルオブジェクトには内部実装を公開します。
    friend class SaveDataDirectory;
    friend class SaveDataFile;

    // テスト用クラスには実装を公開します。
    friend class ::FsSaveDataFileSystemCoreTest;
};

/**
* @brief セーブデータ用ファイルオブジェクトです。
*/
class SaveDataFile : public ExclusiveFile, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(SaveDataFile);

private:
    using ExclusiveFile::SetMode;
    using ExclusiveFile::GetMode;

public:
    /**
    * @brief        デストラクタ
    */
    virtual ~SaveDataFile() NN_NOEXCEPT;

    /**
    * @brief        ファイルの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result ReadBytes(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        バッファの内容をファイルに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result WriteBytes(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       失敗の結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        範囲指定処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲指定処理の種類
    * @param[in]    offset          範囲指定処理開始位置
    * @param[in]    size            範囲指定処理を行うデータサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

private:
    /**
    * @brief        コンストラクタ
    */
    explicit SaveDataFile(os::Mutex& locker) NN_NOEXCEPT;

    /**
    * @brief        ファイルデータアクセス用ユーティリティを初期化します。
    *
    * @return       関数の処理結果を返します。
    */
    Result InitializeTable() NN_NOEXCEPT;

    /**
    * @brief        ファイルオブジェクトを初期化します。
    *
    * @param[in]    mode                ファイルオープンモード
    * @param[in]    pAllocationTable    実データ領域でのデータ管理用アロケーションテーブル
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    pStorage            実データ用ストレージ
    * @param[in]    offsetStorage       実データ用ストレージの開始オフセット
    * @param[in]    sizeStorage         実データ用ストレージのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
                int mode,
                AllocationTable* pAllocationTable,
                uint32_t sizeBlock,
                fs::SubStorage storage,
                int64_t offsetStorage,
                int64_t sizeStorage
           ) NN_NOEXCEPT;

    /**
    * @brief        内部ファイルオブジェクトを取得します。
    */
    FileObject* GetFileObject() NN_NOEXCEPT
    {
        return &m_FileObject;
    }

private:
    os::Mutex& m_LockObject;                 //! ロックオブジェクト
    FileObject m_FileObject;                 //! ファイルオブジェクト
    AllocationTableStorage m_Table;          //! 実データ用ストレージへのリードライト用
    AllocationTable* m_pAllocationTable;     //! 実データ領域でのデータ管理用アロケーションテーブル
    fs::SubStorage m_Storage;                //! 実データ用ストレージ
    int64_t m_SizeBlock;                     //! ブロックサイズ
    int64_t m_OffsetStorage;                 //! 実データ用ストレージの開始オフセット
    int64_t m_SizeStorage;                   //! 実データ用ストレージのサイズ

private:
    // セーブデータアーカイブに内部ファイルオブジェクトを公開します。
    friend class SaveDataFileSystemCore;
};

/**
* @brief セーブデータ用ディレクトリオブジェクトです。
*/
class SaveDataDirectory : public ExclusiveDirectory, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(SaveDataDirectory);

public:
    /**
    * @brief        デストラクタ
    */
    virtual ~SaveDataDirectory() NN_NOEXCEPT;

    /**
    * @brief        ディレクトリ内の子ディレクトリエントリーを取得します。
    *
    * @param[out]   outNumEntries   取得したエントリー数
    * @param[out]   outEntries      取得したエントリー情報
    *                               配列のサイズは numEntries 以上である必要があります。
    * @param[in]    numEntries      取得するエントリー数
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Read(
                       int32_t* outNumEntries,
                       nn::fs::DirectoryEntry outEntries[],
                       int32_t numEntries
                   ) NN_NOEXCEPT NN_OVERRIDE;

private:
    /**
    * @brief        コンストラクタ
    */
    explicit SaveDataDirectory(os::Mutex& locker) NN_NOEXCEPT;

    /**
    * @brief        ディレクトリオブジェクトを初期化します。
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize() NN_NOEXCEPT;

    /**
    * @brief        内部ディレクトリオブジェクトを取得します。
    *
    * @return       内部ディレクトリオブジェクトへのポインタ
    */
    DirectoryObject* GetDirectoryObject() NN_NOEXCEPT
    {
        return &m_DirectoryObject;
    }

    /**
    * @brief        削除の通知を送ります。列挙が破綻しないように調整します。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result NotifyDelete(int64_t id, bool isFile) NN_NOEXCEPT NN_OVERRIDE;

private:
    os::Mutex& m_LockObject;                 //! ロックオブジェクト
    DirectoryObject m_DirectoryObject;       //! ディレクトリオブジェクト

private:
    // セーブデータアーカイブに内部ディレクトリオブジェクトを公開します。
    friend class SaveDataFileSystemCore;
};

}}}

