﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_IntUtil.h>
#include <nn/fs/fs_Result.h>

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/dbm/fs_Bitmap.h>

class MappingTableTest;
class MappingTableMarkUpdateTest;
class MappingTableReadEntryTest;
class MappingTableCommitTest;
class MappingTableIteratorTest;
class MappingTableExpandTest;
class MappingTableAgingTest;
class MappingTableStressTest;

namespace nn { namespace fssystem { namespace save {

/**
* @brief マッピングテーブル
*
* @details
*
* ○ マッピングテーブルについて
*
* 与えられたアドレス領域をブロックに分割して管理します。
* ブロックはファイル書き込みによって消費され、分断が発生します。
* この時、論理インデックスから物理インデックスへの変換をマッピングテーブルで管理します。
* 実データ領域にはマッピングテーブル経由でアクセスします。
*
* フォーマット時には、実データ領域のサイズと共に予約領域サイズを渡します。
* この予約領域サイズ分のブロックは未割当状態を維持します。
* 未割当状態のブロックは未割当ビットマップで管理されます。
*
* ファイル書き込みが発生すれば上書き済みビットマップに登録します。
* コミットや使用されなくなった段階で上書き済みビットマップから
* 未割当ビットマップに移動します。
*
* ○ データ構成について
*
* マッピングテーブルのメタデータは以下の要素から構成されます。
* ・マッピングテーブル
* ・上書き済みビットマップ(物理インデックスベース)
* ・上書き済みビットマップ(論理インデックスベース)
* ・未割当ビットマップ(物理インデックスベース)
* これらはメタデータファイル上で管理されます。
* MappingTableStorage::QueryMappingMetaSize を用いることで
* メタデータ領域のサイズを求めることができます。
*/
class MappingTable
{
private:
    //! マッピングテーブルのエントリー
    struct MappingEntry
    {
        uint32_t indexPhysical;      //! 物理インデックス
        uint32_t indexPhysicalPrev;  //! 更新前の物理インデックス
    };
    NN_STATIC_ASSERT(std::is_pod<MappingEntry>::value);

public:
    typedef uint32_t Index;

    static uint32_t MakeIndex(int64_t value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(value >= 0);
        NN_SDK_ASSERT(nn::util::IsIntValueRepresentable<uint32_t>(value));
        return static_cast<uint32_t>(value);
    }

    //! マッピングテーブルの管理領域
    struct ControlArea
    {
        uint32_t version;                   //! バージョン
        uint32_t countMapEntry;             //! マッピングテーブルのエントリー数
        uint32_t countReserved;             //! 予約領域分のエントリー数
        uint32_t _reserved;
    };
    NN_STATIC_ASSERT(std::is_pod<ControlArea>::value);
    NN_STATIC_ASSERT(sizeof(ControlArea) == 16);

    //! イテレーター
    class Iterator
    {
        friend class MappingTable;

    private:
        uint32_t m_IndexEntry;      //! マッピングテーブルインデックス
        uint32_t m_IndexPhysical;   //! 物理インデックス
        uint32_t m_CountContinue;   //! 連続ブロック数
        uint32_t m_CountRemain;     //! 残りブロック数

    public:
        Iterator() NN_NOEXCEPT
        : m_IndexEntry(0)
        , m_IndexPhysical(0)
        , m_CountContinue(0)
        , m_CountRemain(0)
        {
        }

        //! 物理インデックスを取得します。
        uint32_t GetPhysicalIndex() const NN_NOEXCEPT
        {
            return m_IndexPhysical;
        }

        //! 連続ブロック数を取得します。
        uint32_t GetBlockCount() const NN_NOEXCEPT
        {
            return m_CountContinue;
        }
    };

public:
    //! コンストラクタ
    MappingTable() NN_NOEXCEPT;

    //! デストラクタ
    ~MappingTable() NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルのメタデータサイズを取得します。
    *
    * @param[out]   outTableSize                    マッピングテーブルに必要なサイズ
    * @param[out]   outBitmapSizeUpdatedPhysical    更新済みビットマップ(物理インデックス)に必要なサイズ
    * @param[out]   outBitmapSizeUpdatedVirtual     更新済みビットマップ(論理インデックス)に必要なサイズ
    * @param[out]   outBitmapSizeUnassigned         未割当ビットマップ(物理インデックス)に必要なサイズ
    * @param[in]    countMapEntry                   マッピングテーブルのエントリ数
    * @param[in]    countReserved                   予約領域のエントリ数
    */
    static void QueryMappingMetaSize(
                    int64_t* outTableSize,
                    int64_t* outBitmapSizeUpdatedPhysical,
                    int64_t* outBitmapSizeUpdatedVirtual,
                    int64_t* outBitmapSizeUnassigned,
                    uint32_t countMapEntry,
                    uint32_t countReserved
                ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルのメタデータ領域をフォーマットします。
    *
    * @param[in]    storageControlArea              管理領域を配置するレイヤー
    * @param[in]    storageTable                    マッピングテーブルのメタデータを配置するレイヤー
    * @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    countMapEntry                   マッピングテーブルのエントリ数
    * @param[in]    countReserved                   予約領域のエントリ数
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      fs::SubStorage storageControlArea,
                      fs::SubStorage storageTable,
                      fs::SubStorage storageBitmapUpdatedPhysical,
                      fs::SubStorage storageBitmapUpdatedVirtual,
                      fs::SubStorage storageBitmapUnassigned,
                      uint32_t countMapEntry,
                      uint32_t countReserved
                  ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルのメタデータ領域を拡張します。
    *
    * @param[in]    storageControlArea              管理領域を配置するレイヤー
    * @param[in]    storageTable                    マッピングテーブルを配置するレイヤー
    * @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    countMapEntry                   拡張後のマッピングテーブルのエントリ数
    * @param[in]    countReserved                   拡張後の予約領域のエントリ数
    *
    * @return       関数の処理結果を返します。
    *
    * @details      マッピングテーブルが管理するブロック数を拡張します。
    *               マウントしていない状態で実行してください。
    */
    static Result Expand(
                      fs::SubStorage storageControlArea,
                      fs::SubStorage storageTable,
                      fs::SubStorage storageBitmapUpdatedPhysical,
                      fs::SubStorage storageBitmapUpdatedVirtual,
                      fs::SubStorage storageBitmapUnassigned,
                      uint32_t countMapEntryNew,
                      uint32_t countReservedNew
                  ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルとしてマウントします。
    *
    * @param[in]    storageControlArea              管理領域を配置するレイヤー
    * @param[in]    storageTable                    マッピングテーブルを配置するレイヤー
    * @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
               fs::SubStorage storageControlArea,
               fs::SubStorage storageTable,
               fs::SubStorage storageBitmapUpdatedPhysical,
               fs::SubStorage storageBitmapUpdatedVirtual,
               fs::SubStorage storageBitmapUnassigned
           ) NN_NOEXCEPT;

    /**
    * @brief        コミット操作を実行します。
    *
    * @return       関数の処理結果を返します。
    */
    Result Commit() NN_NOEXCEPT;

    /**
    * @brief        変更を巻き戻します。
    *
    * @return       関数の処理結果を返します。
    */
    Result Rollback() NN_NOEXCEPT;

    /**
    * @brief        論理インデックスから物理インデックスに変換します。
    *
    * @param[out]   outValue    物理インデックス
    * @param[in]    index       論理インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    Result GetPhysicalIndex(uint32_t* outValue, uint32_t index) NN_NOEXCEPT;

    /**
    * @brief        データ書き込み前にマッピングテーブルを更新します。
    *
    * @param[out]   outNeedCopyHead 先頭ブロックのコピーが必要かどうか
    * @param[out]   outHeadSrc      書き込み前の先頭インデックス
    * @param[out]   outHeadDst      書き込み先の先頭インデックス
    * @param[out]   outNeedCopyTail 終端ブロックのコピーが必要かどうか
    * @param[out]   outTailSrc      書き込み前の終端インデックス
    * @param[out]   outTailDst      書き込み先の終端インデック
    * @param[in]    indexVirtual    書き込み開始論理インデックス
    * @param[in]    countContinue   確保する連続ブロック数
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outNeedCopyHead != nullptr
    *               - outHeadSrc != nullptr
    *               - outHeadDst != nullptr
    *               - outNeedCopyTail != nullptr
    *               - outTailSrc != nullptr
    *               - outTailDst != nullptr
    */
    Result MarkUpdate(
               bool* outNeedCopyHead,
               uint32_t* outHeadSrc,
               uint32_t* outHeadDst,
               bool* outNeedCopyTail,
               uint32_t* outTailSrc,
               uint32_t* outTailDst,
               uint32_t indexVirtual,
               uint32_t countContinue
           ) NN_NOEXCEPT;

    /**
    * @brief        イテレーションを開始します。
    *
    * @param[out]   it              イテレーター
    * @param[in]    indexVirtual    開始論理インデックス
    * @param[in]    countMax        イテレーションするブロック数
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - it != nullptr
    */
    Result MakeIterator(
               Iterator* it,
               uint32_t indexVirtual,
               uint32_t countMax
           ) NN_NOEXCEPT;

    /**
    * @brief        イテレーターが指す位置を更新します。
    *
    * @param[in]    it イテレーター
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - it != nullptr
    */
    Result UpdateIterator(Iterator* it) NN_NOEXCEPT;

    /**
    * @brief       キャッシュを無効化します。
    */
    Result InvalidateCache() NN_NOEXCEPT;

private:
    /**
    * @brief        総ブロック数を取得します。
    *
    * @return       総ブロック数を返します。
    */
    uint32_t GetTotalBlockCount() const NN_NOEXCEPT;

    /**
    * @brief        物理インデックスとして範囲内に収まっているかを確認します。
    *
    * @param[in]    index  物理インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       物理インデックスとして正常な値
    * @retval       ResultOutOfRange    物理インデックスではない
    * @retval       ResultOutOfRange    範囲内に収まっていない
    */
    Result CheckPhysicalIndex(uint32_t index) const NN_NOEXCEPT;

    /**
    * @brief        論理インデックスとして範囲内に収まっているかを確認します。
    *
    * @param[in]    index   論理インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       物理インデックスとして正常な値
    * @retval       ResultOutOfRange    物理インデックスではない
    * @retval       ResultOutOfRange    範囲内に収まっていない
    */
    Result CheckVirtualIndex(uint32_t index) const NN_NOEXCEPT;

    /**
    * @brief    マッピングテーブルからエントリーを読み込みます。
    *
    * @param[out] outValue  エントリー読込先
    * @param[in] index      論理インデックス
    *
    * @return   関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    Result ReadMappingEntry(MappingEntry* outValue, uint32_t index) NN_NOEXCEPT;

    /**
    * @brief    マッピングテーブルから複数のエントリーを読み込みます。
    *
    * @param[out] outValue      エントリー読込先
    * @param[in] bufferCount    エントリー読込先バッファの個数
    * @param[in] index          読み込むエントリー先頭の論理インデックス
    * @param[in] count          読み込むエントリーの個数
    *
    * @return   関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    *               - bufferCount >= count
    */
    Result ReadMappingEntries(
               MappingEntry* outValue,
               size_t bufferCount,
               uint32_t index,
               size_t count
           ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルにエントリーを保存します。
    *
    * @param[in]    entry   書き込むエントリー
    * @param[in]    index   論理インデックス
    *
    * @return       関数の処理結果を返します。
    */
    Result WriteMappingEntry(
               const MappingEntry& entry,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルに複数のエントリーを保存します。
    *
    * @param[in]    pEntries    書き込むエントリー
    * @param[in]    count       エントリーの個数
    * @param[in]    index       書き込むエントリー先頭の論理インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - pEntries != nullptr
    *               - 0 <= count
    */
    Result WriteMappingEntries(
               const MappingEntry* pEntries,
               int count,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        要求したブロック数に応じた連続ブロックを確保します。
    *
    * @param[out]   outIndexDst     改変対象物理インデックス
    * @param[out]   outCountAssign  割当できた連続ブロック数
    * @param[in]    indexStart      割り当てる論理インデックス
    * @param[in]    countRequired   確保したいブロック数
    *
    * @return       関数の処理結果を返します。
    *
    * @details      実際に確保できたサイズは
    *               outCountAssign * MappingTable::GetBlockCount()
    *               で計算できます。
    *               指定したサイズに届かない場合は、確保できたサイズを引いてから
    *               再度この関数を実行してください。
    *
    *               未割当領域が足りなくなると、どこまで書き込んだかわからない状態になります。
    *               その時には、エラーを示す Result を返します。
    *
    * @pre
    *               - outIndexDst != nullptr
    *               - outCountAssign != nullptr
    */
    Result MarkAssignment(
               uint32_t* outIndexDst,
               uint32_t* outCountAssign,
               uint32_t indexStart,
               uint32_t countRequired
           ) NN_NOEXCEPT;

    /**
    * @brief        データ書き込み前にマッピングテーブルを更新します。
    *
    * @param[out]   outNeedCopyHead 先頭ブロックのコピーが必要かどうか
    * @param[out]   outHeadSrc      書き込み前の先頭物理インデックス
    * @param[out]   outHeadDst      書き込み先の先頭物理インデックス
    * @param[out]   outNeedCopyTail 終端ブロックのコピーが必要かどうか
    * @param[out]   outTailSrc      書き込み前の終端物理インデックス
    * @param[out]   outTailDst      書き込み先の終端物理インデック
    * @param[in]    indexStart      書き込み開始論理インデックス
    * @param[in]    countContinue   確保する連続ブロック数
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outNeedCopyHead != nullptr
    *               - outHeadSrc != nullptr
    *               - outHeadDst != nullptr
    *               - outNeedCopyTail != nullptr
    *               - outTailSrc != nullptr
    *               - outTailDst != nullptr
    */
    Result MarkUpdateRecursively(
               bool* outNeedCopyHead,
               uint32_t* outHeadSrc,
               uint32_t* outHeadDst,
               bool* outNeedCopyTail,
               uint32_t* outTailSrc,
               uint32_t* outTailDst,
               uint32_t indexStart,
               uint32_t countContinue
           ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブル上の論理インデックス、物理インデックスの変換測を更新します。
    *
    * @param[in]    indexPhysical   物理インデックス
    * @param[in]    indexVirtual    論理インデックス
    * @param[in]    countContinue   ブロック数(1以上)
    *
    * @return       関数の処理結果を返します。
    */
    Result UpdateMapTable(
               uint32_t indexPhysical,
               uint32_t indexVirtual,
               uint32_t countContinue
           ) NN_NOEXCEPT;

    /**
    * @brief        未割当ビットマップに対して、全ての未使用になった範囲のビットを立てます。
    *
    * @param[in]    isCommit    コミットかどうか
    *
    * @return       関数の処理結果を返します。
    *
    * @details      コミットなら更新前に割り当てられていた領域を未使用にし、
    *               ロールバックなら新たに割り当てられた領域を未使用にします。
    */
    Result AddUnassignedListIteratively(
               bool isCommit
           ) NN_NOEXCEPT;

    /**
    * @brief        未割当ビットマップに対して、未使用になった範囲のビットを立てます。
    *
    * @param[in]    index       開始論理インデックス
    * @param[in]    counOne     連続論理ブロック数
    * @param[in]    isCommit    コミットかどうか
    *
    * @return       関数の処理結果を返します。
    *
    * @details      コミットなら更新前に割り当てられていた領域を未割当にし、
    *               ロールバックなら新たに割り当てられた領域を未割当にします。
    */
    Result AddUnassignedList(
               uint32_t index,
               uint32_t countOne,
               bool isCommit
           ) NN_NOEXCEPT;

    /*!
    * @brief        未割当状態のエントリー数を取得します。
    *
    * @param[out]   outCount      未割当状態のエントリー数
    *
    * @return       関数の処理結果を返します。
    *
    * @details      デバッグ用の機能です。
    *
    * @pre
    *               - outCount != nullptr
    */
    Result GetUnassignedCount(uint32_t* outCount) NN_NOEXCEPT;

private:
    //! 管理領域のバージョン
    static const uint32_t CONTROL_AREA_VERSION = 1;

    //! 物理インデックスを示すマスク
    static const uint32_t PHYSICAL_INDEX_MASK = 0x80000000;

    //! 物理インデックスに変換します。
    static uint32_t PhysicalIndex(uint32_t index)
    {
        return (index | PHYSICAL_INDEX_MASK);
    }

    //! 物理インデックスかどうか。
    static bool IsPhysicalIndex(uint32_t index)
    {
        return ((index & PHYSICAL_INDEX_MASK) != 0);
    }

    //! 物理インデックスをインデックスに変換します。
    static uint32_t GetValidIndex(uint32_t index)
    {
        return (index & ~PHYSICAL_INDEX_MASK);
    }

private:
    ControlArea m_ControlArea;                          //! 管理領域
    fs::SubStorage m_StorageTable;                      //! マッピングテーブル用ストレージ
    fs::SubStorage m_StorageBitmapUpdatedPhysical;      //! 更新済みビットマップ(物理インデックス)用ストレージ
    fs::SubStorage m_StorageBitmapUpdatedVirtual;       //! 更新済みビットマップ(論理インデックス)用ストレージ
    fs::SubStorage m_StorageBitmapUnassigned;           //! 未割当ビットマップ(物理インデックス)用ストレージ
    dbm::Bitmap m_BitmapUpdatedPhysical;                //! 更新済みビットマップ(物理インデックス)
    dbm::Bitmap m_BitmapUpdatedVirtual;                 //! 更新済みビットマップ(論理インデックス)
    dbm::Bitmap m_BitmapUnassigned;                     //! 未割当ビットマップ(物理インデックス)

private:
    // テスト用クラスには private API を公開します。
    friend class ::MappingTableTest;
    friend class ::MappingTableMarkUpdateTest;
    friend class ::MappingTableReadEntryTest;
    friend class ::MappingTableCommitTest;
    friend class ::MappingTableIteratorTest;
    friend class ::MappingTableExpandTest;
    friend class ::MappingTableAgingTest;
    friend class ::MappingTableStressTest;
};

}}} // namespace nn::fssrv::save

