﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/save/fs_ProxyFileSystemWithRetryingBufferAllocation.h>
#include <nn/fssystem/save/fs_SaveStorageFile.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystem.h>
#include <nn/fssystem/fs_IMacGenerator.h>

namespace nn { namespace fssystem { namespace save {

// セーブデータ管理領域のためのパラメータ
struct IntegritySaveDataParameters
{
    save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
    uint32_t         countDataBlock;
    int32_t          blockSize;
};
NN_STATIC_ASSERT(std::is_pod<IntegritySaveDataParameters>::value);

// セーブデータクラス
class IntegritySaveDataFileSystemDriver : public ProxyFileSystemWithRetryingBufferAllocation
{
    NN_DISALLOW_COPY(IntegritySaveDataFileSystemDriver);

public:
    IntegritySaveDataFileSystemDriver() NN_NOEXCEPT;

    virtual ~IntegritySaveDataFileSystemDriver() NN_NOEXCEPT;

    //! 空き領域のサイズをバイト数単位で取得します。
    virtual Result GetFreeBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_SaveDataFileSystem.GetFreeBytes(outValue));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    //! 実データ領域のサイズをバイト数単位で取得します。
    virtual Result GetDataAreaBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_SaveDataFileSystem.GetDataAreaBytes(outValue));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        セーブデータ ファイルシステムをフォーマットするのに必要なデータサイズを計算します。
    *
    * @param[out]   outSizeTotal        ファイルシステムを使用するために最適なサイズを格納する領域
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    paramIntegrity      完全性検証パラメータ
    * @param[in]    countDataBlock      データブロック数
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outSizeTotal != nullptr
    */
    static Result QueryTotalSize(
                      int64_t* outSizeTotal,
                      size_t sizeBlock,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
                      uint32_t countDataBlock
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータ領域をフォーマットします。
    *
    * @param[in]    baseStorage         セーブデータ領域として使用するストレージ
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    paramIntegrity      完全性検証パラメータ
    * @param[in]    countDataBlock      データブロック数
    * @param[in]    pBufferManager      バッファマネージャ
    * @param[in]    pMacGenerator       MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - pBufferManager != nullptr
    */
    static Result Format(
                      fs::SubStorage baseStorage,
                      size_t sizeBlock,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
                      uint32_t countDataBlock,
                      IBufferManager* pBufferManager,
                      IMacGenerator* pMacGenerator
                  ) NN_NOEXCEPT;

    /**
    * @brief        拡張データを読み込みます
    *
    * @param[out]   outData        拡張データ
    * @param[in]    baseStorage    セーブデータ領域として使用するストレージ
    * @param[in]    pBufferManager バッファマネージャ
    * @param[in]    pMacGenerator  MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    *
    * @details      最後にコミットした際のデータを読み込みます。
    *               マウントしていなくても読み込むことが可能です。
    */
    static Result ReadExtraData(
                      IntegritySaveDataFileSystem::ExtraData* outData,
                      fs::SubStorage baseStorage,
                      IBufferManager* pBufferManager,
                      IMacGenerator* pMacGenerator
                  ) NN_NOEXCEPT;

    //! セーブデータをマウントします。
    Result Initialize(fs::SubStorage baseStorage, IBufferManager* pBufferManager, IMacGenerator* pMacGenerator) NN_NOEXCEPT;

    //! セーブデータをアンマウントします。
    void Finalize() NN_NOEXCEPT;

    //! 拡張データを更新します
    Result WriteExtraData(const IntegritySaveDataFileSystem::ExtraData& extraData) NN_NOEXCEPT;

    //! 拡張データを取得します
    //! WriteExtraData で書き込んだデータを読み込みます
    void ReadExtraData(IntegritySaveDataFileSystem::ExtraData* outData) NN_NOEXCEPT;

    //! セーブデータ ファイルシステムに対してコミットを行います。
    Result Commit() NN_NOEXCEPT;

    //! セーブデータ管理領域のパラメータを自動設定します。
    static IntegritySaveDataParameters SetUpSaveDataParameters(int32_t blockSize, int64_t size) NN_NOEXCEPT;

private:
    fs::SubStorage m_BaseStorage;
    os::Mutex m_Locker;                                      //!< ファイルシステムレベルの排他制御オブジェクト
    IBufferManager*             m_pBufferManager;            //!< バッファマネージャ
    FilesystemBufferManagerSet  m_BufferManagerSet;          //!< バッファマネージャセット
    IntegritySaveDataFileSystem m_SaveDataFileSystem;        //!< セーブデータファイルシステム
};

}}}
