﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/util/util_Optional.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>
#include <nn/fssystem/save/fs_SaveFileStorage.h>
#include <nn/fssystem/save/fs_SaveFileInFile.h>
#include <nn/fssystem/save/fs_IntegritySaveDataStorage.h>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nn/fssystem/save/fs_BufferedStorage.h>
#include <nn/fssystem/fs_IMacGenerator.h>

class FsIntegritySaveDataFileSystemTest;

namespace nn { namespace fssystem { namespace save {

/**
* @brief 完全性検証を行う一時揮発ファイルシステムクラスです。
*/
class IntegritySaveDataFileSystem : public IFileSystem
{
    NN_DISALLOW_COPY(IntegritySaveDataFileSystem);
public:
    //! タイムスタンプ、位置情報などを書き込むための拡張データ
    struct ExtraData
    {
        char data[512];
    };
    NN_STATIC_ASSERT(std::is_pod<ExtraData>::value);

public:
    typedef void (*GenerateRandomFunction)(void* pData, size_t size);

private:
    static const size_t ControlAreaHashSize = nn::crypto::Sha256Generator::HashSize;

    /**
    * ファイルシステム レイアウト情報ヘッダ
    *
    * ファイルシステム レイアウト情報ヘッダに相当する部分は
    * 完全性検証範囲外とせざるを得ません。
    * マスター管理領域内に配置し、マスター管理領域によって保護されます。
    */
    struct FileSystemLayoutHeader
    {
        int32_t magic;                                  //!< 初期化チェック用コード
        int32_t version;                                //!< アーカイバーのバージョン

        char hash[ControlAreaHashSize];                 //!< 管理領域のハッシュ

        // ファイルシステム内
        int64_t offsetFileSystem;
        int64_t sizeFileSystem;

        int64_t offsetsMeta;
        int64_t sizeMeta;
        int64_t offsetSaveData;
        int64_t sizeSaveDataCoreOriginal;
        int64_t sizeSaveDataCore;

        // マスター管理領域内
        int64_t offsetMasterHash;
        int64_t sizeMasterHash;

        // メタデータ内
        int64_t offsetHierarchicalIntegrityLayeredHashL1;
        int64_t sizeHierarchicalIntegrityLayeredHashL1;
        int64_t offsetHierarchicalIntegrityLayeredHashL2;
        int64_t sizeHierarchicalIntegrityLayeredHashL2;
        int64_t offsetHierarchicalIntegrityLayeredHashL3;
        int64_t sizeHierarchicalIntegrityLayeredHashL3;
        int64_t offsetSaveDataMeta;
        int64_t sizeSaveDataMeta;

        char reserved[336];
    };
    NN_STATIC_ASSERT(sizeof(FileSystemLayoutHeader) == 512);
    NN_STATIC_ASSERT(std::is_pod<FileSystemLayoutHeader>::value);

    //! マスター管理領域
    struct MasterHeader
    {
        static const int SignatureSize = 256; //!< 署名サイズ

        struct ControlArea
        {
            char signature[SignatureSize];    //!< 署名
            FileSystemLayoutHeader header;    //!< ファイルシステムヘッダ
        } controlArea;
        NN_STATIC_ASSERT(std::is_pod<ControlArea>::value);

        HierarchicalIntegrityVerificationMetaInformation hierarchicalIntegrityControlArea;
        SaveDataFileSystemCore::FileSystemHeader saveDataControlArea;

        ExtraData extraData;                    //!< 拡張データ
        ExtraData originalExtraData;            //!< 更新前の拡張データ

        // マスターハッシュ
        // 16 KB まで 0 埋め
        char reserved[16 * 1024
            - sizeof(ControlArea)
            - sizeof(HierarchicalIntegrityVerificationMetaInformation)
            - sizeof(SaveDataFileSystemCore::FileSystemHeader)
            - sizeof(ExtraData) * 2];
    };
    NN_STATIC_ASSERT(sizeof(MasterHeader) == 16 * 1024);
    NN_STATIC_ASSERT(std::is_pod<MasterHeader>::value);

    //! 管理領域を保持するためのクラス
    class ControlAreaHolder
    {
    public:
        ControlAreaHolder() NN_NOEXCEPT;
        Result Reset(const MasterHeader* pAtomic) NN_NOEXCEPT;
        void Clear() NN_NOEXCEPT;
        void Commit(MasterHeader* pAtomic) const NN_NOEXCEPT;
        ExtraData* GetExtraData() NN_NOEXCEPT;
        const nn::fs::SubStorage GetIntegrityControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetSaveDataControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetMasterHashStorage() NN_NOEXCEPT;

    private:
        static size_t GetControlAreaSize() NN_NOEXCEPT;

    private:
        std::unique_ptr<char[], nn::fs::detail::Deleter> m_pHeader;
        std::unique_ptr<nn::fs::MemoryStorage> m_pStorage;
        size_t m_BufferSize;
    };

public:
    /**
    * @brief        完全性検証機能付きセーブデータファイルシステムをフォーマットするために必要なサイズを取得します。
    *
    * @param[out]   outSize                 完全性検証に必要なサイズを取得します。
    * @param[in]    inputParamIntegrity     完全性検証パラメータ
    * @param[in]    sizeBlock               ブロックサイズ
    * @param[in]    countDataBlock          データブロック数
    *
    * @return       関数の処理結果を返します。
    */
    static Result QuerySize(
                      int64_t* outSize,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock
                  ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証機能付きセーブデータファイルシステムとしてフォーマットします。
    *
    * @param[in]    storage ストレージ
    * @param[in]    inputParamIntegrity 完全性検証パラメータ
    * @param[in]    sizeBlock ブロックサイズ
    * @param[in]    countDataBlock データブロック数
    * @param[in]    pIntegrityCacheBufferSet キャッシュマネージャ
    * @param[in]    pCacheBuffer キャッシュマネージャ
    * @param[in]    pLocker キャッシュマネージャのスレッド間同期オブジェクト
    * @param[in]    pMacGenerator MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      fs::SubStorage storage,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock,
                      FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
                      IBufferManager* pCacheBuffer,
                      os::Mutex* pLocker,
                      IMacGenerator* pMacGenerator
                  ) NN_NOEXCEPT;

    /**
    * @brief        拡張データを取得します。
    *
    * @param[out]   outData         拡張データ
    * @param[in]    storage         ストレージ
    * @param[in]    pBufferManager  バッファマネージャ
    * @param[in]    pMacGenerator   MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    *
    * @details      最後にコミットした際の拡張データを読み込みます。
    *               すなわち、WriteExtraData で書き込んだデータは
    *               コミットするまでこの関数では読み込めません。
    *               マウントしていなくても実行可能です。
    */
    static Result ReadExtraData(
                      ExtraData* outData,
                      fs::SubStorage storage,
                      IBufferManager* pBufferManager,
                      IMacGenerator* pMacGenerator
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタ
    */
    IntegritySaveDataFileSystem() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~IntegritySaveDataFileSystem() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        完全性検証付きセーブデータファイルシステムをマウントします。
    *
    * @param[in]    storage                     ストレージ
    * @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
    * @param[in]    pCacheBuffer       キャッシュマネージャ
    * @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
    * @param[in]    pMacGenerator               MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
               fs::SubStorage storage,
               FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
               IBufferManager* pCacheBuffer,
               os::Mutex* pLocker,
               IMacGenerator* pMacGenerator
           ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証付きセーブデータファイルシステムをアンマウントします。
    *
    * @details      アンマウント時に自動Commitは行いません。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ファイルシステムに対してコミット操作を行います。
    *
    * @return       関数の処理結果を返します。
    */
    Result CommitFileSystem() NN_NOEXCEPT
    {
        NN_RESULT_THROW(CommitFileSystemCore());
    }

    /**
    * @brief        拡張データを更新します。
    *
    * @param[in]    extraData   拡張データ
    *
    * @return       関数の処理結果を返します。
    */
    Result WriteExtraData(
               const ExtraData& extraData
           ) NN_NOEXCEPT;

    /**
    * @brief        拡張データを取得します。
    *
    * @param[out]   outData     拡張データ
    *
    * @details      WriteExtraData で書き込んだ値が読み込まれます。
    */
    void ReadExtraData(
             ExtraData* outData
         ) NN_NOEXCEPT;

public:
    /**
    * @brief        ファイルを作成します。
    *
    * @param[in]    path    ファイルパス
    * @param[in]    size    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateFile(const Path& path, int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    path ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outValue    ファイルオブジェクト
    * @param[in]    path        ファイルパス
    * @param[in]    mode        オープンモード
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenFile(
                       IFile** outValue,
                       const Path& path,
                       int mode
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outValue    ディレクトリオブジェクト
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenDirectory(
                       IDirectory** outValue,
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを閉じます。
    *
    * @param[in]    pFile   ファイルオブジェクト
    */
    virtual void CloseFile(IFile* pFile) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを閉じます。
    *
    * @param[in]    pDirectory  ディレクトリオブジェクト
    */
    virtual void CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを削除します。
    *
    * @param[in]    path    ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteFile(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを再帰的に削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectoryRecursively(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        全ての子エントリを再帰的に削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CleanDirectoryRecursively(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルをリネームします。
    *
    * @param[in]    oldPath 変更対象のファイルパス
    * @param[in]    newPath 変更後のファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameFile(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリをリネームします。
    *
    * @param[in]    oldPath 変更対象のディレクトリパス
    * @param[in]    newPath 変更後のディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameDirectory(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したファイルが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ファイルが存在しているかどうか
    * @param[in]    path        ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasFile(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したディレクトリが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ディレクトリが存在しているかどうか
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasDirectory(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        空き領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    空き領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetFreeBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実データ領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    実データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetDataAreaBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

   /**
    * @brief        乱数列生成関数をセットします。
    *
    * @return       なし
    */
    static void SetGenerateRandomFunction(
        GenerateRandomFunction function
    ) NN_NOEXCEPT;

private:
    typedef decltype(buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(nullptr, nullptr)) MasterHeaderBufferType;

private:
   /**
    * @brief        マスター管理領域を読み込みます。
    *
    * @param[out]   outHeader                    マスター管理領域の読み込み先
    * @param[in]    headerBufferSize             読み込み先バッファのサイズ
    * @param[in]    storage                      マスター管理領域を読み込むストレージ
    * @param[in]    pMacGenerator                MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outHeader != nullptr
    *               - outIsProvisionallyCommitted != nullptr
    *               - outIsFirstControlAreaMounted != nullptr
    */
    static Result ReadMasterControlArea(
                      MasterHeader* outHeader,
                      size_t headerBufferSize,
                      fs::SubStorage storage,
                      IMacGenerator* pMacGenerator
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータをマウントします。
    *
    * @param[in]    layoutHeader                レイアウトヘッダ
    * @param[in]    storageFileSystem           セーブデータファイルシステムをマウントするストレージ
    * @param[in]    sizeBlock                   ブロックサイズ
    * @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
    * @param[in]    pCacheBuffer                キャッシュマネージャ
    * @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
    *
    * @return       関数の処理結果を返します。
    */
    Result MountSaveData(
               const FileSystemLayoutHeader& layoutHeader,
               fs::SubStorage storageFileSystem,
               size_t sizeBlock,
               FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
               IBufferManager* pCacheBuffer,
               os::Mutex* pLocker
           ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証セーブデータをマウントします。
    *
    * @param[in]    layoutHeader                レイアウトヘッダ
    * @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
    * @param[in]    pSaveDataCoreCacheBufferSet キャッシュマネージャ
    * @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
    *
    * @return       関数の処理結果を返します。
    */
    Result MountIntegritySaveData(
               const FileSystemLayoutHeader& layoutHeader,
               FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
               IBufferManager* pSaveDataCoreCacheBuffer,
               os::Mutex* pLocker
           ) NN_NOEXCEPT;

    /**
    * @brief        ファイルシステムをコミットします。
    *
    * @return       関数の処理結果を返します。
    */
    Result CommitFileSystemCore() NN_NOEXCEPT;

    /**
    * @brief        署名を付けます。
    *
    * @param[in]    pStorage      計算したハッシュを格納するストレージ
    * @param[in]    pMacGenerator MAC 生成クラス
    * @param[in]    pHash         計算したハッシュを格納するバッファ
    * @param[in]    hashSize      pHash のサイズ
    * @param[in]    pData         計算対象データ
    * @param[in]    dataSize      pData のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Sign(
                      fs::IStorage* pStorage,
                      IMacGenerator* pMacGenerator,
                      void* pHash,
                      size_t hashSize,
                      const void* pData,
                      size_t dataSize
                  ) NN_NOEXCEPT;

    /**
    * @brief        データの正当性を検証します。
    *
    * @param[in]    pMacGenerator MAC 生成クラス
    * @param[in]    pData         計算対象データ
    * @param[in]    dataSize      pData のサイズ
    * @param[in]    pSignature    比較に使用する署名
    * @param[in]    signatureSize pSignature のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Verify(
                      IMacGenerator* pMacGenerator,
                      const void* pData,
                      size_t dataSize,
                      const void* pSignature,
                      size_t signatureSize
                  ) NN_NOEXCEPT;

    /**
    * @brief        マスターヘッダの正当性を検証します。
    *
    * @param[in]    pMasterHeader   マスターヘッダ
    *
    * @return       正当か否かを返します
    */
    static bool VerifyMasterHeaderContent(
                    MasterHeader* pMasterHeader
                ) NN_NOEXCEPT;

    /**
    * @brief        指定のパラメータのセーブデータにおける各データの配置を計算します。
    *
    * @param[out]   outFileSystemLayoutHeader           レイアウトヘッダ
    * @param[in]    inputParamIntegrity                 完全性検証パラメータ
    * @param[in]    sizeBlock                           ブロックサイズ
    * @param[in]    countDataBlock                      データブロック数
    *
    * @return       関数の処理結果を返します。
    */
    static Result FormatFileSystemLayoutHeader(
                      FileSystemLayoutHeader* outFileSystemLayoutHeader,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock
                  ) NN_NOEXCEPT;

    Result GetMasterHeader(MasterHeaderBufferType* outAtomic) NN_NOEXCEPT;

    void StoreMasterHeader(MasterHeader* pAtomic, size_t size) NN_NOEXCEPT;

    os::Mutex* GetLocker() NN_NOEXCEPT
    {
        return m_pCommitLocker;
    }

private:
    ControlAreaHolder m_ControlArea;                     //!< 管理領域
    fs::SubStorage m_BaseStorageControlArea;             //!< 管理領域ストレージ
    BufferedStorage m_StorageBuffering;                  //!< ストレージ用バッファ
    BufferedStorage m_StorageBufferingMeta;              //!< メタ情報ストレージ用バッファ
    fs::SubStorage m_FileSystemStorage;                  //!< ファイルシステム用サブストレージ
    fs::SubStorage m_MetaStorage;                        //!< メタデータ用サブストレージ
    IntegritySaveDataStorage m_IntegrityStorage;         //!< 完全性検証ストレージ
    SaveDataFileSystemCore m_FileSystem;                 //!< 完全性検証対象のファイルシステム
    IBufferManager* m_pBufferManager;                    //!< バッファマネージャ
    IBufferManager::CacheHandle m_MasterHandle;          //!< マスターヘッダを格納したメモリのハンドル
    os::Mutex* m_pCommitLocker;                          //!< ファイルシステムに対する同期オブジェクト
    bool m_IsInitialized;                                //!< 初期化されているかどうか
    bool m_IsExtraDataModified;                          //!< 拡張データが更新されたか
    IMacGenerator* m_pMacGenerator;                      //!< MAC 生成クラス

private:
    friend class IntegritySaveDataFile;
    friend class IntegritySaveDataDirectory;

private:
    // テスト用クラスには実装を公開します。
    friend class ::FsIntegritySaveDataFileSystemTest;
};

/**
* @brief コミット対応ファイルオブジェクトです。
*/
class IntegritySaveDataFile : public FileInFile, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(IntegritySaveDataFile);

private:
    /**
    * @brief        コンストラクタ
    */
    explicit IntegritySaveDataFile(
                 IFile* pFile,
                 IntegritySaveDataFileSystem* pFileSystem
             ) NN_NOEXCEPT;

public:
    /**
    * @brief        デストラクタ
    */
    virtual ~IntegritySaveDataFile() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        下位ファイルオブジェクトへの参照を解除します。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ファイルの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result ReadBytes(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        バッファの内容をファイルに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result WriteBytes(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       失敗の結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

private:
    IntegritySaveDataFileSystem* m_pFileSystem;

private:
    friend class IntegritySaveDataFileSystem;
};

/**
* @brief コミット対応ディレクトリオブジェクトです。
*/
class IntegritySaveDataDirectory : public IDirectory, nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(IntegritySaveDataDirectory);

private:
    /**
    * @brief        コンストラクタ
    */
    explicit IntegritySaveDataDirectory(
                 IDirectory* pDirectory,
                 IntegritySaveDataFileSystem* pFileSystem
             ) NN_NOEXCEPT;

public:
    /**
    * @brief        デストラクタ
    */
    virtual ~IntegritySaveDataDirectory() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        下位ディレクトリオブジェクトへの参照を解除します。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        下位ディレクトリオブジェクトを取得します。
    */
    IDirectory* GetDirectory() NN_NOEXCEPT
    {
        return m_pDirectory;
    }

    /**
    * @brief        ディレクトリ内の子ディレクトリエントリーを取得します。
    *
    * @param[out]   outNumEntries   取得したエントリー数
    * @param[out]   outEntries      取得したエントリー情報
    *                               配列のサイズは numEntries 以上である必要があります。
    * @param[in]    numEntries      取得するエントリー数
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Read(
                       int32_t* outNumEntries,
                       nn::fs::DirectoryEntry outEntries[],
                       int32_t numEntries
                   ) NN_NOEXCEPT NN_OVERRIDE;

private:
    IntegritySaveDataFileSystem* m_pFileSystem;
    IDirectory* m_pDirectory;

private:
    friend class IntegritySaveDataFileSystem;
};

}}}

