﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>

#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>
#include <nn/fssystem/save/fs_SavePath.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief ファイルシステムでの排他アクセス制御可能なエントリークラスです。
*/
class IExclusiveObject
{
    NN_DISALLOW_COPY(IExclusiveObject);

public:
    typedef int64_t FileOrDirectoryId;

public:
    /**
    * @brief コンストラクタ
    */
    IExclusiveObject() NN_NOEXCEPT
        : m_pNext(nullptr)
        , m_Mode(0)
        , m_Id(0)
    {
    }

    /**
    * @brief コンストラクタ
    */
    virtual ~IExclusiveObject() NN_NOEXCEPT
    {
    }

    /**
    * @brief 次のオブジェクトを設定します。
    */
    void SetNext(IExclusiveObject* pNext) NN_NOEXCEPT
    {
        m_pNext = pNext;
    }

    /**
    * @brief 次のオブジェクトを取得します。
    */
    IExclusiveObject* GetNext() const NN_NOEXCEPT
    {
        return m_pNext;
    }

    /**
    * @brief ファイルのオープンモードを設定します。
    */
    void SetMode(int mode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(
            0,
            mode & (~(nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend))
        );
        m_Mode = mode;
    }

    /**
    * @brief ファイルのオープンモードを取得します。
    */
    inline int GetMode() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(
            0,
            m_Mode & (~(nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend))
        );
        return m_Mode;
    }

    /**
    * @brief ファイルかどうかを取得します。
    */
    bool IsFile() const NN_NOEXCEPT
    {
        return m_Mode != 0;
    }

    /**
    * @brief ファイル Id もしくは ディレクトリ Id を設定します。
    */
    void SetId(FileOrDirectoryId id) NN_NOEXCEPT
    {
        m_Id = id;
    }

    /**
    * @brief ファイル Id もしくは ディレクトリ Id を取得します。
    */
    FileOrDirectoryId GetId() const NN_NOEXCEPT
    {
        return m_Id;
    }

    /**
    * @brief リストに繋がっているオブジェクト数を取得します。
    */
    uint32_t GetOpenedObjectCount() NN_NOEXCEPT
    {
        uint32_t count = 0;
        IExclusiveObject* pObj = m_pNext;
        while( pObj != nullptr )
        {
            count++;
            pObj = pObj->GetNext();
        }
        return count;
    }

    /**
    * @brief エントリーの削除を通知します。
    *
    * @param[in] path 削除するパス
    *
    * @return 関数の処理結果を返します。
    */
    virtual Result NotifyDelete(int64_t id, bool isFile) = 0;

private:
    IExclusiveObject *m_pNext;     //!< 次の開いているオブジェクトへのリンク
    int m_Mode;                    //!< ファイルオープンモード
    FileOrDirectoryId m_Id;        //!< ファイル Id もしくは ディレクトリ Id
};

/**
* @brief ファイルシステムでの排他アクセス制御可能なファイルクラスです。
*/
class ExclusiveFile
    : public IFile,
      public IExclusiveObject
{
    NN_DISALLOW_COPY(ExclusiveFile);

public:
    /**
    * @brief コンストラクタ
    */
    ExclusiveFile() NN_NOEXCEPT
    {
    }

    /**
    * @brief デストラクタ
    */
    virtual ~ExclusiveFile() NN_NOEXCEPT
    {
    }

    /**
    * @brief エントリーの削除を通知します。
    *
    * @param[in] path 削除するパス
    *
    * @return 関数の処理結果を返します。
    */
    virtual Result NotifyDelete(int64_t id, bool isFile) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(id);
        NN_UNUSED(isFile);
        NN_SDK_ASSERT(false);
        NN_RESULT_SUCCESS;
    };
};

/**
* @brief ファイルシステムでの排他アクセス制御可能なディレクトリクラスです。
*/
class ExclusiveDirectory
    : public IDirectory,
      public IExclusiveObject
{
    NN_DISALLOW_COPY(ExclusiveDirectory);

public:
    /**
    * @brief コンストラクタ
    */
    ExclusiveDirectory() NN_NOEXCEPT
    {
    }

    /**
    * @brief デストラクタ
    */
    virtual ~ExclusiveDirectory() NN_NOEXCEPT
    {
    }
};

/**
* @brief 排他アクセス対応ファイルシステムです。
*/
class IExclusiveFileSystem : public IFileSystem
{
    NN_DISALLOW_COPY(IExclusiveFileSystem);

private:
    typedef IExclusiveObject::FileOrDirectoryId FileOrDirectoryId;

public:
    /**
    * @brief コンストラクタ
    */
    IExclusiveFileSystem() NN_NOEXCEPT
    : m_LockObject(true)
    , m_pOpened(nullptr)
    {
    }

    /**
    * @brief デストラクタ
    */
    virtual ~IExclusiveFileSystem() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pOpened == nullptr);
    }

    /**
    * @brief        ファイルを作成します。
    *
    * @param[in]    path    ファイルパス
    * @param[in]    size    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateFile(
                       const Path& path,
                       int64_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    path ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outValue    ファイルオブジェクト
    * @param[in]    path        ファイルパス
    * @param[in]    mode        オープンモード
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ファイルオブジェクト用にメモリが確保されます。
    *               確保されたメモリは @ref IExclusiveFileSystem::CloseFile
    *               を用いて解放する必要があります。
    */
    virtual Result OpenFile(
                       IFile** outValue,
                       const Path& path,
                       int mode
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outValue    ディレクトリオブジェクト
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ディレクトリオブジェクト用にメモリが確保されます。
    *               確保されたメモリは @ref IExclusiveFileSystem::CloseDirectory
    *               を用いて解放する必要があります。
    */
    virtual Result OpenDirectory(
                       IDirectory** outValue,
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルオブジェクト用リソースを解放します。
    *
    * @param[in]    pFile   ファイルオブジェクト
    */
    virtual void CloseFile(IFile* pFile) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリオブジェクト用リソースを解放します。
    *
    * @param[in]    pDirectory  ディレクトリオブジェクト
    */
    virtual void CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを削除します。
    *
    * @param[in]    path    ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteFile(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルをリネームします。
    *
    * @param[in]    oldPath 変更対象のファイルパス
    * @param[in]    newPath 変更後のファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameFile(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリをリネームします。
    *
    * @param[in]    oldPath 変更対象のディレクトリパス
    * @param[in]    newPath 変更後のディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameDirectory(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したファイルが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ファイルが存在しているかどうか
    * @param[in]    path        ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasFile(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したディレクトリが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ディレクトリが存在しているかどうか
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasDirectory(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定のモードで開かれているファイルがあるかどうか判定します。
    *
    * @param[in]    mode    検査するオープンモード
    *
    * @return       判定結果を返します。
    *
    * @pre
    *               - mode != 0
    */
    bool HasOpenedFiles(int mode) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(mode, 0);
        for( auto pOpened = m_pOpened; pOpened != nullptr; pOpened = pOpened->GetNext() )
        {
            if( (pOpened->GetMode() & mode) != 0 )
            {
                return true;
            }
        }
        return false;
    }

    /**
    * @brief        開かれているエントリがあるかどうか判定します。
    *
    * @return       判定結果を返します。
    */
    bool HasOpenedEntries() const NN_NOEXCEPT
    {
        return m_pOpened != nullptr;
    }

protected:
    /**
    * @brief        ファイルを作成します。
    *
    * @param[in]    path    ファイルパス
    * @param[in]    size    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateFileImpl(const Path& path, int64_t size) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    path ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateDirectoryImpl(const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outValue    ファイルオブジェクト
    * @param[in]    path        ファイルパス
    * @param[in]    mode        オープンモード
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenFileImpl(
                       ExclusiveFile** outValue,
                       const Path& path,
                       int mode
                   ) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outValue    ディレクトリオブジェクト
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenDirectoryImpl(
                       ExclusiveDirectory** outValue,
                       const Path& path
                   ) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルを閉じます。
    *
    * @param[in]    pFile   ファイルオブジェクト
    */
    virtual void CloseFileImpl(ExclusiveFile* pFile) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを閉じます。
    *
    * @param[in]    pDirectory  ディレクトリオブジェクト
    */
    virtual void CloseDirectoryImpl(ExclusiveDirectory* pDirectory) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルを削除します。
    *
    * @param[in]    path    ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteFileImpl(const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリを削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectoryImpl(const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルをリネームします。
    *
    * @param[in]    oldPath 変更対象のファイルパス
    * @param[in]    newPath 変更後のファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameFileImpl(const Path& oldPath, const Path& newPath) NN_NOEXCEPT = 0;

    /**
    * @brief        ディレクトリをリネームします。
    *
    * @param[in]    oldPath 変更対象のディレクトリパス
    * @param[in]    newPath 変更後のディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameDirectoryImpl(const Path& oldPath, const Path& newPath) NN_NOEXCEPT = 0;

    /**
    * @brief        指定したファイルが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ファイルが存在しているかどうか
    * @param[in]    path        ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasFileImpl(bool* outValue, const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        指定したディレクトリが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ディレクトリが存在しているかどうか
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasDirectoryImpl(bool* outValue, const Path& path) NN_NOEXCEPT = 0;

    /**
    * @brief        ファイルパスからファイル ID を取得します。
    *
    * @param[out]   outId   ファイル ID
    * @param[in]    path    フルパス
    * @param[in]    isFile  ファイルであれば true、ディレクトリであれば false
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetFileIdFromPath(int64_t *outId, const Path& path, bool isFile) NN_NOEXCEPT = 0;

    os::Mutex* GetFileOperationLocker() NN_NOEXCEPT
    {
        return &m_LockObject;
    }

private:
    /**
    * @brief        このモードでアクセス可能かどうかを診断します。
    *
    * @param[out]   outId               ID
    * @param[out]   outDirectoryOpened  オープンされているかどうか
    * @param[in]    path                パス
    * @param[in]    mode                オープンモード
    * @param[in]    isFile              ファイルかどうか
    * @param[in]    isDelete            ファイル削除かどうか
    *
    * @return       関数の処理結果を返します。
    *
    * @details      outDirectoryOpened は対象がディレクトリであり、
    *               かつリザルトが成功または ResultPermissionDenied である場合のみ有効です。
    */
    Result CheckAccessPolicy(
               int64_t* outId,
               bool* outDirectoryOpened,
               const Path& path,
               int mode,
               bool isFile,
               bool isDelete
           ) NN_NOEXCEPT;

    /**
    * @brief        このモードでアクセス可能かどうかを診断します。
    *
    * @param[out]   outId       ID
    * @param[in]    path        パス
    * @param[in]    mode        オープンモード
    * @param[in]    isFile      ファイルかどうか
    * @param[in]    isDelete    ファイル削除かどうか
    *
    * @return       関数の処理結果を返します。
    */
    Result CheckAccessPolicy(
               int64_t* outId,
               const Path& path,
               int mode,
               bool isFile,
               bool isDelete
           ) NN_NOEXCEPT
    {
        auto opened = false;
        return CheckAccessPolicy(outId, &opened, path, mode, isFile, isDelete);
    }

    /**
    * @brief        エントリが存在するかどうかをチェックします。
    *
    * @param[in]    path    パス
    *
    * @return       エントリが存在するかどうかを返します。
    */
    bool CheckEntryExistence(
             const Path& path
         ) NN_NOEXCEPT;

    /**
    * @brief        ID からエントリとディレクトリが親子関係にあるか調べる
    *
    * @param[out]   outIsSubEntry       checkEntryId が baseDirectoryId の子か否か
    * @param[in]    baseDirectoryId     ディレクトリの ID
    * @param[in]    checkEntryId        エントリの ID
    * @param[in]    isFile              ファイルであれば true、ディレクトリであれば false
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ID からエントリとディレクトリが親子関係にあるか調べる
    */
    virtual Result CheckSubEntry(
        bool* outIsSubEntry,
        StorageIndex baseDirectoryId,
        StorageIndex checkEntryId,
        bool isFile
    ) const NN_NOEXCEPT = 0;

    /**
    * @brief        開いているオブジェクトとしてリンクします。
    *
    * @param[in]    pObj    オブジェクト
    */
    void LinkObject(IExclusiveObject* pObj) NN_NOEXCEPT;

    /**
    * @brief 開いていたオブジェクトへのリンクを解除します。
    *
    * @param[in] pObj オブジェクト
    */
    void UnlinkObject(IExclusiveObject* pObj) NN_NOEXCEPT;

private:
    os::Mutex m_LockObject;
    IExclusiveObject *m_pOpened;     //! 開いているオブジェクトへのリンク
};

}}}

