﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_MemoryManagementPrivate.h>
#include <nn/result/result_HandlingUtility.h>


namespace nn { namespace fssystem { namespace save {

    class AutoMemoryStorage : public fs::IStorage
    {
    public:
        AutoMemoryStorage() NN_NOEXCEPT
        {
        }

        virtual ~AutoMemoryStorage() NN_NOEXCEPT NN_OVERRIDE
        {
            Close();
        }

        Result Create(size_t size) NN_NOEXCEPT
        {
            m_Buffer = nn::fs::detail::MakeUnique<char[]>(size);
            if( m_Buffer != nullptr )
            {
                m_Size = size;
                NN_RESULT_SUCCESS;
            }
            else
            {
                m_Size = 0;
                NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedInAutoMemoryStorageA());
            }
        }

        void Close() NN_NOEXCEPT
        {
            m_Buffer.reset();
        }

        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

            if( size == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            if( buffer == nullptr )
            {
                return nn::fs::ResultNullptrArgument();
            }
            if( offset < 0 || static_cast<int64_t>(m_Size) < offset )
            {
                return nn::fs::ResultInvalidOffset();
            }

            auto copySize = std::min(size, static_cast<size_t>(m_Size - offset));
            std::memcpy(buffer, m_Buffer.get() + offset, copySize);

            NN_RESULT_SUCCESS;
        }

        virtual Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

            if( size == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            if( buffer == nullptr )
            {
                return nn::fs::ResultNullptrArgument();
            }
            if( offset < 0 || static_cast<int64_t>(m_Size) < offset )
            {
                return nn::fs::ResultInvalidOffset();
            }

            auto copySize = std::min(size, static_cast<size_t>(m_Size - offset));
            std::memcpy(m_Buffer.get() + offset, buffer, copySize);

            NN_RESULT_SUCCESS;
        }

        virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = static_cast<int64_t>(m_Size);
            NN_RESULT_SUCCESS;
        }

        virtual Result OperateRange(
                           OperationId operationId,
                           int64_t offset,
                           int64_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(size);

            switch( operationId )
            {
            case OperationId::Invalidate:
                NN_RESULT_SUCCESS;
            default:
                return nn::fs::ResultUnsupportedOperation();
            }
        }

        void* GetBuffer() NN_NOEXCEPT
        {
            return m_Buffer.get();
        }

        const void* GetBuffer() const NN_NOEXCEPT
        {
            return m_Buffer.get();
        }

    private:
        std::unique_ptr<char[], nn::fs::detail::Deleter> m_Buffer;
        size_t m_Size;
    };

}}}

