﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFileSystem.h>

namespace nn { namespace fssystem {


    /**
    * @brief ファイルシステムの特定ディレクトリをルートとみなすファイルシステムです。
    */
    class SubdirectoryFileSystem : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
    {
        NN_DISALLOW_COPY(SubdirectoryFileSystem);

    public:
        static const int EntryNameLengthMax = 768;

        /**
        * @brief    コンストラクタです。
        *
        * @pre
        *   - pBaseFileSystem は有効なファイルシステムを指している
        *   - pRootPath は EntryNameLengthMax バイト以下の文字列を指している
        *   - pRootPath は '/' から始まる
        *
        * @details  pBaseFileSystem はインスタンスが破棄されるまで参照されます。
        */
        SubdirectoryFileSystem(nn::fs::fsa::IFileSystem* pBaseFileSystem, const char* pRootPath) NN_NOEXCEPT;
        SubdirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* pRootPath) NN_NOEXCEPT;
        SubdirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* pRootPath, bool isUncPreserved) NN_NOEXCEPT;
        virtual ~SubdirectoryFileSystem() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteFile(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCreateDirectory(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteDirectory(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCommitProvisionally(int64_t counter) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoRollback() NN_NOEXCEPT NN_OVERRIDE;

    private:
        Result Initialize(const char* pRootPath) NN_NOEXCEPT;

        nn::fs::fsa::IFileSystem* m_pBaseFileSystem;
        std::shared_ptr<nn::fs::fsa::IFileSystem> m_pSharedBaseFileSystem;
        char* m_RootPath;
        size_t m_RootPathBufferLength;
        bool m_IsUncPreserved;

    protected:
        Result ResolveFullPath(char* outPath, size_t outPathSize, const char* offsetPath) const NN_NOEXCEPT;
    };

}}
