﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/fs_ApplicationTemporaryFileSystem.h>
#include <nn/fssystem/fs_SaveDataFileSystem.h>
#include <nn/fssystem/fs_ISaveDataFileSystemCacheManager.h>

namespace nn { namespace fssystem {

template<typename SaveDataFileSystemClass>
class SaveDataFileSystemCacheRegisterBase : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
{
public:
    SaveDataFileSystemCacheRegisterBase(
        std::shared_ptr<SaveDataFileSystemClass>&& pFileSystem,
        ISaveDataFileSystemCacheManager* pManager) NN_NOEXCEPT
    : m_pFileSystem(std::move(pFileSystem))
    , m_pManager(pManager)
    {
    }

    virtual ~SaveDataFileSystemCacheRegisterBase() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pManager->Register(std::move(m_pFileSystem));
    }

public:
    SaveDataFileSystemClass * GetFileSystem() NN_NOEXCEPT
    {
        return m_pFileSystem.get();
    }

private:
    virtual Result DoOpenFile(
        std::unique_ptr<nn::fs::fsa::IFile>* pOutValue, const char* path, nn::fs::OpenMode mode
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->OpenFile(pOutValue, path, mode);
    }

    virtual Result DoOpenDirectory(
        std::unique_ptr<nn::fs::fsa::IDirectory>* pOutValue,
        const char* path,
        nn::fs::OpenDirectoryMode mode
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->OpenDirectory(pOutValue, path, mode);
    }

    virtual Result DoGetEntryType(
        nn::fs::DirectoryEntryType* pOutValue, const char* path
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->GetEntryType(pOutValue, path);
    }

    virtual Result DoCreateFile(
        const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->CreateFile(path, size, option);
    }

    virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->DeleteFile(path);
    }

    virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->CreateDirectory(path);
    }

    virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->DeleteDirectory(path);
    }

    virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->DeleteDirectoryRecursively(path);
    }

    virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->DoCleanDirectoryRecursively(path);
    }

    virtual Result DoRenameFile(
        const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->RenameFile(currentPath, newPath);
    }

    virtual Result DoRenameDirectory(
        const char* currentPath, const char* newPath
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->RenameDirectory(currentPath, newPath);
    }

    virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->Commit();
    }

    virtual Result DoCommitProvisionally(int64_t counter) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->DoCommitProvisionally(counter);
    }

    virtual Result DoRollback() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->DoRollback();
    }

    virtual Result DoGetFreeSpaceSize(
        int64_t* pOutValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->GetFreeSpaceSize(pOutValue, path);
    }

    virtual Result DoGetTotalSpaceSize(
        int64_t* pOutValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFileSystem->GetTotalSpaceSize(pOutValue, path);
    }

private:
    std::shared_ptr<SaveDataFileSystemClass> m_pFileSystem;
    ISaveDataFileSystemCacheManager* m_pManager;
};

typedef SaveDataFileSystemCacheRegisterBase<nn::fssystem::SaveDataFileSystem> SaveDataFileSystemCacheRegister;
typedef SaveDataFileSystemCacheRegisterBase<nn::fssystem::ApplicationTemporaryFileSystem> ApplicationTemporaryFileSystemCacheRegister;

}}
