﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace fssystem {


namespace detail {

    template <typename T>
    class SharedLock
    {
    public:
        explicit SharedLock(T& mutex)
            : m_pMutex(&mutex)
        {
            m_pMutex->lock_shared();
        }
        ~SharedLock()
        {
            m_pMutex->unlock_shared();
        }

    private:
        T* m_pMutex;
    };
}

    /**
    @brief 有効であれば下位ファイルシステムに処理を委譲するファイルシステム
    */
    class ProxyFileSystem : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
    {
    protected:

        class ProxyFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
        {
        public:
            nn::util::IntrusiveListNode m_ListNode;

        public:
            ProxyFile(ProxyFileSystem* pParent, std::unique_ptr<IFile>&& pBaseFile, Result resultForInvalidState) NN_NOEXCEPT
                : m_pParent(pParent),
                m_pBaseFile(std::move(pBaseFile)),
                m_ResultForInvalidState(resultForInvalidState)
            {
            }

            virtual ~ProxyFile() NN_NOEXCEPT
            {
                // TODO: 親が先に破棄されるケースのケア
                auto pParent = m_pParent;
                if( pParent != nullptr )
                {
                    pParent->Remove(this);
                }
            }

            void Invalidate()
            {
                std::lock_guard<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                m_pBaseFile.reset();
                m_pParent = nullptr;
            }

            virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseFile->Read(outValue, offset, buffer, size, option);
            }

            virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseFile->GetSize(outValue);
            }

            virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseFile->Write(offset, buffer, size, option);
            }
            virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseFile->Flush();
            }
            virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseFile->SetSize(size);
            }

            virtual Result DoOperateRange(
                void* outBuffer,
                size_t outBufferSize,
                fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseFile->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize);
            }

        private:
            ProxyFileSystem* m_pParent;
            std::unique_ptr<IFile> m_pBaseFile;
            Result m_ResultForInvalidState;
            os::ReaderWriterLock m_ValidityLock;

        private:
            Result CheckValidity()
            {
                if( m_pBaseFile == nullptr )
                {
                    return m_ResultForInvalidState;
                }
                else
                {
                    NN_RESULT_SUCCESS;
                }
            }

        };

        class ProxyDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
        {

        public:
            nn::util::IntrusiveListNode m_ListNode;

            void Invalidate()
            {
                std::lock_guard<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                m_pBaseDirectory.reset();
                m_pParent = nullptr;
            }

            ProxyDirectory(ProxyFileSystem* pParent, std::unique_ptr<IDirectory>&& pBaseDirectory, Result resultForInvalidState) NN_NOEXCEPT
                : m_pParent(pParent),
                m_pBaseDirectory(std::move(pBaseDirectory)),
                m_ResultForInvalidState(resultForInvalidState)
            {
            }

            virtual ~ProxyDirectory() NN_NOEXCEPT
            {
                std::lock_guard<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                if( m_pParent != nullptr )
                {
                    m_pParent->Remove(this);
                }
            }

            Result DoRead(int64_t *outValue, nn::fs::DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseDirectory->Read(outValue, entryBuffer, entryBufferCount);
            }

            Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
            {
                detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
                NN_RESULT_DO(CheckValidity());
                return m_pBaseDirectory->GetEntryCount(outValue);
            }

        protected:
            ProxyFileSystem* m_pParent;
            std::unique_ptr<IDirectory> m_pBaseDirectory;
            Result m_ResultForInvalidState;
            os::ReaderWriterLock m_ValidityLock;

        protected:
            Result CheckValidity()
            {
                if( m_pBaseDirectory == nullptr )
                {
                    return m_ResultForInvalidState;
                }
                else
                {
                    NN_RESULT_SUCCESS;
                }
            }

        };

    public:
        ProxyFileSystem(IFileSystem* pBaseFileSystem, Result resultForInvalidState) NN_NOEXCEPT
            : m_pBaseFileSystem(pBaseFileSystem),
            m_ResultForInvalidState(resultForInvalidState),
            m_ListMutex(false)
        {
        }
        virtual ~ProxyFileSystem() NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> scopedLock(m_ListMutex);
            for (auto& iter : m_OpenFileList)
            {
                iter.Invalidate();
            }
            for (auto& iter : m_OpenDirectoryList)
            {
                iter.Invalidate();
            }
        }

        void Remove(ProxyFile* pFile)
        {
            std::lock_guard<os::Mutex> scopedLock(m_ListMutex);
            if (m_OpenFileList.size() == 0)
            {
                // Invalidate に割り込まれたケースのため、何もしない
                // TODO: さらに OpenFile にも割り込まれたケースのケア
                return;
            }
            m_OpenFileList.erase(m_OpenFileList.iterator_to(*pFile));
        }
        void Remove(ProxyDirectory* pDirectory)
        {
            std::lock_guard<os::Mutex> scopedLock(m_ListMutex);
            if (m_OpenDirectoryList.size() == 0)
            {
                return;
            }
            m_OpenDirectoryList.erase(m_OpenDirectoryList.iterator_to(*pDirectory));
        }

        void Invalidate()
        {
            {
                std::lock_guard<os::Mutex> scopedLock(m_ListMutex);
                for (auto& iter : m_OpenFileList)
                {
                    iter.Invalidate();
                }
                m_OpenFileList.clear();

                for (auto& iter : m_OpenDirectoryList)
                {
                    iter.Invalidate();
                }
                m_OpenDirectoryList.clear();
            }

            std::lock_guard<os::ReaderWriterLock> scopedWriterLock(m_ValidityLock);
            m_pBaseFileSystem = nullptr;
        }

        virtual Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->CreateFile(path, size, option);
        }
        virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->DeleteFile(path);
        }
        virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->CreateDirectory(path);
        }
        virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->DeleteDirectory(path);
        }
        virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->DeleteDirectoryRecursively(path);
        }
        virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->CleanDirectoryRecursively(path);
        }
        virtual Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->RenameFile(currentPath, newPath);
        }
        virtual Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->RenameDirectory(currentPath, newPath);
        }
        virtual Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->GetEntryType(outValue, path);
        }
        virtual Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->GetFreeSpaceSize(outValue, path);
        }
        virtual Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->GetTotalSpaceSize(outValue, path);
        }
        virtual Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->GetFileTimeStampRaw(outTimeStamp, path);
        }
        virtual Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            std::unique_ptr<nn::fs::fsa::IFile> baseFile;
            NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&baseFile, path, mode));
            std::unique_ptr<ProxyFile> proxyFile(new ProxyFile(this, std::move(baseFile), m_ResultForInvalidState));
            NN_RESULT_THROW_UNLESS(proxyFile != nullptr, nn::fs::ResultAllocationMemoryFailedInProxyFileSystemA());
            {
                std::lock_guard<os::Mutex> scopedLock(m_ListMutex);
                m_OpenFileList.push_back(*proxyFile.get());
            }
            *outValue = std::move(proxyFile);
            NN_RESULT_SUCCESS;
        }
        virtual Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            std::unique_ptr<nn::fs::fsa::IDirectory> baseDirectory;
            NN_RESULT_DO(m_pBaseFileSystem->OpenDirectory(&baseDirectory, path, mode));
            std::unique_ptr<ProxyDirectory> proxyDirectory(new ProxyDirectory(this, std::move(baseDirectory), m_ResultForInvalidState));
            NN_RESULT_THROW_UNLESS(proxyDirectory != nullptr, nn::fs::ResultAllocationMemoryFailedInProxyFileSystemB());
            {
                std::lock_guard<os::Mutex> scopedLock(m_ListMutex);
                m_OpenDirectoryList.push_back(*proxyDirectory.get());
            }
            *outValue = std::move(proxyDirectory);
            NN_RESULT_SUCCESS;
        }
        virtual Result DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->QueryEntry(outBuffer, outBufferSize, inBuffer, inBufferSize, queryId, path);
        }
        virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->Commit();
        }
        virtual Result DoCommitProvisionally(int64_t counter) NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->CommitProvisionally(counter);
        }
        virtual Result DoRollback() NN_NOEXCEPT NN_OVERRIDE
        {
            detail::SharedLock<os::ReaderWriterLock> scopedReaderLock(m_ValidityLock);
            NN_RESULT_DO(CheckValidity());
            return m_pBaseFileSystem->Rollback();
        }

    protected:
        IFileSystem* m_pBaseFileSystem;
        Result m_ResultForInvalidState;

        typedef nn::util::IntrusiveList < ProxyFile,      nn::util::IntrusiveListMemberNodeTraits<ProxyFile, &ProxyFile::m_ListNode>> FileListType;
        typedef nn::util::IntrusiveList < ProxyDirectory, nn::util::IntrusiveListMemberNodeTraits<ProxyDirectory, &ProxyDirectory::m_ListNode>> DirectoryListType;
        FileListType m_OpenFileList;
        DirectoryListType m_OpenDirectoryList;
    private:
        os::ReaderWriterLock m_ValidityLock;
        os::Mutex            m_ListMutex;

    protected:
        Result CheckValidity()
        {
            if( m_pBaseFileSystem == nullptr )
            {
                return m_ResultForInvalidState;
            }
            else
            {
                NN_RESULT_SUCCESS;
            }
        }

    };


}}
