﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>
#include <nn/fssystem/fs_NcaHeader.h>

namespace nn { namespace fssystem {

    static const size_t IntegrityLayerCountRomFs = 7;
    static const size_t IntegrityHashLayerBlockSize = 1024 * 16;

}}

namespace nn { namespace fssystem {

/**
* @brief 完全性RomFsストレージクラスです。
*/
class IntegrityRomFsStorage : public nn::fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(IntegrityRomFsStorage);

public:
    /**
    * @brief        コンストラクタ
    */
    IntegrityRomFsStorage() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~IntegrityRomFsStorage() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        階層署名付きRomFsをマウントします。
    *
    * @param[in]    hashStorage             メタデータ、階層ハッシュ
    * @param[in]    bodyStorage             実データ
    * @param[in]    pBufferManager          キャッシュ用バッファマネージャー
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
                fs::SubStorage hashStorage,
                fs::SubStorage bodyStorage,
                IBufferManager* pBufferManager
            ) NN_NOEXCEPT;

    /**
    * @brief        階層署名付きRomFsをマウントします。
    *
    * @param[in]    infoLevelHash          メタデータ
    * @param[in]    masterHash             マスターハッシュ
    * @param[in]    storageInfo            階層ハッシュ、実データ
    * @param[in]    pBufferManager         キャッシュ用バッファマネージャー
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
               save::HierarchicalIntegrityVerificationInformation infoLevelHash,
               Hash masterHash,
               save::HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo,
               IBufferManager* pBufferManager
           ) NN_NOEXCEPT;

    /**
    * @brief        ストレージを閉じます。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        署名検証つきで実データ領域からデータを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      読み込もうとしたデータに対する署名が不正だった場合には、0 フィルされたデータが
    *               読込先バッファに書き込まれ、nn::fs::ResultFailedVerifySign を返します。
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.Read(offset, buffer, size);
    }

    /**
    * @brief        実データ領域にデータを書き込み、署名を更新します。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.Write(offset, buffer, size);
    }

    /**
    * @brief        実データの総バイトサイズを取得します。
    *
    * @param[out]   outValue    実データの総バイトサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.GetSize(outValue);
    }

    /**
    * @brief        サイズは変更できません。初期化時のみ指定できます。
    *
    * @return       常に nn::fs::ResultUnsupportedOperation
    */
    virtual Result SetSize(int64_t) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.Flush();
    }

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  pOutBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   pInBuffer        範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* pOutBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* pInBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IntegrityStorage.OperateRange(
            pOutBuffer, outBufferSize, operationId, offset, size, pInBuffer, inBufferSize);
    }

    /**
    * @brief        完全性検証ストレージへの変更をコミットします。
    *
    * @return       関数の処理結果を返します。
    */
    Result Commit() NN_NOEXCEPT
    {
        return m_IntegrityStorage.Commit();
    }

    /**
    * @brief        バッファセットを取得します。
    *
    * @return       バッファセットを返します。
    */
    save::FilesystemBufferManagerSet* GetBuffers() NN_NOEXCEPT
    {
        return m_IntegrityStorage.GetBuffers();
    }

private:
    save::HierarchicalIntegrityVerificationStorage m_IntegrityStorage;
    save::FilesystemBufferManagerSet m_Buffers;
    os::Mutex m_Locker;
    Hash m_MasterHash;
    std::unique_ptr<fs::MemoryStorage> m_pMasterHashStorage;

};

}}

