﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_IndirectStorage.h>

namespace nn { namespace fssystem {

template< bool IsContinuousCheckNeeded, typename TFunc >
Result IndirectStorage::OperatePerEntry(int64_t offset, int64_t size, TFunc func) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
    NN_SDK_REQUIRES_LESS_EQUAL(0, size);
    NN_SDK_REQUIRES(IsInitialized());

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(m_Table.IsInclude(offset, size), fs::ResultOutOfRange());

    // Find() で offset < 0 を弾くので、0 <= offset は保証される
    BucketTree::Visitor visitor;
    NN_RESULT_DO(m_Table.Find(&visitor, offset));
    {
        const auto entryOffset = visitor.Get<Entry>()->GetVirtualOffset();
        NN_RESULT_THROW_UNLESS(
            0 <= entryOffset && m_Table.IsInclude(entryOffset),
            fs::ResultInvalidIndirectEntryOffset()
        );
    }

    const auto endOffset = offset + static_cast<int64_t>(size);
    auto processOffset = offset;
    BucketTree::ContinuousReadingInfo continuousReading;

    while( processOffset < endOffset )
    {
        const auto entry = *visitor.Get<Entry>();

        // entry1Offset(processOffset) が不正
        const auto entry1Offset = entry.GetVirtualOffset();
        NN_RESULT_THROW_UNLESS(
            entry1Offset <= processOffset,
            fs::ResultInvalidIndirectEntryOffset()
        );

        // 対応するストレージがない
        NN_RESULT_THROW_UNLESS(
            0 <= entry.storageIndex && entry.storageIndex < StorageCount,
            fs::ResultInvalidIndirectEntryStorageIndex()
        );

        if( NN_STATIC_CONDITION(IsContinuousCheckNeeded) )
        {
            if( continuousReading.CheckNeedScan() )
            {
                NN_RESULT_DO(visitor.ScanContinuousReading<ContinuousReadingEntry>(
                    &continuousReading, processOffset, static_cast<size_t>(endOffset - processOffset)));
            }

            // 元データをまとめて読み込む
            if( continuousReading.CanDo() )
            {
                NN_RESULT_THROW_UNLESS(
                    entry.storageIndex == 0,
                    fs::ResultInvalidIndirectEntryStorageIndex()
                );

                const auto dataOffset = processOffset - entry1Offset;

                NN_RESULT_DO(func(
                    &m_DataStorage[0],
                    entry.GetPhysicalOffset() + dataOffset,
                    processOffset,
                    static_cast<int64_t>(continuousReading.GetReadSize())));

                continuousReading.Done();
            }
        }

        int64_t entry2Offset;
        if( visitor.CanMoveNext() )
        {
            NN_RESULT_DO(visitor.MoveNext());

            entry2Offset = visitor.Get<Entry>()->GetVirtualOffset();

            NN_RESULT_THROW_UNLESS(
                m_Table.IsInclude(entry2Offset),
                fs::ResultInvalidIndirectEntryOffset()
            );
        }
        else
        {
            entry2Offset = m_Table.GetEnd();
        }

        // entry2Offset が不正
        NN_RESULT_THROW_UNLESS(
            processOffset < entry2Offset, fs::ResultInvalidIndirectEntryOffset());
        // この時点で 0 <= entry1Offset <= readOffset < entry2Offset が保証される

        const auto dataOffset = processOffset - entry1Offset;
        const auto dataSize = (entry2Offset - entry1Offset) - dataOffset;
        NN_SDK_ASSERT_LESS(0, dataSize);

        const auto remainingSize = endOffset - processOffset; // <= size <= size_t(-1)
        // 32bit 環境で size_t(-1) < dataSize になっても問題ない（↑が保証されるので）
        const auto processSize = std::min(remainingSize, dataSize);
        NN_SDK_ASSERT_LESS_EQUAL(processSize, size);

        if( NN_STATIC_CONDITION(!IsContinuousCheckNeeded) || !continuousReading.IsDone() || entry.storageIndex != 0 )
        {
            NN_RESULT_DO(func(
                &m_DataStorage[entry.storageIndex],
                entry.GetPhysicalOffset() + dataOffset,
                processOffset,
                processSize));
        }

        processOffset += processSize;
    }
    NN_RESULT_SUCCESS;
}

}}
