﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/result/result_HandlingUtility.h>
#if !defined(NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK) && !defined(NN_COMPATIBLE_SDK) && defined(NN_DBM_CREATE_METADATA) && defined(_MSC_VER)
#include <nn/util/util_FormatString.h>
#endif

#include <nn/fssystem/fs_DbmRomTypes.h>
#include <nn/fssystem/fs_DbmHierarchicalRomFileTableTemplate.h>

namespace nn { namespace fssystem {

#ifdef NN_DBM_CREATE_METADATA

/*!
    @brief ディレクトリエントリーに必要なサイズを求めます。

            初回ファイルツリー作成(メタデータ構築)時に使用します。

    @param[in] entryNameLength エントリー名の長さ(終端のNULL文字を含まない)

    @return ディレクトリエントリーに必要なサイズ
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
uint32_t HierarchicalRomFileTableTemplate<
        DirectoryBucketStorage_,
        DirectoryEntryStorage_,
        FileBucketStorage_,
        FileEntryStorage_
    >::QueryDirectoryEntrySize(uint32_t entryNameLength)
{
    NN_ABORT_UNLESS(entryNameLength <= RomPathTool::MAX_PATH_LENGTH);
    return DirectoryEntryMapTable::QueryEntrySize(entryNameLength * sizeof(RomPathChar));
}

/*!
    @brief ファイルエントリーに必要なサイズを求めます。

            初回ファイルツリー作成(メタデータ構築)時に使用します。

    @param[in] entryNameLength エントリー名の長さ(終端のNULL文字を含まない)

    @return ファイルエントリーに必要なサイズ
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
uint32_t HierarchicalRomFileTableTemplate<
        DirectoryBucketStorage_,
        DirectoryEntryStorage_,
        FileBucketStorage_,
        FileEntryStorage_
    >::QueryFileEntrySize(uint32_t entryNameLength)
{
    NN_ABORT_UNLESS(entryNameLength <= RomPathTool::MAX_PATH_LENGTH);
    return FileEntryMapTable::QueryEntrySize(entryNameLength * sizeof(RomPathChar));
}

/*!
    :private

    @brief ディレクトリエントリーバケット用ストレージに必要なサイズを取得します。

    @param[in] countDirectoryBucket バケット数

    @return ディレクトリエントリーバケット用ストレージに必要なサイズ
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
uint32_t HierarchicalRomFileTableTemplate<
        DirectoryBucketStorage_,
        DirectoryEntryStorage_,
        FileBucketStorage_,
        FileEntryStorage_
    >::QueryDirectoryEntryBucketStorageSize(uint32_t countDirectoryBucket)
{
    return DirectoryEntryMapTable::QueryBucketStorageSize(countDirectoryBucket);
}

/*!
    :private

    @brief ファイルエントリーバケット用ストレージに必要なサイズを取得します。

    @param[in] countFileBucket バケット数

    @return ファイルエントリーバケット用ストレージに必要なサイズを取得します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
uint32_t HierarchicalRomFileTableTemplate<
        DirectoryBucketStorage_,
        DirectoryEntryStorage_,
        FileBucketStorage_,
        FileEntryStorage_
    >::QueryFileEntryBucketStorageSize(uint32_t countFileBucket)
{
    return FileEntryMapTable::QueryBucketStorageSize(countFileBucket);
}

/*!
    :private

    @brief ファイル管理テーブルをフォーマットします。

           IStorage を初期化します。使用する前に一度呼び出します。

    @param[in] pDirectoryBucket ディレクトリエントリーバケット用ストレージ
    @param[in] offsetDirectoryBucket ディレクトリエントリーバケット用ストレージに割り当てる領域のオフセット
    @param[in] sizeDirectoryBucket ディレクトリエントリーバケット用ストレージに割り当てる領域のサイズ(バイト数)
    @param[in] pDirectoryEntry ディレクトリエントリー用ストレージ
    @param[in] offsetDirectoryEntry ディレクトリエントリー用ストレージに割り当てる領域のオフセット
    @param[in] sizeDirectoryEntry ディレクトリエントリー用ストレージに割り当てる領域のサイズ(バイト数)
    @param[in] pFileBucket ファイルエントリーバケット用ストレージ
    @param[in] offsetFileBucket フイルエントリーバケット用ストレージに割り当てる領域のオフセット
    @param[in] sizeFileBucket ファイルエントリーバケット用ストレージに割り当てる領域のサイズ(バイト数)
    @param[in] pFileEntry ファイルエントリー用ストレージ
    @param[in] offsetFileEntry ファイルエントリー用ストレージに割り当てる領域のオフセット
    @param[in] sizeFileEntry ファイルエントリー用ストレージに割り当てる領域のサイズ(バイト数)

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::Format(
           DirectoryBucketStorage_* pDirectoryBucket,
           int64_t offsetDirectoryBucket,
           uint32_t sizeDirectoryBucket,
           DirectoryEntryStorage_* pDirectoryEntry,
           int64_t offsetDirectoryEntry,
           uint32_t sizeDirectoryEntry,
           FileBucketStorage_* pFileBucket,
           int64_t offsetFileBucket,
           uint32_t sizeFileBucket,
           FileEntryStorage_* pFileEntry,
           int64_t offsetFileEntry,
           uint32_t sizeFileEntry
       )
{
    // ディレクトリエントリー用テーブルをフォーマットします。
    NN_RESULT_DO(DirectoryEntryMapTable::Format(
                 pDirectoryBucket,
                 offsetDirectoryBucket,
                 FileEntryMapTable::QueryBucketCount(sizeDirectoryBucket),
                 pDirectoryEntry,
                 offsetDirectoryEntry,
                 sizeDirectoryEntry
             ));

    // ファイルエントリー用テーブルをフォーマットします。
    NN_RESULT_DO(FileEntryMapTable::Format(
                 pFileBucket,
                 offsetFileBucket,
                 FileEntryMapTable::QueryBucketCount(sizeFileBucket),
                 pFileEntry,
                 offsetFileEntry,
                 sizeFileEntry
             ));

    return ResultSuccess();
}
#endif // NN_DBM_CREATE_METADATA

/*!
    :private

    @brief コンストラクタです。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
HierarchicalRomFileTableTemplate<
    DirectoryBucketStorage_,
    DirectoryEntryStorage_,
    FileBucketStorage_,
    FileEntryStorage_
>::HierarchicalRomFileTableTemplate()
{
}

/*!
    :private

    @brief ストレージをマウントします。

    @param[in] pDirectoryBucket ディレクトリバケット用ストレージ
    @param[in] offsetDirectoryBucket ディレクトリバケット用ストレージに割り当てる領域のオフセット
    @param[in] sizeDirectoryBucket ディレクトリバケット用ストレージに割り当てる領域のサイズ(バイト数)
    @param[in] pDirectoryEntry ディレクトリエントリー用ストレージ
    @param[in] offsetDirectoryEntry ディレクトリエントリー用ストレージに割り当てる領域のオフセット
    @param[in] sizeDirectoryEntry ディレクトリエントリー用ストレージに割り当てる領域のサイズ(バイト数)
    @param[in] pFileBucket ファイルエントリーバケット用ストレージ
    @param[in] offsetFileBucket ファイルエントリーバケット用ストレージに割り当てる領域のオフセット
    @param[in] sizeFileBucket ファイルエントリーバケット用ストレージに割り当てる領域のサイズ(バイト数)
    @param[in] pFileEntry ファイルエントリー用ストレージ
    @param[in] offsetFileEntry ファイルエントリー用ストレージに割り当てる領域のオフセット
    @param[in] sizeFileEntry ファイルエントリー用ストレージに割り当てる領域のサイズ(バイト数)

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::Initialize(
           DirectoryBucketStorage* pDirectoryBucket,
           int64_t offsetDirectoryBucket,
           uint32_t sizeDirectoryBucket,
           DirectoryEntryStorage* pDirectoryEntry,
           int64_t offsetDirectoryEntry,
           uint32_t sizeDirectoryEntry,
           FileBucketStorage* pFileBucket,
           int64_t offsetFileBucket,
           uint32_t sizeFileBucket,
           FileEntryStorage* pFileEntry,
           int64_t offsetFileEntry,
           uint32_t sizeFileEntry
       )
{
    NN_SDK_ASSERT_NOT_NULL(pDirectoryBucket);
    NN_SDK_ASSERT_NOT_NULL(pDirectoryEntry);
    NN_SDK_ASSERT_NOT_NULL(pFileBucket);
    NN_SDK_ASSERT_NOT_NULL(pFileEntry);

    // ディレクトリ管理用ストレージをマウントします。
    NN_RESULT_DO(m_TableDirectory.Initialize(
                 pDirectoryBucket,
                 offsetDirectoryBucket,
                 FileEntryMapTable::QueryBucketCount(sizeDirectoryBucket),
                 pDirectoryEntry,
                 offsetDirectoryEntry,
                 sizeDirectoryEntry
             ));

    // ファイル管理用ストレージをマウントします。
    NN_RESULT_DO(m_TableFile.Initialize(
                 pFileBucket,
                 offsetFileBucket,
                 FileEntryMapTable::QueryBucketCount(sizeFileBucket),
                 pFileEntry,
                 offsetFileEntry,
                 sizeFileEntry
             ));

    return  ResultSuccess();
}

/*!
    :private

    @brief ストレージをアンマウントします。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
void HierarchicalRomFileTableTemplate<
         DirectoryBucketStorage_,
         DirectoryEntryStorage_,
         FileBucketStorage_,
         FileEntryStorage_
     >::Finalize()
{
    // ディレクトリ管理用ストレージをアンマウントします。
    m_TableDirectory.Finalize();

    // ファイル管理用ストレージをアンマウントします。
    m_TableFile.Finalize();
}

#ifdef NN_DBM_CREATE_METADATA
/*!
    :private

    @brief ルートディレクトリエントリーを作成します。

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::CreateRootDirectory()
{
    StoragePosition rootPosition;
    EntryKey rootKey;
    rootKey.key.parentDir = POSITION_ROOT_DIRECTORY;
    RomPathTool::InitEntryName(&rootKey.name);
    DirectoryEntry rootEntry;
    rootEntry.posNext = POSITION_NONE;
    rootEntry.posDirectory = POSITION_NONE;
    rootEntry.posFile = POSITION_NONE;
#ifdef ENABLE_DIRECTORY_METADATA
    std::memset(&rootEntry.info, 0, sizeof(DirectoryInfo));
#endif
    return m_TableDirectory.Add(&rootPosition, rootKey, rootEntry);
}
#endif // NN_DBM_CREATE_METADATA

/*!
    :private

    @brief 親の親ディレクトリを取得します。

    @param[out] pPosition 親の親ディレクトリエントリーのストレージ位置
    @param[out] pDirKey 親の親ディレクトリキー
    @param[out] pDirEntry 親の親ディレクトリエントリー
    @param[in] pos 親ディレクトリエントリーのストレージ位置
    @param[in] name 親ディレクトリ名
    @param[in] pFileName フルパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetGrandparent(
           StoragePosition* pPosition,
           EntryKey* pDirKey,
           DirectoryEntry* pDirEntry,
           StoragePosition pos,
           RomPathTool::RomEntryName name,
           const RomPathChar* pFullPath
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pPosition);
    NN_SDK_ASSERT_NOT_NULL(pDirKey);
    NN_SDK_ASSERT_NOT_NULL(pDirEntry);



    // 親の親ディレクトリ ID を取得します。
    RomEntryKey grandparentKey;
    DirectoryEntry grandparentEntry;
    NN_RESULT_DO(m_TableDirectory.GetByPosition(&grandparentKey, &grandparentEntry, NULL, NULL, pos));
    pDirKey->key.parentDir = grandparentKey.parentDir;

    // 親ディレクトリ名を取得します。
    NN_RESULT_DO(RomPathTool::GetParentDirectoryName(&pDirKey->name, name, pFullPath));

    // 親ディレクトリを取得します。
    NN_RESULT_DO(GetDirectoryEntry(pPosition, pDirEntry, *pDirKey));

    return ResultSuccess();
}

/*!
    :private

    @brief 指定した親ディレクトを取得します。

    @param[out] pParentPosition 親ディレクトリインデックス
    @param[out] pParentDirKey 親ディレクトリキー
    @param[out] pParentDirEntry 親ディレクトリエントリー
    @param[in] pParser パスパーサー

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindParentDirRecursive(
           StoragePosition* pParentPosition,
           EntryKey* pParentDirKey,
           DirectoryEntry* pParentDirEntry,
           RomPathTool::PathParser& parser,
           const RomPathChar* pFullPath
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pParentPosition);
    NN_SDK_ASSERT_NOT_NULL(pParentDirKey);
    NN_SDK_ASSERT_NOT_NULL(pParentDirEntry);

    StoragePosition dirPosition = POSITION_ROOT_DIRECTORY;
    EntryKey dirKey;
    DirectoryEntry dirEntry;

    // ルートディレクトリを取得します。
    dirKey.key.parentDir = POSITION_ROOT_DIRECTORY;
    NN_RESULT_DO(parser.GetNextDirectoryName(&dirKey.name));
    NN_RESULT_DO(GetDirectoryEntry(&dirPosition, &dirEntry, dirKey));

    StoragePosition parentPosition = dirPosition;
    while (! parser.IsParseFinished())
    {
        EntryKey oldKey = dirKey;

        // 次のディレクトリ名を取得します。
        NN_RESULT_DO(parser.GetNextDirectoryName(&dirKey.name));

        if (RomPathTool::IsCurrentDirectory(dirKey.name))
        {
            // "."の場合何もしない。
            dirKey = oldKey;
            continue;
        }
        else if (RomPathTool::IsParentDirectory(dirKey.name))
        {
            if (parentPosition == POSITION_ROOT_DIRECTORY)
            {
                // ルートディレクトリの親ディレクトリは取得できません。
                return ResultDbmInvalidOperation();
            }

            // ".."の場合は親ディレクトリを取得します。
            NN_RESULT_DO(GetGrandparent(
                         &parentPosition,
                         &dirKey,
                         &dirEntry,
                         dirKey.key.parentDir,
                         dirKey.name,
                         pFullPath
                     ));
        }
        else
        {
            // 親ディレクトリ ID + ディレクトリ名から、ディレクトリエントリーを取得します。
            dirKey.key.parentDir = parentPosition;
            NN_RESULT_DO(GetDirectoryEntry(&dirPosition, &dirEntry, dirKey));

            parentPosition = dirPosition;
        }
    }

    // 親ディレクトリ情報を設定します。
    *pParentPosition = parentPosition;
    *pParentDirKey = dirKey;
    *pParentDirEntry = dirEntry;

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したパスからディレクトまたはファイルを取得します。

    @param[out] pParentDirKey 親ディレクトリキー
    @param[out] pParentDirEntry 親ディレクトリエントリー
    @param[out] pKey エントリーキー
    @param[in] bFindDir ディレクトリエントリーかどうか
    @param[in] pFullPath 分離するフルパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindPathRecursive(
           EntryKey* pParentDirKey,
           DirectoryEntry* pParentDirEntry,
           EntryKey* pKey,
           bool bFindDir,
           const RomPathChar* pFullPath
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pParentDirKey);
    NN_SDK_ASSERT_NOT_NULL(pParentDirEntry);
    NN_SDK_ASSERT_NOT_NULL(pKey);
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // フルパスを解析するためのパーサを初期化します。
    RomPathTool::PathParser parser;
    NN_RESULT_DO(parser.Initialize(pFullPath));

    // 親ディレクトリを探索します。
    StoragePosition parentPosition = 0;
    NN_RESULT_DO(FindParentDirRecursive(
                 &parentPosition,
                 pParentDirKey,
                 pParentDirEntry,
                 parser,
                 pFullPath
             ));

    // 最後のパス部分を取得します。
    if (bFindDir)
    {
        RomPathTool::RomEntryName name;

        // パスの最後の部分をディレクトリ名として取得します。
        NN_RESULT_DO(parser.GetAsDirectoryName(&name));

        if (RomPathTool::IsCurrentDirectory(name))
        {
            // 最後のパスが"."であった場合

            // 親ディレクトリとして取得したキーをカレントディレクトリとします。
            *pKey = *pParentDirKey;

            // 親ディレクトリを更新します。
            if (pKey->key.parentDir != POSITION_ROOT_DIRECTORY)
            {
                StoragePosition pos;
                NN_RESULT_DO(GetGrandparent(
                             &pos,
                             pParentDirKey,
                             pParentDirEntry,
                             pKey->key.parentDir,
                             pKey->name,
                             pFullPath
                         ));
            }
        }
        else if (RomPathTool::IsParentDirectory(name))
        {
            // 最後のパスが".."であった場合は、カレントディレクトリ情報を
            // 取得しつつ親ディレクトリ方法を更新します。

            if (parentPosition == POSITION_ROOT_DIRECTORY)
            {
                // ルートディレクトリの親ディレクトリは取得できません。
                return ResultDbmInvalidOperation();

            }

            StoragePosition pos;

            // 親の親ディレクトリ ID を取得します。
            DirectoryEntry currEntry;
            NN_RESULT_DO(GetGrandparent(
                         &pos,
                         pKey,
                         &currEntry,
                         pParentDirKey->key.parentDir,
                         pParentDirKey->name,
                         pFullPath
                     ));

            // 親ディレクトリを更新します。
            if (pKey->key.parentDir != POSITION_ROOT_DIRECTORY)
            {
                NN_RESULT_DO(GetGrandparent(
                             &pos,
                             pParentDirKey,
                             pParentDirEntry,
                             pKey->key.parentDir,
                             pKey->name,
                             pFullPath
                         ));
            }
        }
        else
        {
            pKey->name = name;

            // 親ディレクトリ ID を設定します。
            // (pKey->name.length == 0 であればルートディレクトリを示しています)
            pKey->key.parentDir = (pKey->name.length > 0) ? parentPosition : POSITION_ROOT_DIRECTORY;
        }
    }
    else
    {
        // パスがディレクトリを示していたかどうかチェックします。
        if (parser.IsDirectoryPath())
        {
            // 不正な操作です。
            return ResultDbmInvalidOperation();
        }

        // パスの最後の部分を取得します。
        pKey->key.parentDir = parentPosition;
        NN_RESULT_DO(parser.GetAsFileName(&pKey->name));
    }

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したパスから親ディレクトリとディレクトリキーを取得します。

    @param[out] pParentDirKey 親ディレクトリキー
    @param[out] pParentDirEntry 親ディレクトリエントリー
    @param[out] pDirKey ディレクトリキー
    @param[in] pFullPath 分離するフルパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindDirectoryRecursive(
           EntryKey* pParentDirKey,
           DirectoryEntry* pParentDirEntry,
           EntryKey* pDirKey,
           const RomPathChar* pFullPath
       ) const
{
    return FindPathRecursive(
               pParentDirKey,
               pParentDirEntry,
               pDirKey,
               true,
               pFullPath
           );
}

/*!
    :private

    @brief 指定したパスから親ディレクトリとファイルキーを取得します。

    @param[out] pParentDirKey 親ディレクトリキー
    @param[out] pParentDirEntry 親ディレクトリエントリー
    @param[out] pFileKey ファイルキー
    @param[in] pFullPath 分離するフルパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindFileRecursive(
           EntryKey* pParentDirKey,
           DirectoryEntry* pParentDirEntry,
           EntryKey* pFileKey,
           const RomPathChar* pFullPath
       ) const
{
    return FindPathRecursive(
               pParentDirKey,
               pParentDirEntry,
               pFileKey,
               false,
               pFullPath
           );
}

#ifdef NN_DBM_CREATE_METADATA
/*!
    :private

    @brief 指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。

    @param[in] key キー
    @param[in] resultIfExist 同名のエントリーが存在したときの結果

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::CheckSameEntryExists(
         const EntryKey& key,
         Result resultIfExist
     ) const
{
    // 同名ディレクトリの存在をチェックします。
    {
        StoragePosition pos;
        DirectoryEntry dirEntry;
        Result result = m_TableDirectory.Get(&pos, &dirEntry, key);
        if (! ResultDbmKeyNotFound::Includes(result))
        {
            if (result.IsFailure())
            {
                // ストレージによるエラーです。
                return result;
            }
            else
            {
                // 同名のエントリーが存在します。
                return resultIfExist;
            }
        }
    }

    // 同名ファイルの存在をチェックします。
    {
        StoragePosition pos;
        FileEntry fileEntry;
        Result result = m_TableFile.Get(&pos, &fileEntry, key);
        if (! ResultDbmKeyNotFound::Includes(result))
        {
            if (result.IsFailure())
            {
                // ストレージによるエラーです。
                return result;
            }
            else
            {
                // 同名のエントリーが存在します。
                return resultIfExist;
            }
        }
    }

    // 同名のエントリーは存在しません。
    return ResultSuccess();
}
#endif // NN_DBM_CREATE_METADATA

/*!
    :private

    @brief ディレクトリエントリーを取得します。

    @param[out] pPosition ディレクトリエントリーのストレージ位置
    @param[out] pEntry ディレクトリエントリー
    @param[in] key ディレクトリキー

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetDirectoryEntry(
           StoragePosition* pPosition,
           DirectoryEntry* pEntry,
           const EntryKey& key
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pPosition);
    NN_SDK_ASSERT_NOT_NULL(pEntry);

    Result result = m_TableDirectory.Get(pPosition, pEntry, key);
    if (result.IsFailure())
    {
        if (ResultDbmKeyNotFound::Includes(result))
        {
            // ファイルをディレクトリとして使用していないかどうかチェックします。
            StoragePosition filePosition;
            FileEntry fileEntry;
            result = m_TableFile.Get(&filePosition, &fileEntry, key);
            if (result.IsFailure())
            {
                if (! ResultDbmKeyNotFound::Includes(result))
                {
                    return result;
                }
                // ディレクトリが見つかりませんでした。
                return ResultDbmDirectoryNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

/*!
    :private

    @brief ディレクトリエントリーを取得します。

    @param[out] pEntry ディレクトリエントリー
    @param[in] dirId ディレクトリ ID

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetDirectoryEntry(
           DirectoryEntry* pEntry,
           RomDirectoryId dirId
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pEntry);

    StoragePosition pos = DirectoryIdToPosition(dirId);

    RomEntryKey key;
    Result result = m_TableDirectory.GetByPosition(&key, pEntry, NULL, NULL, pos);
    if (result.IsFailure())
    {
        if (ResultDbmKeyNotFound::Includes(result))
        {
            // ファイルをディレクトリとして使用していないかどうかチェックします。
            FileEntry fileEntry;
            result = m_TableFile.GetByPosition(&key, &fileEntry, NULL, NULL, pos);
            if (result.IsFailure())
            {
                if (! ResultDbmKeyNotFound::Includes(result))
                {
                    return result;
                }
                // ディレクトリが見つかりませんでした。
                return ResultDbmDirectoryNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

/*!
    :private

    @brief ファイルエントリーを取得します。

    @param[out] pPosition ファイルエントリーのストレージ位置
    @param[out] pEntry ファイルエントリー
    @param[in] key ファイルキー

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetFileEntry(
           StoragePosition* pPosition,
           FileEntry* pEntry,
           const EntryKey& key
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pPosition);
    NN_SDK_ASSERT_NOT_NULL(pEntry);

    Result result = m_TableFile.Get(pPosition, pEntry, key);
    if (result.IsFailure())
    {
        if (ResultDbmKeyNotFound::Includes(result))
        {
            // ディレクトリをファイルとして使用していないかどうかチェックします。
            StoragePosition dirPosition;
            DirectoryEntry dirEntry;
            result = m_TableDirectory.Get(&dirPosition, &dirEntry, key);
            if (result.IsFailure())
            {
                if (! ResultDbmKeyNotFound::Includes(result))
                {
                    return result;
                }
                // ファイルが見つかりませんでした。
                return ResultDbmFileNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

/*!
    :private

    @brief ファイルエントリーを取得します。

    @param[out] pEntry ファイルエントリー
    @param[in] fileId ファイル ID

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetFileEntry(
           FileEntry* pEntry,
           RomFileId fileId
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pEntry);

    StoragePosition pos = FileIdToPosition(fileId);

    RomEntryKey key;
    Result result = m_TableFile.GetByPosition(&key, pEntry, NULL, NULL, pos);
    if (result.IsFailure())
    {
        if (ResultDbmKeyNotFound::Includes(result))
        {
            // ディレクトリをファイルとして使用していないかどうかチェックします。
            DirectoryEntry dirEntry;
            result = m_TableDirectory.GetByPosition(&key, &dirEntry, NULL, NULL, pos);
            if (result.IsFailure())
            {
                if (! ResultDbmKeyNotFound::Includes(result))
                {
                    return result;
                }
                // ファイルが見つかりませんでした。
                return ResultDbmFileNotFound();
            }
            else
            {
                // 不正な操作です。
                return ResultDbmInvalidOperation();
            }
        }
    }
    return result;
}

#ifdef NN_DBM_CREATE_METADATA
/*!
    :private

    @brief ディレクトリを作成します。

    @param[out] pDirId ディレクトリ ID
    @param[in] pFullPath ディレクトリパス
    @param[in] dirInfo ディレクトリ情報

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::CreateDirectory(RomDirectoryId* pDirId, const RomPathChar* pFullPath, const DirectoryInfo& dirInfo)
{
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、新規ディレクトリ用のキーを生成します。
    EntryKey parentKey;
    DirectoryEntry parentEntry;
    EntryKey newKey;
    NN_RESULT_DO(FindDirectoryRecursive(
                 &parentKey,
                 &parentEntry,
                 &newKey,
                 pFullPath
             ));

    // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
    // 存在した場合は、すでにエントリーが存在していることを結果として返します。
    NN_RESULT_DO(CheckSameEntryExists(
                 newKey,
                 ResultDbmAlreadyExists()
             ));

    // 新規ディレクトリエントリーを初期化します。
    DirectoryEntry newEntry;
    newEntry.posNext = POSITION_NONE;
    newEntry.posDirectory = POSITION_NONE;
    newEntry.posFile = POSITION_NONE;
#ifdef ENABLE_DIRECTORY_METADATA
    newEntry.info = dirInfo;
#else
    NN_UNUSED(dirInfo);
#endif

    // 新規ディレクトリエントリーをハッシュテーブルに登録します。
    StoragePosition newPosition;
    Result result = m_TableDirectory.Add(&newPosition, newKey, newEntry);
    if (result.IsFailure())
    {
        if (ResultDbmKeyFull::Includes(result))
        {
            return ResultDbmDirectoryEntryFull();
        }
        return result;
    }

    // キーバリューストレージ内でのアドレスをディレクトリIDとして返します。
    *pDirId = PositionToDirectoryId(newPosition);

    // 新規ディレクトリエントリーを親ディレクトリにおける子ディレクトリリストの末尾として追加します。
    if (parentEntry.posDirectory == POSITION_NONE)
    {
        parentEntry.posDirectory = newPosition;

        // 親ディレクトリのリンク情報を更新します。
        NN_RESULT_DO(m_TableDirectory.SetByPosition(newKey.key.parentDir, parentEntry, nn::fs::WriteOption()));
    }
    else
    {
        StoragePosition posTail = parentEntry.posDirectory;
        while (NN_STATIC_CONDITION(true))
        {
            RomEntryKey currKey;
            DirectoryEntry currEntry;
            NN_RESULT_DO(m_TableDirectory.GetByPosition(&currKey, &currEntry, NULL, NULL, posTail));
            if (currEntry.posNext == POSITION_NONE)
            {
                currEntry.posNext = newPosition;

                // リンク情報を更新します。
                NN_RESULT_DO(m_TableDirectory.SetByPosition(posTail, currEntry, nn::fs::WriteOption()));

                break;
            }

            posTail = currEntry.posNext;
        }
    }

    return ResultSuccess();
}

/*!
    :private

    @brief ファイルを作成します。

    @param[out] pFileId ファイル ID
    @param[in] pFullPath ファイルパス
    @param[in] fileInfo ファイル情報

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::CreateFile(RomFileId* pFileId, const RomPathChar* pFullPath, const FileInfo& fileInfo)
{
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、新規ファイル用のキーを生成します。
    EntryKey parentKey;
    DirectoryEntry parentEntry;
    EntryKey newKey;
    NN_RESULT_DO(FindFileRecursive(
                 &parentKey,
                 &parentEntry,
                 &newKey,
                 pFullPath
             ));

    // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
    // 存在した場合は、すでにエントリーが存在していることを結果として返します。
    NN_RESULT_DO(CheckSameEntryExists(
                 newKey,
                 ResultDbmAlreadyExists()
             ));

    // 新規ファイルエントリーを初期化します。
    FileEntry newEntry;
    newEntry.posNext = POSITION_NONE;
    newEntry.info = fileInfo;

    // 新規ファイルエントリーをハッシュテーブルに登録します。
    StoragePosition newPosition = POSITION_NONE;
    Result result = m_TableFile.Add(&newPosition, newKey, newEntry);
    if (result.IsFailure())
    {
        if (ResultDbmKeyFull::Includes(result))
        {
            return ResultDbmFileEntryFull();
        }
        return result;
    }

    // キーバリューストレージ内でのアドレスをファイルIDとして返します。
    *pFileId = PositionToFileId(newPosition);

    // 新規ファイルエントリーを親ディレクトリにおける子ファイルリストの末尾として追加します。
    if (parentEntry.posFile == POSITION_NONE)
    {
        parentEntry.posFile = newPosition;

        // 親ディレクトリのリンク情報を更新します。
        NN_RESULT_DO(m_TableDirectory.SetByPosition(newKey.key.parentDir, parentEntry, nn::fs::WriteOption()));
    }
    else
    {
        StoragePosition posTail = parentEntry.posFile;
        while (NN_STATIC_CONDITION(true))
        {
            RomEntryKey currKey;
            FileEntry currEntry;
            NN_RESULT_DO(m_TableFile.GetByPosition(&currKey, &currEntry, NULL, NULL, posTail));
            if (currEntry.posNext == POSITION_NONE)
            {
                currEntry.posNext = newPosition;

                // リンク情報を更新します。
                NN_RESULT_DO(m_TableFile.SetByPosition(posTail, currEntry, nn::fs::WriteOption()));

                break;
            }

            posTail = currEntry.posNext;
        }
    }

    return ResultSuccess();
}
#endif // NN_DBM_CREATE_METADATA

/*!
    :private

    @brief 指定したフルパスをディレクトリ ID に変換します。

    @param[out] pDirId ディレクトリ ID
    @param[in] pFullPath ディレクトリパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::ConvertPathToDirectoryId(
           RomDirectoryId* pDirId,
           const RomPathChar* pFullPath
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pDirId);
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    EntryKey parentKey;
    DirectoryEntry parentEntry;
    EntryKey key;
    NN_RESULT_DO(FindDirectoryRecursive(
                 &parentKey,
                 &parentEntry,
                 &key,
                 pFullPath
             ));

    // ディレクトリエントリーを取得します。
    StoragePosition pos;
    DirectoryEntry entry;
    NN_RESULT_DO(GetDirectoryEntry(&pos, &entry, key));

    *pDirId = PositionToDirectoryId(pos);

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したフルパスをファイル ID に変換します。

    @param[out] pFileId ファイル ID
    @param[in] pFullPath ファイルパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::ConvertPathToFileId(
           RomFileId* pFileId,
           const RomPathChar* pFullPath
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pFileId);
    NN_SDK_ASSERT_NOT_NULL(pFullPath);


    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    EntryKey parentKey;
    DirectoryEntry parentEntry;
    EntryKey key;
    NN_RESULT_DO(FindDirectoryRecursive(
                 &parentKey,
                 &parentEntry,
                 &key,
                 pFullPath
             ));

    // ファイルエントリーを取得します。
    StoragePosition pos;
    FileEntry entry;
    NN_RESULT_DO(GetFileEntry(&pos, &entry, key));

    *pFileId = PositionToFileId(pos);

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したディレクトリ情報を取得します。

    @param[out] pDirInfo ディレクトリ情報
    @param[in] dirKey ディレクトリキー

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetDirectoryInformation(
           DirectoryInfo* pDirInfo,
           const EntryKey& dirKey
       ) const
{
    // ディレクトリエントリーを取得します。
    StoragePosition pos;
    DirectoryEntry entry;
    NN_RESULT_DO(GetDirectoryEntry(&pos, &entry, dirKey));

#ifdef ENABLE_DIRECTORY_METADATA
    // ディレクトリの付加情報をコピーします。
    *pDirInfo = entry.info;
#else
    NN_UNUSED(pDirInfo);
#endif

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したディレクトリ情報を取得します。

    @param[out] pDirInfo ディレクトリ情報
    @param[in] dirId ディレクトリ ID

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetDirectoryInformation(
           DirectoryInfo* pDirInfo,
           RomDirectoryId dirId
       ) const
{
    // ディレクトリエントリーを取得します。
    DirectoryEntry entry;
    NN_RESULT_DO(GetDirectoryEntry(&entry, dirId));

#ifdef ENABLE_DIRECTORY_METADATA
    // ディレクトリの付加情報をコピーします。
    *pDirInfo = entry.info;
#else
    NN_UNUSED(pDirInfo);
#endif

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したディレクトリ情報を取得します。

    @param[out] pDirInfo ディレクトリ情報
    @param[in] pFullPath ディレクトリパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::GetDirectoryInformation(
           DirectoryInfo* pDirInfo,
           const RomPathChar* pFullPath
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pDirInfo);
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    EntryKey parentKey;
    DirectoryEntry parentEntry;
    EntryKey key;
    NN_RESULT_DO(FindDirectoryRecursive(
                 &parentKey,
                 &parentEntry,
                 &key,
                 pFullPath
             ));

    return GetDirectoryInformation(pDirInfo, key);
}

/*!
    :private

    @brief ファイルを開きます。

    @param[out] pFileInfo ファイル情報
    @param[in] fileKey ファイルキー

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::OpenFile(FileInfo* pFileInfo, const EntryKey& fileKey) const
{
    NN_SDK_ASSERT_NOT_NULL(pFileInfo);

    // ファイルエントリーを取得します。
    FileEntry fileEntry;
    StoragePosition pos;
    NN_RESULT_DO(GetFileEntry(&pos, &fileEntry, fileKey));

    // ファイルの付加情報をコピーします。
    *pFileInfo = fileEntry.info;

    return ResultSuccess();
}

/*!
    :private

    @brief ファイルを開きます。

    @param[out] pFileInfo ファイル情報
    @param[in] fileId ファイル ID

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::OpenFile(FileInfo* pFileInfo, RomFileId fileId) const
{
    NN_SDK_ASSERT_NOT_NULL(pFileInfo);

    // ファイルエントリーを取得します。
    FileEntry fileEntry;
    NN_RESULT_DO(GetFileEntry(&fileEntry, fileId));

    // ファイルの付加情報をコピーします。
    *pFileInfo = fileEntry.info;

    return ResultSuccess();
}

/*!
    :private

    @brief ファイルを開きます。

    @param[out] pFileInfo ファイル情報
    @param[in] pFullPath ファイルパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::OpenFile(FileInfo* pFileInfo, const RomPathChar* pFullPath) const
{
    NN_SDK_ASSERT_NOT_NULL(pFileInfo);
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ファイル用キ－を生成します。
    EntryKey parentKey;
    DirectoryEntry parentEntry;
    EntryKey fileKey;
    NN_RESULT_DO(FindFileRecursive(
                 &parentKey,
                 &parentEntry,
                 &fileKey,
                 pFullPath
             ));

    return OpenFile(pFileInfo, fileKey);
}

/*!
    :private

    @brief 指定したディレクトリ以下のディレクトリとファイル列挙を開始します。

           イテレータを @ref FindNextDirectory、@ref FindNextFile に
           渡すことでディレクトリ名、ファイル名をイテレーションできます。
           また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

    @param[in,out] pfi イテレータ
    @param[in] dirKey ディレクトリキー

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindOpen(FindPosition* pfi, const EntryKey& dirKey) const
{
    NN_SDK_ASSERT_NOT_NULL(pfi);

    pfi->nextPositionDirectory = POSITION_NONE;
    pfi->nextPositionFile = POSITION_NONE;

    // イテレーション対象のディレクトリエントリーを取得します。
    StoragePosition pos;
    DirectoryEntry dirEntry;
    NN_RESULT_DO(GetDirectoryEntry(&pos, &dirEntry, dirKey));

    // 子ディレクトリ、ファイルリストへのインデックスを取得します。
    pfi->nextPositionDirectory = dirEntry.posDirectory;
    pfi->nextPositionFile = dirEntry.posFile;

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したディレクトリ以下のディレクトリとファイル列挙を開始します。

           イテレータを @ref FindNextDirectory、@ref FindNextFile に
           渡すことでディレクトリ名、ファイル名をイテレーションできます。
           また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

    @param[in,out] pfi イテレータ
    @param[in] dirId ディレクトリ ID

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindOpen(FindPosition* pfi, RomDirectoryId dirId) const
{
    NN_SDK_ASSERT_NOT_NULL(pfi);

    pfi->nextPositionDirectory = POSITION_NONE;
    pfi->nextPositionFile = POSITION_NONE;

    // イテレーション対象のディレクトリエントリーを取得します。
    DirectoryEntry dirEntry;
    NN_RESULT_DO(GetDirectoryEntry(&dirEntry, dirId));

    // 子ディレクトリ、ファイルリストへのインデックスを取得します。
    pfi->nextPositionDirectory = dirEntry.posDirectory;
    pfi->nextPositionFile = dirEntry.posFile;

    return ResultSuccess();
}

/*!
    :private

    @brief 指定したディレクトリ以下のディレクトリとファイル列挙を開始します。

           イテレータを @ref FindNextDirectory、@ref FindNextFile に
           渡すことでディレクトリ名、ファイル名をイテレーションできます。
           また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

    @param[in,out] pfi イテレータ
    @param[in] pFullPath ディレクトリパス

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindOpen(FindPosition* pfi, const RomPathChar* pFullPath) const
{
    NN_SDK_ASSERT_NOT_NULL(pfi);
    NN_SDK_ASSERT_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、
    // イテレーション対象ディレクトリ用キ－を生成します。
    EntryKey parentKey;
    DirectoryEntry parentEntry;
    EntryKey dirKey;
    NN_RESULT_DO(FindDirectoryRecursive(
                 &parentKey,
                 &parentEntry,
                 &dirKey,
                 pFullPath
             ));

    return FindOpen(pfi, dirKey);
}

/*!
    :private

    @brief ディレクトリ以下の次のディレクトリを取得します。

           この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

    @param[out] pDirectoryName ディレクトリ名を格納するバッファ
                               サイズ RomPathTool::MAX_PATH_LENGTH + 1のバッファを渡してください。
    @param[in,out] pfi イテレータ

    @return 関数の処理結果を返します。
            イテレーション完了時には ResultDbmFindFinished を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindNextDirectory(
           RomPathChar* pDirectoryName,
           FindPosition* pfi
       ) const
{
    NN_SDK_ASSERT_NOT_NULL(pfi);
    NN_SDK_ASSERT_NOT_NULL(pDirectoryName);

    // 子ディレクトリリストの終端チェックを行います。
    if (pfi->nextPositionDirectory == POSITION_NONE)
    {
        // 探索は終了しました
        return ResultDbmFindFinished();
    }

    // 現在のディレクトリエントリーを取得します。
    RomEntryKey key;
    DirectoryEntry entry;
    size_t length = 0;
    NN_RESULT_DO(m_TableDirectory.GetByPosition(&key, &entry, pDirectoryName, &length, pfi->nextPositionDirectory));
    pDirectoryName[length / sizeof(RomPathChar)] = StringTraitsRom::NUL;

    // 次のディレクトリへのインデックスに更新します。
    pfi->nextPositionDirectory = entry.posNext;

    return ResultSuccess();
}

/*!
    :private

    @brief ディレクトリ以下の次のファイルを取得します。

           この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

    @param[out] pFileName ベースファイル名
                          サイズ RomPathTool::MAX_PATH_LENGTH + 1 のバッファを渡してください。
    @param[in,out] pfi イテレータ

    @return 関数の処理結果を返します。
            イテレーション完了時には ResultDbmFindFinished を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::FindNextFile(RomPathChar* pFileName, FindPosition* pfi) const
{
    NN_SDK_ASSERT_NOT_NULL(pfi);
    NN_SDK_ASSERT_NOT_NULL(pFileName);

    // 子ファイルリストの終端チェックを行います。
    if (pfi->nextPositionFile == POSITION_NONE)
    {
        // 探索は終了しました
        return ResultDbmFindFinished();
    }

    // 現在のファイルエントリーを取得します。
    RomEntryKey key;
    FileEntry entry;
    size_t length = 0;
    NN_RESULT_DO(m_TableFile.GetByPosition(&key, &entry, pFileName, &length, pfi->nextPositionFile));
    pFileName[length / sizeof(RomPathChar)] = StringTraitsRom::NUL;

    // 次のファイルへのインデックスに更新します。
    pfi->nextPositionFile = entry.posNext;

    return ResultSuccess();
}

#if defined(NN_DBM_CREATE_METADATA)
/*!
    :private

    @brief ROM ファイルシステムに必要なバッファサイズを取得します。

    @param[out] pSize ROM ファイルシステムに必要なサイズ

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::QueryRomFileSystemSize(
           uint32_t* pSizeDirectoryEntry,
           uint32_t* pSizeFileEntry
       )
{
    NN_SDK_ASSERT_NOT_NULL(pSizeDirectoryEntry);
    NN_SDK_ASSERT_NOT_NULL(pSizeFileEntry);

    *pSizeDirectoryEntry = m_TableDirectory.GetTotalEntrySize();
    *pSizeFileEntry = m_TableFile.GetTotalEntrySize();

    return ResultSuccess();
}
#endif // NN_DBM_CREATE_METADATA

#if !defined(NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK) && !defined(NN_COMPATIBLE_SDK) && defined(NN_DBM_CREATE_METADATA) && defined(_MSC_VER)
/*!
    :private

    @brief 指定したディレクトリ以下のファイルツリーをデバッグ表示します。

    @param[in] pRootKey ディレクトリキー
    @param[in] pRootEntry ディレクトリエントリー
    @param[in] depth ファイル階層の深さ

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::DumpDirectory(StoragePosition posParent, int depth)
{
    RomPathChar readBuf[RomPathTool::MAX_PATH_LENGTH + 1];
    DirectoryEntry dirEntry;
    RomPathChar buf[1024];

    {
        size_t length = 0;

        EntryKey dirKey;
        NN_RESULT_DO(m_TableDirectory.GetByPosition(&dirKey.key, &dirEntry, readBuf, &length, posParent));
        readBuf[length / sizeof(RomPathChar)] = StringTraitsRom::NUL;

        for (int i = 0; i < depth * 2; i++)
        {
            RomPathTool::Print(" ");
        }

        if (length > 0)
        {
            nn::util::SNPrintf(buf, sizeof(buf) / sizeof(RomPathChar), "+ / %s [%08X]\n", readBuf, posParent);
        }
        else
        {
            nn::util::SNPrintf(buf, sizeof(buf) / sizeof(RomPathChar), "+ / (null) [%08X]\n", posParent);
        }
        RomPathTool::Print(buf);
    }

    StoragePosition posDirectory = dirEntry.posDirectory;
    while (posDirectory != POSITION_NONE)
    {
        // 再帰的にダンプを行います。
        // ディレクトリ数が多いとスタックオーバーフローの可能性があります。
        NN_RESULT_DO(DumpDirectory(posDirectory, depth + 1));

        // 次のディレクトリエントリーを取得します。
        RomEntryKey nextDirKey;
        DirectoryEntry nextDirEntry;
        NN_RESULT_DO(m_TableDirectory.GetByPosition(&nextDirKey, &nextDirEntry, NULL, NULL, posDirectory));

        // 兄弟リンクを辿ります。
        posDirectory = nextDirEntry.posNext;
    }

    StoragePosition posFile = dirEntry.posFile;
    while (posFile != POSITION_NONE)
    {
        // 現在のファイルエントリーを取得します。
        RomEntryKey fileKey;
        FileEntry fileEntry;
        size_t length = 0;
        NN_RESULT_DO(m_TableFile.GetByPosition(&fileKey, &fileEntry, readBuf, &length, posFile));
        readBuf[length / sizeof(RomPathChar)] = StringTraitsRom::NUL;

        for (int i = 0; i < (depth + 1) * 2; i++)
        {
            RomPathTool::Print(" ");
        }
        nn::util::SNPrintf(buf, sizeof(buf) / sizeof(RomPathChar), "+ / %s [%08X]\n", readBuf, posFile);
        RomPathTool::Print(buf);

        // 兄弟リンクを辿ります。
        posFile = fileEntry.posNext;
    }

    return ResultSuccess();
}

/*!
    :private

    @brief ファイル構成をダンプします。

           デバッグ用途の関数です。内部でスタックを多く消費します。

    @return 関数の処理結果を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
Result HierarchicalRomFileTableTemplate<
           DirectoryBucketStorage_,
           DirectoryEntryStorage_,
           FileBucketStorage_,
           FileEntryStorage_
       >::DumpTree()
{
    NN_SDK_LOG("==== DUMP START ====\n");

    // ルートディレクトリ以下をダンプします。
    NN_RESULT_DO(DumpDirectory(POSITION_ROOT_DIRECTORY, 0));

    NN_SDK_LOG("==== DUMP FINISHED ====\n");

    return ResultSuccess();
}
#endif // !defined(NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK) && !defined(NN_COMPATIBLE_SDK) && defined(NN_DBM_CREATE_METADATA) && defined(_MSC_VER)

}}
