﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fsa/fs_IFileSystem.h>

#include <nn/fs/detail/fs_Newable.h>

namespace nn { namespace fat {
    class FatFileSystem;
}}

namespace nn { namespace fssystem {


    /**
        @brief FatFileSystem 上でファイル連結により 4GB 超えファイルを表現するファイルシステムです。
    */
    class ConcatenationFileSystem : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
    {
        NN_DISALLOW_COPY(ConcatenationFileSystem);

    public:
        static const int EntryNameLengthMax = 768;

        explicit ConcatenationFileSystem(std::unique_ptr<nn::fat::FatFileSystem>&& pBaseFileSystem) NN_NOEXCEPT;
        virtual ~ConcatenationFileSystem() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteFile(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCreateDirectory(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteDirectory(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCleanDirectoryRecursively(const char *path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;

        Result GetFileSize(int64_t* outValue, const char *path) NN_NOEXCEPT;

    private:
        std::unique_ptr<nn::fat::FatFileSystem> m_pBaseFileSystem; // TODO: shared_ptr 化
        const int64_t m_DivisionSize;

    private:
        Result ResolveFullPath(char* outPath, size_t outPathSize, const char* offsetPath) const NN_NOEXCEPT;
        Result GetInternalFileCount(int* outValue, const char* path) NN_NOEXCEPT;
        bool IsConcatenationFile(const char* path) NN_NOEXCEPT;
        Result CleanDirectoryRecursivelyImpl(const char* path) NN_NOEXCEPT;

    };

}}

