﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <array>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/detail/fs_Newable.h>

namespace nn { namespace fssystem {

// 連結ファイル
template <int N>
class ConcatenationFile : public fs::fsa::IFile, public fs::detail::Newable
{
public:
    explicit ConcatenationFile(std::array<std::shared_ptr<fs::fsa::IFile>, N> pFileArray) NN_NOEXCEPT
        : m_pFileArray(pFileArray)
    {
    }

    virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        int64_t totalOffset = 0;
        size_t totalReadSize = 0;
        for (auto pFile : m_pFileArray)
        {
            int64_t fileSize;
            NN_RESULT_DO(pFile->GetSize(&fileSize));

            auto beginOffset = std::max(totalOffset, offset);
            auto endOffset = std::min(totalOffset + fileSize, offset + static_cast<int64_t>(size));
            if (beginOffset <= endOffset)
            {
                size_t readSize;
                NN_RESULT_DO(pFile->Read(&readSize, beginOffset - totalOffset, static_cast<char*>(buffer) + totalReadSize, static_cast<size_t>(endOffset - beginOffset), option));
                totalReadSize += readSize;

                if (static_cast<int64_t>(readSize) < endOffset - beginOffset)
                {
                    break;
                }
            }

            totalOffset += fileSize;

            if (totalOffset >= offset + static_cast<int64_t>(size))
            {
                break;
            }
        }

        *outValue = totalReadSize;
        NN_RESULT_SUCCESS;
    }

    virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        int64_t totalOffset = 0;
        size_t totalWrittenSize = 0;
        for (auto pFile : m_pFileArray)
        {
            int64_t fileSize;
            NN_RESULT_DO(pFile->GetSize(&fileSize));

            auto beginOffset = std::max(totalOffset, offset);
            auto endOffset = std::min(totalOffset + fileSize, offset + static_cast<int64_t>(size));
            if (beginOffset <= endOffset)
            {
                NN_RESULT_DO(pFile->Write(beginOffset - totalOffset, static_cast<const char*>(buffer) + totalWrittenSize, static_cast<size_t>(endOffset - beginOffset), option));
                totalWrittenSize += static_cast<size_t>(endOffset - beginOffset);
            }

            totalOffset += fileSize;

            if (totalOffset > offset + static_cast<int64_t>(size))
            {
                break;
            }
        }

        NN_RESULT_SUCCESS;
    }

    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
    {
        for (auto pFile : m_pFileArray)
        {
            NN_RESULT_DO(pFile->Flush());
        }
        NN_RESULT_SUCCESS;
    }
    virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        int64_t totalSize = 0;
        for (auto pFile : m_pFileArray)
        {
            int64_t size = 0;
            NN_RESULT_DO(pFile->GetSize(&size));
            totalSize += size;
        }

        *outValue = totalSize;
        NN_RESULT_SUCCESS;
    }
    virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(size);
        return fs::ResultUnsupportedOperation();
    }

    virtual Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outBuffer);
        NN_UNUSED(outBufferSize);
        NN_UNUSED(operationId);
        NN_UNUSED(offset);
        NN_UNUSED(size);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);
        return fs::ResultUnsupportedOperation();
    }

protected:
    std::array<std::shared_ptr<fs::fsa::IFile>, N> m_pFileArray;
};


}}

