﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Allocator.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/util/util_BinTypes.h>
#include <nn/util/util_BytePtr.h>

namespace nn { namespace fssystem {

namespace utilTool {
    class BucketTreeChecker;
}

/**
 * @brief   アドレスをキーに用いた、固定長データを持つエントリを探索するためのツリーです。
 *
 *  アドレスを元に、それに紐づくエントリ（データ）を探索するためのツリーです。\n
 *  ツリーはノードを持ち、ノードはエントリを持っています。
 *  アドレスからノードを特定し、そのノードからエントリが特定されます。
 */
class BucketTree
{
    NN_DISALLOW_COPY(BucketTree);

public:
    static const uint32_t Signature = NN_UTIL_CREATE_SIGNATURE_4('B','K','T','R'); //!< シグネチャ
    static const uint32_t Version; //!< バージョン番号

    static const size_t NodeSizeMin = 1024;       //!< ノードサイズの最小値
    static const size_t NodeSizeMax = 512 * 1024; //!< ノードサイズの最大値

public:
    class Visitor;

    /**
     * @brief   ツリーのヘッダを表します。
     */
    struct Header
    {
        uint32_t signature;  //!< シグネチャ
        uint32_t version;    //!< バージョン番号
        int32_t  entryCount; //!< エントリの総数
        int32_t  _reserved;  // 予約領域のため使用禁止です。

    public:
        /**
         * @brief   フォーマットします。
         *
         * @param[in]   entryCount エントリの総数
         *
         * @pre
         *      - 0 <= entryCount
         */
        void Format(int entryCount) NN_NOEXCEPT;

        /**
         * @brief   エラーチェックをします。
         *
         * @retval  ResultSuccess   データは正常です。
         * @retval  上記以外        エラーを検出しました。
         */
        Result Verify() const NN_NOEXCEPT;
    };
    NN_STATIC_ASSERT(std::is_pod<Header>::value);

    /**
     * @brief   ツリーのノードのヘッダを表します。
     */
    struct NodeHeader
    {
        int32_t index;  //!< 各階層でのノードのインデックス
        int32_t count;  //!< ノードに含まれる要素の数
        int64_t offset; //!< 管理しているオフセットの最大値

    public:
        /**
         * @brief   エラーチェックをします。
         *
         * @param[in]   nodeIndex   ノードのインデックス
         * @param[in]   nodeSize    １ノードのサイズ
         * @param[in]   entrySize   １エントリのサイズ
         *
         * @retval  ResultSuccess   データは正常です。
         * @retval  上記以外        エラーを検出しました。
         */
        Result Verify(
                   int nodeIndex,
                   size_t nodeSize,
                   size_t entrySize
               ) const NN_NOEXCEPT;
    };
    NN_STATIC_ASSERT(std::is_pod<NodeHeader>::value);

    /**
     * @brief   ストレージのデータをまとめて読み込むための情報です。
     */
    class ContinuousReadingInfo
    {
    public:
        /**
         * @brief   コンストラクタです。
         */
        ContinuousReadingInfo() NN_NOEXCEPT
            : m_ReadSize(0)
            , m_SkipCount(0)
            , m_IsDone(false)
        {
        }

        /**
         * @brief   リセットします。
         */
        void Reset() NN_NOEXCEPT
        {
            m_ReadSize = 0;
            m_SkipCount = 0;
            m_IsDone = false;
        }

        /**
         * @brief   まとめ読みの確認をスキップする回数を設定します。
         */
        void SetSkipCount(int count) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

            m_SkipCount = count;
        }

        /**
         * @brief   まとめ読みの確認をスキップする回数を取得します。
         */
        int GetSkipCount() const NN_NOEXCEPT
        {
            return m_SkipCount;
        }

        /**
         * @brief   まとめ読みの確認が必要か確認します。
         */
        bool CheckNeedScan() NN_NOEXCEPT
        {
            --m_SkipCount;
            return m_SkipCount <= 0;
        }

        /**
         * @brief   まとめ読みが完了したことを通知します。
         */
        void Done() NN_NOEXCEPT
        {
            m_ReadSize = 0;
            m_IsDone = true;
        }

        /**
         * @brief   まとめ読みが可能か取得します。
         */
        bool CanDo() const NN_NOEXCEPT
        {
            return 0 < m_ReadSize;
        }

        /**
         * @brief   まとめ読みが終わっているか取得します。
         */
        bool IsDone() const NN_NOEXCEPT
        {
            return m_IsDone;
        }

        /**
         * @brief   まとめ読みのサイズを設定します。
         */
        void SetReadSize(size_t size) NN_NOEXCEPT
        {
            m_ReadSize = size;
        }

        /**
         * @brief   まとめ読みのサイズを取得します。
         */
        size_t GetReadSize() const NN_NOEXCEPT
        {
            return m_ReadSize;
        }

    private:
        size_t m_ReadSize;  //!< まとめ読みのサイズ
        int m_SkipCount;    //!< まとめ読みの確認を省略する回数
        bool m_IsDone;      //!< まとめ読みが終わっているかどうか
    };

    //! アロケータです。
    //! @todo 適切な型に変更する
    typedef MemoryResource IAllocator;

public:
    /**
     * @brief   ヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    static int64_t QueryHeaderStorageSize() NN_NOEXCEPT
    {
        return sizeof(Header);
    }

    /**
     * @brief   ノード部分のストレージのサイズを取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   entrySize   １エントリのサイズ
     * @param[in]   entryCount  エントリの総数
     *
     * @return  ノード部分のストレージサイズを返します。
     *
     * @pre
     *      - sizeof(int64_t) <= entrySize
     *      - entrySize + sizeof(NodeHeader) <= nodeSize
     *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
     *      - nodeSize が２の累乗
     *      - 0 <= entryCount
     */
    static int64_t QueryNodeStorageSize(
                       size_t nodeSize,
                       size_t entrySize,
                       int entryCount
                   ) NN_NOEXCEPT;

    /**
     * @brief   エントリ部分のストレージのサイズを取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   entrySize   １エントリのサイズ
     * @param[in]   entryCount  エントリの総数
     *
     * @return  エントリ部分のストレージサイズを返します。
     *
     * @pre
     *      - sizeof(int64_t) <= entrySize
     *      - entrySize + sizeof(NodeHeader) <= nodeSize
     *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
     *      - nodeSize が２の累乗
     *      - 0 <= entryCount
     */
    static int64_t QueryEntryStorageSize(
                       size_t nodeSize,
                       size_t entrySize,
                       int entryCount
                   ) NN_NOEXCEPT;

public:
    /**
     * @brief   コンストラクタです。
     */
    BucketTree() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    ~BucketTree() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
     * @brief   初期化します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   nodeStorage     ノードストレージ
     * @param[in]   entryStorage    エントリストレージ
     * @param[in]   nodeSize        １ノードのサイズ
     * @param[in]   entrySize       １エントリのサイズ
     * @param[in]   entryCount      エントリの総数
     *
     * @retval  ResultSuccess                   正常に処理が終了しました。
     * @retval  ResultInvalidArgument           引数の指定が不適切です
     * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
     * @retval  上記以外                        ストレージの読み込みに失敗しました。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - sizeof(int64_t) <= entrySize
     *      - entrySize + sizeof(NodeHeader) <= nodeSize
     *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
     *      - nodeSize が２の累乗
     *      - 0 < entryCount
     *      - 未初期化
     *
     * この関数では、ストレージの読み込み処理を行います。
     */
    Result Initialize(
               IAllocator* pAllocator,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage,
               size_t nodeSize,
               size_t entrySize,
               int entryCount
           ) NN_NOEXCEPT;

    /**
     * @brief   エントリ無しで初期化します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   endOffset   管理する領域の終了オフセット
     *
     * @pre
     *      - NodeSizeMin <= nodeSize && nodeSize <= NodeSizeMax
     *      - nodeSize が２の累乗
     *      - 0 <= endOffset
     *      - 未初期化
     */
    void Initialize(size_t nodeSize, int64_t endOffset) NN_NOEXCEPT;

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   初期化済みかどうかを取得します。
     *
     * @retval true     初期化済みです。
     * @retval false    初期化していません。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return 0 < m_NodeSize;
    }

    /**
     * @brief   エントリが無いか取得します。
     *
     * @retval true     エントリはありません。
     * @retval false    エントリがあります。
     */
    bool IsEmpty() const NN_NOEXCEPT
    {
        return m_EntrySize == 0;
    }

    /**
     * @brief   仮想アドレスからエントリを検索します。
     *
     * @param[in,out]   pVisitor        巡回オブジェクトのポインタ
     * @param[in]       virtualAddress  エントリ検索用の仮想アドレス
     *
     * @retval  ResultSuccess                   正常に処理が終了しました。
     * @retval  ResultOutOfRange                エントリが見つかりませんでした。
     * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
     * @retval  上記以外                        ストレージの読み込みに失敗しました。
     *
     * @pre
     *      - pVisitor != nullptr
     *      - 0 <= virtualAddress
     *      - 初期化済み
     *
     * この関数では、ストレージの読み込み処理を行います。
     */
    Result Find(Visitor* pVisitor, int64_t virtualAddress) const NN_NOEXCEPT;

    /**
     * @brief   キャッシュを無効化します。
     *
     * @return  処理結果を返します。
     */
    Result InvalidateCache() NN_NOEXCEPT;

    /**
     * @brief   エントリの総数を取得します。
     *
     * @return  エントリの総数を返します。
     */
    int GetEntryCount() const NN_NOEXCEPT
    {
        return m_EntryCount;
    }

    /**
     * @brief   アロケータを取得します。
     *
     * @return  アロケータのポインタを返します。
     */
    IAllocator* GetAllocator() const NN_NOEXCEPT
    {
        return m_NodeL1.GetAllocator();
    }

    /**
     * @brief   管理している領域の先頭を取得します。
     *
     * @return  管理している領域の先頭を返します。
     */
    int64_t GetBegin() const NN_NOEXCEPT
    {
        // Initialize() で 0 <= m_BeginOffset は保障している
        return m_BeginOffset;
    }

    /**
     * @brief   管理している領域の末尾を取得します。
     *
     * @return  管理している領域の末尾を返します。
     */
    int64_t GetEnd() const NN_NOEXCEPT
    {
        // Initialize() で m_BeginOffset < m_EndOffset は保障している
        return m_EndOffset;
    }

    /**
     * @brief   管理している領域のサイズを取得します。
     *
     * @return  管理している領域のサイズを返します。
     */
    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_EndOffset - m_BeginOffset;
    }

    /**
     * @brief   指定したオフセットが管理している領域内にあるか判定します。
     *
     * @param[in]   offset  調査するオフセット
     *
     * @return  指定したオフセットが管理している領域内にあるかどうかを返します。
     */
    bool IsInclude(int64_t offset) const NN_NOEXCEPT
    {
        return m_BeginOffset <= offset && offset < m_EndOffset;
    }

    /**
     * @brief   指定した範囲が管理している領域内にあるか判定します。
     *
     * @param[in]   offset  調査するオフセット
     * @param[in]   size    調査する範囲のサイズ
     *
     * @return  指定した範囲が管理している領域内にあるかどうかを返します。
     */
    bool IsInclude(int64_t offset, int64_t size) const NN_NOEXCEPT
    {
        return 0 < size &&
               m_BeginOffset <= offset && // 0 <= m_BeginOffset なので 0 <= offset を保障
               size <= m_EndOffset - offset;
    }

private:
    /**
     * @brief   ツリーノードのバッファです。
     */
    class NodeBuffer
    {
        NN_DISALLOW_COPY(NodeBuffer);

    public:
        //! コンストラクタです。
        NodeBuffer() NN_NOEXCEPT
            : m_pAllocator(nullptr)
            , m_pHeader(nullptr)
        {
        }

        //! ムーブコンストラクタです。
        NodeBuffer(NodeBuffer&& buffer) NN_NOEXCEPT
            : m_pAllocator(buffer.m_pAllocator)
            , m_pHeader(buffer.m_pHeader)
        {
            buffer.m_pAllocator = nullptr;
            buffer.m_pHeader = nullptr;
        }

        // デストラクタです。
        ~NodeBuffer() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_pHeader == nullptr);
        }

        //! アロー演算子です。
        NodeHeader* operator->() const NN_NOEXCEPT
        {
            return Get();
        }

        //! ムーブ代入演算子です。
        NodeBuffer& operator=(NodeBuffer&& buffer) NN_NOEXCEPT
        {
            if( this != &buffer )
            {
                NN_SDK_REQUIRES(m_pHeader == nullptr);

                m_pAllocator = buffer.m_pAllocator;
                m_pHeader = buffer.m_pHeader;

                buffer.m_pAllocator = nullptr;
                buffer.m_pHeader = nullptr;
            }
            return *this;
        }

        //! バッファを確保します。
        bool Allocate(IAllocator* pAllocator, size_t nodeSize) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_pHeader == nullptr);

            m_pAllocator = pAllocator;
            m_pHeader = pAllocator->allocate(nodeSize, sizeof(int64_t));

            NN_SDK_ASSERT(nn::util::BytePtr(m_pHeader).IsAligned(sizeof(int64_t)));

            return m_pHeader != nullptr;
        }

        //! バッファを解放します。
        void Free(size_t nodeSize) NN_NOEXCEPT
        {
            if( m_pHeader )
            {
                m_pAllocator->deallocate(m_pHeader, nodeSize);
                m_pHeader = nullptr;
            }
            m_pAllocator = nullptr;
        }

        //! バッファを０埋めします。
        void FillZero(size_t nodeSize) const NN_NOEXCEPT
        {
            if( m_pHeader )
            {
                std::memset(m_pHeader, 0, nodeSize);
            }
        }

        //! ノードヘッダを取得します。
        NodeHeader* Get() const NN_NOEXCEPT
        {
            return reinterpret_cast<NodeHeader*>(m_pHeader);
        }

        //! ノードを取得します。
        template< typename T >
        T* Get() const NN_NOEXCEPT
        {
            NN_STATIC_ASSERT(std::is_pod<T>::value);
            NN_STATIC_ASSERT(sizeof(NodeHeader) == sizeof(T));
            return reinterpret_cast<T*>(m_pHeader);
        }

        //! アロケータを取得します。
        IAllocator* GetAllocator() const NN_NOEXCEPT
        {
            return m_pAllocator;
        }

    private:
        IAllocator* m_pAllocator;   //!< アロケータのポインタ
        void* m_pHeader;            //!< バッファのポインタ
    };

private:
    /**
     * @brief   ノードに含まれるエントリの数を取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   entrySize   １エントリのサイズ
     *
     * @return  ノードに含まれるエントリの数を返します。
     */
    static int GetEntryCount(size_t nodeSize, size_t entrySize) NN_NOEXCEPT
    {
        // NOTE: 引数は事前検証済みのものを使うためここでは検証しない
        return static_cast<int>((nodeSize - sizeof(NodeHeader)) / entrySize);
    }

    /**
     * @brief   ノードに含まれるオフセットの数を取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     *
     * @return  ノードに含まれるオフセットの数を返します。
     */
    static int GetOffsetCount(size_t nodeSize) NN_NOEXCEPT
    {
        // NOTE: 引数は事前検証済みのものを使うためここでは検証しない
        return static_cast<int>((nodeSize - sizeof(NodeHeader)) / sizeof(int64_t));
    }

    /**
     * @brief   エントリセットの数を取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   entrySize   １エントリのサイズ
     * @param[in]   entryCount  エントリの総数
     *
     * @return  エントリセットの数を返します。
     */
    static int GetEntrySetCount(
                   size_t nodeSize,
                   size_t entrySize,
                   int entryCount
               ) NN_NOEXCEPT;

    /**
     * @brief   L2 のノード数を取得します。
     *
     * @param[in]   nodeSize    １ノードのサイズ
     * @param[in]   entrySize   １エントリのサイズ
     * @param[in]   entryCount  エントリの総数
     *
     * @return  L2 のノード数を返します。
     */
    static int GetNodeL2Count(
                   size_t nodeSize,
                   size_t entrySize,
                   int entryCount
               ) NN_NOEXCEPT;

private:
    /**
     * @brief   ストレージのデータがまとめて読み込めるか確認するためのパラメータです。
     */
    template< typename TEntry >
    struct ContinuousReadingParam
    {
        int64_t offset;
        size_t size;
        NodeHeader entrySet;
        int entryIndex;
        TEntry entry;
    };

private:
    /**
     * @brief   ストレージのデータをまとめて読み込めるか確認します。
     */
    template< typename TEntry >
    Result ScanContinuousReading(
        ContinuousReadingInfo* pOutInfo,
        const ContinuousReadingParam<TEntry>& param
    ) const NN_NOEXCEPT;

    /**
     * @brief   L2 ノードが存在するかどうかを取得します。
     *
     * @return  L2 ノードが存在するかどうかを返します。
     */
    bool IsExistL2() const NN_NOEXCEPT
    {
        return m_OffsetCount < m_EntrySetCount;
    }

    /**
     * @brief   L1 ノード内に L2 ノードのオフセットが存在するかどうかを取得します。
     *
     * @return  L1 ノード内に L2 ノードのオフセットが存在するかどうかを返します。
     */
    bool IsExistOffsetL2OnL1() const NN_NOEXCEPT
    {
                           // L1 ノード内に L2 オフセットあり
        return IsExistL2() && m_NodeL1->count < m_OffsetCount;
    }

    /**
     * @brief   エントリセットのインデックスを取得します。
     *
     * @param[in]   nodeIndex       L2 ノードのインデックス
     * @param[in]   offsetIndex     L2 ノード内のオフセットのインデックス
     *
     * @return  エントリセットのインデックスを返します。
     */
    int GetEntrySetIndex(int nodeIndex, int offsetIndex) const NN_NOEXCEPT
    {
        return (m_OffsetCount - m_NodeL1->count) + // L1 ノードに詰めたエントリセットの数
                m_OffsetCount * nodeIndex +
                offsetIndex;
    }

private:
    mutable fs::SubStorage m_NodeStorage;   //!< ノードストレージ
    mutable fs::SubStorage m_EntryStorage;  //!< テーブルストレージ
    NodeBuffer m_NodeL1;                    //!< L1 ノードのキャッシュ
    size_t m_NodeSize;                      //!< １ノードのサイズ
    size_t m_EntrySize;                     //!< １エントリのサイズ
    int m_EntryCount;                       //!< エントリの総数
    int m_OffsetCount;                      //!< １ノード当たりのオフセット数
    int m_EntrySetCount;                    //!< ツリーが管理するエントリセットの総数
    int64_t m_BeginOffset;                  //!< ツリーの先頭オフセット
    int64_t m_EndOffset;                    //!< ツリーの末尾オフセット

    friend class BucketTreeTest;
    friend class BucketTreeBuilder;
    friend class utilTool::BucketTreeChecker;
};


/**
 * @brief   BucketTree のエントリを巡回するクラスです。
 *
 * 内部では、BucketTree::Find() を呼び出した BucketTree のインスタンスと、
 * その初期化で渡したアロケータを参照しています。\n
 * このクラスのインスタンスを解放した後に、
 * BucketTree のインスタンスとアロケータを解放するように注意してください。
 */
class BucketTree::Visitor
{
    NN_DISALLOW_COPY(Visitor);

public:
    /**
     * @brief   コンストラクタです。
     */
    Visitor() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    ~Visitor() NN_NOEXCEPT;

    /**
     * @brief   巡回位置を次のエントリに進めます。
     *
     * @retval  ResultSuccess       正常に処理が終了しました。
     * @retval  ResultOutOfRange    ツリー範囲外に移動しようとしました。
     * @retval  上記以外            ストレージの読み込みに失敗しました。
     *
     * この関数では、ストレージの読み込み処理を行います。
     */
    Result MoveNext() NN_NOEXCEPT;

    /**
     * @brief   巡回位置を前のエントリに戻します。
     *
     * @retval  ResultSuccess       正常に処理が終了しました。
     * @retval  ResultOutOfRange    ツリー範囲外に移動しようとしました。
     * @retval  上記以外            ストレージの読み込みに失敗しました。
     *
     * この関数では、ストレージの読み込み処理を行います。
     */
    Result MovePrevious() NN_NOEXCEPT;

    /**
     * @brief   インスタンスが有効か判定します。
     *
     * @return  有効なインスタンスかどうかを返します。
     */
    bool IsValid() const NN_NOEXCEPT
    {
        return 0 <= m_EntryIndex;
    }

    /**
     * @brief   巡回位置を次のエントリに進められるか判定します。
     *
     * @return  巡回位置を次のエントリに進められるかどうかを返します。
     */
    bool CanMoveNext() const NN_NOEXCEPT
    {
        return IsValid() &&
               (m_EntryIndex + 1 < m_EntrySet.info.count ||
                m_EntrySet.info.index + 1 < m_EntrySetCount);
    }

    /**
     * @brief   巡回位置を前のエントリに戻せられるか判定します。
     *
     * @return  巡回位置を前のエントリに戻せられるかどうかを返します。
     */
    bool CanMovePrevious() const NN_NOEXCEPT
    {
        return IsValid() &&
               (0 < m_EntryIndex || 0 < m_EntrySet.info.index);
    }

    /**
     * @brief   ストレージのデータをまとめて読み込めるか確認します。
     *
     * @param[out]  pOutInfo    まとめ読みの情報を出力するアドレス
     * @param[in]   offset      まとめ読みの開始オフセット
     * @param[in]   size        まとめ読みのサイズ
     *
     * @retval  ResultSuccess   正常に処理が終了しました。
     * @retval  上記以外        ストレージの読み込みに失敗しました。
     */
    template< typename TEntry >
    Result ScanContinuousReading(
        ContinuousReadingInfo* pOutInfo,
        int64_t offset,
        size_t size
    ) const NN_NOEXCEPT;

    /**
     * @brief   エントリを取得します。
     *
     * @return  エントリの先頭アドレスを返します。
     *
     * @pre
     *      - IsValid() != false
     */
    const void* Get() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        return m_pEntry;
    }

    /**
     * @brief   エントリを取得します。
     *
     * @return  エントリの先頭アドレスを返します。
     *
     * @pre
     *      - IsValid() != false
     */
    template< typename T >
    const T* Get() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        return reinterpret_cast<const T*>(m_pEntry);
    }

    /**
     * @brief   巡回しているツリーを取得します。
     *
     * @return  巡回しているツリーのポインタを返します。
     */
    const BucketTree* GetTree() const NN_NOEXCEPT
    {
        return m_pTree;
    }

private:
    /**
     * @brief   エントリセットのヘッダです。
     */
    union EntrySetHeader
    {
        NodeHeader header;  //!< ノードヘッダ
        struct Info
        {
            int32_t index;  //!< エントリセットのインデックス
            int32_t count;  //!< エントリセットに含まれるエントリ数
            int64_t end;    //!< エントリセットの終了オフセット
            int64_t begin;  //!< エントリセットの開始オフセット
        }
        info;   //!< エントリセットの情報
        NN_STATIC_ASSERT(std::is_pod<Info>::value);
    };

    NN_STATIC_ASSERT(std::is_pod<EntrySetHeader>::value);

private:
    /**
     * @brief   初期化します。
     *
     * @param[in]   pTree   巡回するツリーのポインタ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pTree != nullptr
     */
    Result Initialize(const BucketTree* pTree) NN_NOEXCEPT;

    /**
     * @brief   エントリを検索します。
     *
     * @param[in]   virtualAddress  エントリ検索用の仮想アドレス
     *
     * @retval  ResultSuccess       エントリが見つかりました。
     * @retval  ResultOutOfRange    エントリが見つかりませんでした。
     * @retval  上記以外            ストレージの読み込みに失敗しました。
     *
     * @pre
     *      - 初期化済み
     *
     * この関数では、ストレージの読み込み処理を行います。
     */
    Result Find(int64_t virtualAddress) NN_NOEXCEPT;

    /**
     * @brief   ストレージからエントリセットを検索します。
     *
     * @param[out]  pOutIndex       見つけたインデックスの出力先
     * @param[in]   virtualAddress  エントリ検索用仮想アドレス
     * @param[in]   nodeIndex       検索する L2 ノードのインデックス
     *
     * @retval  ResultSuccess       エントリセットが見つかりました。
     * @retval  ResultOutOfRange    エントリセットが見つかりませんでした。
     * @retval  上記以外            ストレージの読み込みに失敗しました。
     *
     * この関数では、ストレージの読み込み処理を行います。
     */
    Result FindEntrySet(
               int* pOutIndex,
               int64_t virtualAddress,
               int nodeIndex
           ) NN_NOEXCEPT;

    Result FindEntrySetWithBuffer(
               int* pOutIndex,
               int64_t virtualAddress,
               int nodeIndex,
               char* buffer
           ) NN_NOEXCEPT;

    Result FindEntrySetWithoutBuffer(
               int* pOutIndex,
               int64_t virtualAddress,
               int nodeIndex
           ) NN_NOEXCEPT;

    /**
     * @brief   ストレージからエントリを検索します。
     *
     * @param[in]   virtualAddress  エントリ検索用仮想アドレス
     * @param[in]   entrySetIndex   検索するエントリセットのインデックス
     *
     * @retval  ResultSuccess       エントリが見つかりました。
     * @retval  ResultOutOfRange    エントリが見つかりませんでした。
     * @retval  上記以外            ストレージの読み込みに失敗しました。
     *
     * この関数では、ストレージの読み込み処理を行います。
     */
    Result FindEntry(int64_t virtualAddress, int entrySetIndex) NN_NOEXCEPT;

    Result FindEntryWithBuffer(
               int64_t virtualAddress,
               int entrySetIndex,
               char* buffer
           ) NN_NOEXCEPT;

    Result FindEntryWithoutBuffer(
               int64_t virtualAddress,
               int entrySetIndex
           ) NN_NOEXCEPT;

private:
    const BucketTree* m_pTree;  //!< エントリを保持するツリーのポインタ
    void* m_pEntry;             //!< エントリデータの先頭アドレス
    int m_EntryIndex;           //!< エントリセット内のエントリのインデックス
    int m_EntrySetCount;        //!< エントリセットの数
    EntrySetHeader m_EntrySet;  //!< セントリセットの情報

    friend class BucketTree;
};


}} // nn::fssystem
