﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <memory>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/os/os_LightEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Semaphore.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace fssystem {

class PooledThread;
class AsynchronousRunner;

//! スレッドプールを管理するクラス
class ThreadPool
{
    NN_DISALLOW_COPY(ThreadPool);

public:
    // スレッドスタックを解放する関数です。
    typedef void (*StackDeallocator)(char* buffer, size_t bufferSize, void* pArgument);

public:
    /**
    * @brief      コンストラクタです。
    *
    * @param[in]  pThreads    スレッド
    * @param[in]  threadCount スレッド数
    */
    ThreadPool(PooledThread* pThreads, int threadCount) NN_NOEXCEPT;

    /**
    * @brief      デストラクタです。
    */
    ~ThreadPool() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
    * @brief      スレッドプールを初期化します。
    *
    * @param[in]  stackBuffer       スレッドスタックのバッファ
    * @param[in]  stackBufferSize   スレッドスタックのバッファサイズ
    */
    Result Initialize(char* stackBuffer, size_t stackBufferSize) NN_NOEXCEPT
    {
        return Initialize(stackBuffer, stackBufferSize, Deallocate, nullptr);
    }

    /**
    * @brief      スレッドプールを初期化します。
    *
    * @param[in]  stackBuffer       スレッドスタックのバッファ
    * @param[in]  stackBufferSize   スレッドスタックのバッファサイズ
    * @brief[in]  stackDeallocator  スレッドスタックの解放関数
    * @brief[in]  pStackArgument    スレッドスタックの解放関数に渡すユーザ引数
    */
    Result Initialize(
               char* stackBuffer,
               size_t stackBufferSize,
               StackDeallocator stackDeallocator,
               void* pStackArgument
           ) NN_NOEXCEPT;

    /**
    * @brief      スレッドプールを破棄します。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief    AsynchronousRunners を実行準備します。
    *
    * @param    [out]   outActiveRunnerCount    実行準備できた AsynchronousRunners の数
    * @param    [in]    pRunnerList             AsynchronousRunners のリスト
    * @param    [in]    runnerListCount         実行準備したい AsynchronousRunners の数
    *
    * @pre      0 < runnerListCount <= MaxRunnerCount
    *
    * @post     0 < outAcquireCount <= runnerListCount
    */
    template<typename T, int64_t Size>
    void ActivateAsynchronousRunners(
        int64_t* outActiveRunnerCount,
        T (&runnerList)[Size],
        int64_t runnerListCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(Size, runnerListCount);
        AsynchronousRunner* pRunners[Size];
        for(int i = 0; i < runnerListCount; ++i)
        {
            pRunners[i] = &runnerList[i];
        }
        ActivateAsynchronousRunnersImpl(outActiveRunnerCount, pRunners, runnerListCount);
    }

private:
    // スレッドスタックの解放関数です。
    static void Deallocate(char* buffer, size_t bufferSize, void* pArgument) NN_NOEXCEPT
    {
        NN_UNUSED(buffer);
        NN_UNUSED(bufferSize);
        NN_UNUSED(pArgument);
    }

private:
    // スレッドを解放します。
    void Release() NN_NOEXCEPT
    {
        m_Semaphore.Release();
    }

    void ActivateAsynchronousRunnersImpl(int64_t* outActiveRunnerCount, AsynchronousRunner** pRunnerList, int64_t runnerListCount) NN_NOEXCEPT;

private:
    PooledThread* const m_pThreads;
    const int m_ThreadCount;
    os::Semaphore m_Semaphore;
    char* m_StackBuffer;
    size_t m_StackBufferSize;
    StackDeallocator m_StackDeallocator;
    void* m_pStackArgument;

    friend class PooledThread;
};

//! スレッドプールの所有するスレッドの情報を保持するクラス
class PooledThread
{
    NN_DISALLOW_COPY(PooledThread);

public:
    /**
    * @brief      コンストラクタです。
    */
    PooledThread() NN_NOEXCEPT
        : m_pOwner(nullptr)
        , m_Thread()
        , m_StartEvent(os::EventClearMode_AutoClear)
        , m_EndEvent(os::EventClearMode_AutoClear)
        , m_ThreadPriority(NN_SYSTEM_THREAD_PRIORITY(fs, WorkerThreadPool))
        , m_IsRunning(false)
        , m_pRunner()
        , m_ServiceContext()
    {
    }

    // スレッド処理を開始します。
    void Start() NN_NOEXCEPT;

    void Release() NN_NOEXCEPT
    {
        m_pRunner = nullptr;
        m_IsRunning.store(false);
        m_pOwner->Release();
    }

    void Wait() NN_NOEXCEPT
    {
        m_EndEvent.Wait();
    }

private:
    // 初期化をします。
    Result Initialize(
               ThreadPool* pOwner,
               void* pStack,
               size_t stackSize,
               os::ThreadFunction func
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOwner);
        NN_SDK_REQUIRES_NOT_NULL(pStack);
        NN_SDK_REQUIRES_GREATER(stackSize, static_cast<size_t>(0));
        NN_SDK_REQUIRES_ALIGNED(pStack, os::ThreadStackAlignment);
        NN_SDK_REQUIRES_NOT_NULL(func);

        if( m_pOwner == nullptr )
        {
            NN_RESULT_DO(os::CreateThread(
                &m_Thread,
                func,
                this,
                pStack,
                stackSize,
                NN_SYSTEM_THREAD_PRIORITY(fs, WorkerThreadPool)
            ));

            os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(fs, WorkerThreadPool));
            os::StartThread(&m_Thread);

            m_pOwner = pOwner;
        }

        NN_RESULT_SUCCESS;
    }

    // 終了処理をします。
    void Finalize() NN_NOEXCEPT
    {
        if( m_pOwner != nullptr )
        {
            NN_SDK_ASSERT(m_pRunner == nullptr);

            m_StartEvent.Signal();
            os::DestroyThread(&m_Thread);

            m_pOwner = nullptr;
        }
    }

    // スレッドの確保を試みます。
    bool TryAcquire() NN_NOEXCEPT
    {
        auto isRunning = false;
        return m_IsRunning.compare_exchange_strong(isRunning, true);
    }

    // スレッド処理を実行します。
    void Invoke() NN_NOEXCEPT;

private:
    void SetRunner(AsynchronousRunner* pRunner) NN_NOEXCEPT;

    // スレッド処理です。
    static void ThreadFunction(void* pArgument) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pArgument);
        reinterpret_cast<PooledThread*>(pArgument)->Invoke();
    }

private:
    ThreadPool* m_pOwner;
    os::ThreadType m_Thread;
    os::LightEvent m_StartEvent;
    os::LightEvent m_EndEvent;
    volatile int m_ThreadPriority;
    std::atomic<bool> m_IsRunning;
    AsynchronousRunner* m_pRunner;
    ServiceContext m_ServiceContext;

    friend class ThreadPool;
};

//! スレッドプールを使用して処理を非同期実行するクラス
class AsynchronousRunner
{
    NN_DISALLOW_COPY(AsynchronousRunner);

public:
    /**
    * @brief      コンストラクタです。
    */
    AsynchronousRunner() NN_NOEXCEPT
        : m_Result(ResultSuccess())
        , m_IsResultWaiting(false)
        , m_pThread(nullptr)
    {
    }

    /**
    * @brief      デストラクタです。
    */
    virtual ~AsynchronousRunner() NN_NOEXCEPT
    {
    }

    /**
    * @brief      非同期処理の結果を取得します。
    *
    * @return     処理結果を返します。
    *
    * 内部で非同期処理が終了するのを待ちます。
    */
    Result GetResult() NN_NOEXCEPT
    {
        Wait();
        return m_Result;
    }

    void Start() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pThread);
        m_IsResultWaiting = true;
        m_pThread->Start();
    }

protected:
    /**
    * @brief      Run() で非同期実行する処理を実装します。
    *
    * @return     処理結果を返します。
    */
    virtual Result DoRun() NN_NOEXCEPT = 0;

    void Release() NN_NOEXCEPT
    {
        if( m_pThread != nullptr )
        {
            Wait();
            m_pThread->Release();
            m_pThread = nullptr;
        }
    }

private:
    void Wait() NN_NOEXCEPT
    {
        if( m_IsResultWaiting )
        {
            NN_SDK_ASSERT_NOT_NULL(m_pThread);
            m_pThread->Wait();
            m_IsResultWaiting = false;
        }
    }

    void SetThread(PooledThread* pThread) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pThread == nullptr);
        m_pThread = pThread;
    }

    // 実際に処理を実行します。
    void Invoke() NN_NOEXCEPT
    {
        m_Result = DoRun();
    }

private:
    Result m_Result;
    bool m_IsResultWaiting;
    PooledThread* m_pThread;

    friend class PooledThread;
};

}}
