﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/crypto/crypto_AesEncryptor.h>

namespace nn { namespace fssystem {

/**
 * @brief   AES-CTR 暗号化ストレージです。
 */
class AesCtrStorage : public fs::IStorage, public fs::detail::Newable
{
    NN_DISALLOW_COPY(AesCtrStorage);

public:
    static const size_t BlockSize = crypto::AesEncryptor128::BlockSize;
    static const size_t KeySize = 16;
    static const size_t IvSize  = 16;

public:
    /**
    * @brief   IV を生成します。
    *
    * @param[out]  pOutIv       IV の出力先のポインタ
    * @param[in]   outSize      IV のサイズ
    * @param[in]   upperValue   IV の上位 8 バイトの数値
    * @param[in]   offset       オフセット値
    *
    * @pre
    *      - pOutIv != nullptr
    *      - outSize == IvSize
    *      - 0 <= offset
    */
    static void MakeIv(
                    void* pOutIv,
                    size_t outSize,
                    uint64_t upperValue,
                    int64_t offset
                ) NN_NOEXCEPT;

    /**
    * @brief   コンストラクタです。
    *
    * @param[in]   pBaseStorage                         元になるストレージのポインタ
    * @param[in]   pKey                                 鍵の先頭アドレス
    * @param[in]   keySize                              鍵のサイズ
    * @param[in]   pIv                                  IV の先頭アドレス
    * @param[in]   ivSize                               IV のサイズ
    *
    * @pre
    *      - pBaseStorage != nullptr
    *      - pKey != nullptr
    *      - keySize == KeySize
    *      - pIv != nullptr
    *      - ivSize == IvSize
    */
    AesCtrStorage(
        IStorage* pBaseStorage,
        const void* pKey,
        size_t keySize,
        const void* pIv,
        size_t ivSize
    ) NN_NOEXCEPT;

    /**
     * @brief   ストレージを読み込みます。
     *
     * @param[in]   offset  オフセット
     * @param[in]   buffer  バッファの先頭アドレス
     * @param[in]   size    サイズ
     */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   ストレージに書き出します。
     *
     * @param[in]   offset  オフセット
     * @param[in]   buffer  バッファの先頭アドレス
     * @param[in]   size    サイズ
     */
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    /**
     * @brief   ストレージのサイズを取得します。
     *
     * @param[out]  pOutValue   サイズの出力先
     */
    virtual Result GetSize(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       範囲指定処理を行います。
     *
     * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
     * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
     * @param[in]   operationId      範囲指定処理の種類
     * @param[in]   offset           範囲指定処理開始位置
     * @param[in]   size             範囲指定処理を行うデータサイズ
     * @param[in]   inBuffer         範囲指定処理に渡すバッファ
     * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
     *
     * @return      関数の処理結果を返します。
     */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

private:
    IStorage* const m_pBaseStorage;
    char m_Key[KeySize];
    char m_Iv[IvSize];
};

}}
