﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_BucketTree.h>

namespace nn { namespace fssystem {

namespace utilTool {
    class AesCtrCounterExtendedStorageChecker;
}

/**
 * @brief   AES-CTR により暗号化されたストレージのデータを読み込むクラスです。
 */
class AesCtrCounterExtendedStorage : public fs::IStorage, public fs::detail::Newable
{
    NN_DISALLOW_COPY(AesCtrCounterExtendedStorage);

public:
    static const size_t BlockSize = 16;   //!< AES-CTR のブロックサイズです。
    static const size_t KeySize = 16;     //!< AES-CTR の鍵のサイズです。
    static const size_t CounterSize = 16; //!< AES-CTR のカウンタのサイズです。

    static const size_t NodeSize = 16 * 1024; //!< BucketTree のノードサイズ

    typedef BucketTree::IAllocator IAllocator;

    typedef void(*DecryptFunction)(void* pDst, size_t dstSize, int index, const void* pEncryptedKey, size_t encryptedKeySize, const void* pIv, size_t ivSize, const void* pSrc, size_t srcSize);

    class IDecryptor
    {
    public:
        virtual ~IDecryptor() NN_NOEXCEPT
        {
        }

        virtual void Decrypt(
            void* buffer, size_t size,
            void* encryptedKey, size_t keySize,
            void* pIv, size_t ivSize) NN_NOEXCEPT = 0;

        virtual bool HasExternalDecryptionKey() const NN_NOEXCEPT = 0;
    };

public:
    /**
     * @brief   ヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    static int64_t QueryHeaderStorageSize() NN_NOEXCEPT
    {
        return BucketTree::QueryHeaderStorageSize();
    }

    /**
     * @brief   ノード部分のストレージのサイズを取得します。
     *
     * @param[in]   entryCount  エントリの総数
     *
     * @return  ノード部分のストレージサイズを返します。
     */
    static int64_t QueryNodeStorageSize(int entryCount) NN_NOEXCEPT
    {
        return BucketTree::QueryNodeStorageSize(NodeSize, sizeof(Entry), entryCount);
    }

    /**
     * @brief   エントリ部分のストレージサイズを取得します。
     *
     * @param[in]   entryCount  エントリの総数
     *
     * @return  エントリ部分のストレージサイズを返します。
     */
    static int64_t QueryEntryStorageSize(int entryCount) NN_NOEXCEPT
    {
        return BucketTree::QueryEntryStorageSize(NodeSize, sizeof(Entry), entryCount);
    }

public:
    /**
     * @brief   外部関数で復号化する Decryptor を作成
     *
     * @param[out]  ppOutValue  生成されたDecryptorへのポインタ
     * @param[in]   function    復号化関数
     * @param[in]   keyIndex    鍵インデックス
     *
     * @return  関数の実行結果を返します。
     */
    static Result CreateExternalDecryptor(
        std::unique_ptr<IDecryptor>* ppOutValue,
        DecryptFunction function, int keyIndex) NN_NOEXCEPT;

    /**
     * @brief   ソフトウェアで復号化する Decryptor を作成
     *
     * @param[out]  ppOutValue                          生成されたDecryptorへのポインタ
     *
     * @return  関数の実行結果を返します。
     */
    static Result CreateSoftwareDecryptor(std::unique_ptr<IDecryptor>* ppOutValue) NN_NOEXCEPT;

public:
    /**
     * @brief   コンストラクタです。
     */
    AesCtrCounterExtendedStorage() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    virtual ~AesCtrCounterExtendedStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        Finalize();
    }

    /**
     * @brief   初期化をします。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   pKey            復号に使用する鍵の先頭アドレス
     * @param[in]   keySize         復号に使用する鍵のサイズ
     * @param[in]   secureValue     復号に使用するセキュア数値
     * @param[in]   counterOffset   復号に使用するカウンタのオフセット
     * @param[in]   dataStorage     暗号化したデータを格納したストレージ
     * @param[in]   nodeStorage     テーブル検索のための木構造を格納したノードストレージ
     * @param[in]   entryStorage    カウンタのテーブルデータを格納したストレージ
     * @param[in]   entryCount      テーブルデータのエントリの総数
     * @param[in]   pDecryptor      AesCtr の復号に使う関数
     *
     * @return  関数の実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - pKey != nullptr
     *      - keySize == KeySize
     *      - 0 <= counterOffset
     *      - pDecryptor != nullptr
     *      - 未初期化
     *
     * nodeStorage, entryStorage で使用するストレージは、4KB アライメントすることを強く推奨します。
     */
    Result Initialize(
               IAllocator* pAllocator,
               const void* pKey,
               size_t keySize,
               uint32_t secureValue,
               int64_t counterOffset,
               fs::SubStorage dataStorage,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage,
               int entryCount,
               std::unique_ptr<IDecryptor>&& pDecryptor
           ) NN_NOEXCEPT;

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   初期化済みかどうかを取得します。
     *
     * @retval true     初期化済みです。
     * @retval false    初期化していません。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_Table.IsInitialized();
    }

    /**
     * @brief   読み込みを行います。
     *
     * @param[in]   offset  読み込むオフセット
     * @param[out]  buffer  読み込んだデータを格納するバッファ
     * @param[in]   size    読み込むサイズ
     *
     * @return  関数の実行結果を返します。
     *
     * @pre
     *      - 0 <= offset
     *      - is_aligned(offset, BlockSize) != false
     *      - size == 0 || buffer != nullptr
     *      - is_aligned(size, BlockSize) != false
     *      - 初期化済み
     *
     * この関数は、以下の条件を満たした場合に限りスレッドセーフとなります。
     *      - Initialize() で指定したストレージすべてがスレッドセーフ
     */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   ストレージのサイズを取得します。
     *
     * @return  関数の実行結果を返します。
     *
     * @pre
     *      - pOutSize != nullptr
     */
    virtual Result GetSize(int64_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);

        *pOutSize = m_Table.GetSize();

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   テーブルのエントリです。
     */
    struct Entry
    {
        char offset[sizeof(int64_t)];   //!< オフセット
        int32_t reserved;
        int32_t generation;             //!< 更新世代

        //! オフセットを設定します。
        void SetOffset(int64_t value) NN_NOEXCEPT
        {
            // 8 バイトアライメント対策
            std::memcpy(this->offset, &value, sizeof(int64_t));
        }

        //! オフセットを取得します。
        int64_t GetOffset() const NN_NOEXCEPT
        {
            // 8 バイトアライメント対策
            int64_t value;
            std::memcpy(&value, this->offset, sizeof(int64_t));
            return value;
        }
    };

    NN_STATIC_ASSERT(std::is_pod<AesCtrCounterExtendedStorage::Entry>::value);
    NN_STATIC_ASSERT(NN_ALIGNOF(AesCtrCounterExtendedStorage::Entry) == 4);
    NN_STATIC_ASSERT(sizeof(AesCtrCounterExtendedStorage::Entry) == 16);

private:

    /**
     * @brief   初期化をします。（テスト向け）
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   pKey            復号に使用する鍵の先頭アドレス
     * @param[in]   keySize         復号に使用する鍵のサイズ
     * @param[in]   secureValue     復号に使用するセキュア数値
     * @param[in]   dataStorage     暗号化したデータを格納したストレージ
     * @param[in]   tableStorage    カウンタのテーブルを格納したストレージ
     *
     * @return  関数の実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - pKey != nullptr
     *      - keySize == KeySize
     *      - 未初期化
     */
    Result Initialize(
               IAllocator* pAllocator,
               const void* pKey,
               size_t keySize,
               uint32_t secureValue,
               fs::SubStorage dataStorage,
               fs::SubStorage tableStorage
           ) NN_NOEXCEPT;

private:
    BucketTree m_Table;                          //!< カウンタのテーブル
    fs::SubStorage m_DataStorage;                //!< 暗号化したデータを格納したストレージ
    char m_Key[KeySize];                         //!< 復号に使用する鍵
    uint32_t m_SecureValue;                      //!< 復号に使用するセキュア数値
    int64_t m_CounterOffset;                     //!< 復号に使用するカウンタのオフセット
    std::unique_ptr<IDecryptor> m_pDecryptor;    //!< AesCtr の復号に使う関数

    friend class AesCtrCounterExtendedStorageTest;
    friend class utilTool::AesCtrCounterExtendedStorageChecker;
};

}}
