﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/dbm/fs_FileObjectTemplate.h>

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
FileObjectTemplate<
    TDirectoryName,
    TFileName
>::FileObjectTemplate() NN_NOEXCEPT
    : m_pFileSystem(nullptr),
      m_CurrBlockCount(0),
      m_CurrIndex(AllocationTable::IndexEnd),
      m_NextIndex(AllocationTable::IndexEnd)
{
    std::memset(&m_FileInfo, 0, sizeof(m_FileInfo));
    m_FileKey.parentDir = FileSystemObjectTempl::IndexNone;
    std::memset(&m_FileKey.name, 0, sizeof(m_FileKey.name));
}

/**
* @brief        ファイルの付加情報を取得します。
*
* @param[out]   outInfo     ファイルの付加情報
*
* @pre          oOutInfo が NULL ではない。
*
* @details      ファイルの付加情報を取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
void FileObjectTemplate<
         TDirectoryName,
         TFileName
     >::GetOptionalInfo(FileOptionalInfo* outInfo) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInfo);

    std::memcpy(outInfo, &m_FileInfo.fo, sizeof(FileOptionalInfo));
}

/**
* @brief        ファイルサイズを変更します。
*
* @param[in]    size                            変更後のファイルサイズ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   ファイルサイズが変更できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
* @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
* @retval       ResultOutOfRange                newSize が範囲外です。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       ResultAllocationTableFull       アロケーションテーブルに空きが無い。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
*
* @details      ファイルサイズを変更します。
*               あわせて領域の確保や開放処理をおこないます。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileObjectTemplate<
           TDirectoryName,
           TFileName
       >::Resize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    return m_pFileSystem->ResizeFile(&m_FileInfo, m_FileKey, size);
}

/**
* @brief        ファイルデータ本体へのイテレーションを行います。
*
* @param[out]   outIndex            実データ領域でのインデックス
* @param[out]   outFinished         イテレーションが完了したか
* @param[in]    blockOffset         イテレーションを開始する先頭ブロックからのブロック数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に取得できました。
* @retval       ResultOutOfRange    イテレータの値が不正です。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
*
* @details      ファイルの先頭から読む場合には blockOffset に 0 を指定します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileObjectTemplate<
           TDirectoryName,
           TFileName
       >::IterateBegin(
           uint32_t* outIndex,
           bool* outFinished,
           uint32_t blockOffset0
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(outFinished);

    // 先頭インデックスでのブロック情報を取得します。
    m_CurrIndex = m_FileInfo.fs.index;
    if( m_CurrIndex == AllocationTable::IndexEnd )
    {
        // イテレーションは完了しました。
        *outFinished = true;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(
        m_pFileSystem->GetNextAllocationBlock(&m_NextIndex, &m_CurrBlockCount, m_CurrIndex)
    );

    uint32_t blockOffset = blockOffset0;
    while( blockOffset >= m_CurrBlockCount )
    {
        if( m_CurrIndex == AllocationTable::IndexEnd )
        {
            // 範囲外です。
            return nn::fs::ResultOutOfRange();
        }

        m_CurrIndex = m_NextIndex;

        // 現在の連続領域外であれば次の連続領域を取得します。
        uint32_t nextIndex;
        uint32_t blockCount;
        NN_RESULT_DO(
            m_pFileSystem->GetNextAllocationBlock(&nextIndex, &blockCount, m_NextIndex)
        );
        m_NextIndex = nextIndex;
        blockOffset -= m_CurrBlockCount;
        m_CurrBlockCount = blockCount;
    }

    *outIndex = m_CurrIndex + blockOffset;
    *outFinished = false;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルデータの次のブロックを取得します。
*
* @param[out]   outIndex            実データ領域での次のブロックのインデックス
* @param[out]   outFinished         イテレーションが完了したか
* @param[in]    index               実データ領域でのブロックのインデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に取得できました。
* @retval       ResultOutOfRange    イテレータの値が不正です。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      ファイルデータの次のブロックを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileObjectTemplate<
           TDirectoryName,
           TFileName
       >::IterateNext(
           uint32_t* outIndex,
           bool* outFinished,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);

    if( m_CurrIndex == AllocationTable::IndexEnd )
    {
        // 範囲外です。
        return nn::fs::ResultOutOfRange();
    }

    // 連続領域の最後のインデックスであれば次の連続領域を取得します。
    if( index == m_CurrIndex + m_CurrBlockCount - 1 )
    {
        if( m_NextIndex == AllocationTable::IndexEnd )
        {
            // イテレーションは完了しました。
            *outFinished = true;
            NN_RESULT_SUCCESS;
        }

        // 次の連続領域を更新します。
        m_CurrIndex = m_NextIndex;
        *outIndex = m_NextIndex;

        uint32_t nextIndex;
        NN_RESULT_DO(
            m_pFileSystem->GetNextAllocationBlock(&nextIndex, &m_CurrBlockCount, m_NextIndex)
        );
        m_NextIndex = nextIndex;
    }
    else
    {
        *outIndex = index + 1;
    }

    *outFinished = false;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルデータの前のブロックを取得します。
*
* @param[out]   outIndex            実データ領域での前のブロックのインデックス
* @param[out]   outFinished         イテレーションが完了したか
* @param[in]    index               実データ領域でのブロックのインデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に取得できました。
* @retval       ResultOutOfRange    内部データが範囲外をさしています。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      ファイルデータの前のブロックを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result FileObjectTemplate<
           TDirectoryName,
           TFileName
       >::IteratePrevious(
           uint32_t* outIndex,
           bool* outFinished,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);

    if( m_CurrIndex == AllocationTable::IndexEnd )
    {
        // 範囲外です。
        return nn::fs::ResultOutOfRange();
    }

    // 連続領域の最初のインデックスであれば前の連続領域を取得します。
    if( index == m_CurrIndex )
    {
        uint32_t prevIndex;
        uint32_t blockCount;
        NN_RESULT_DO(
            m_pFileSystem->GetPrevAllocationBlock(&prevIndex, &blockCount, m_CurrIndex)
        );
        m_NextIndex = m_CurrIndex;
        m_CurrIndex = prevIndex;
        m_CurrBlockCount = blockCount;
        *outIndex = m_CurrIndex;
    }
    else
    {
        *outIndex = index - 1;
    }

    // イテレーションが完了したか
    *outFinished = (m_CurrIndex == AllocationTable::IndexEnd);

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルオブジェクトを初期化します。
*
* @param[in]    pFileSystem         ファイルシステム
* @param[in]    key                 ファイルキー
*
* @pre          pFileSystem が NULL ではない。
*
* @details      ファイルオブジェクトを初期化します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
void FileObjectTemplate<
         TDirectoryName,
         TFileName
     >::Initialize(
             FileSystemObjectTempl* pFileSystem,
             const FileKey& key
         ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFileSystem);

    m_pFileSystem = pFileSystem;
    m_FileKey = key;
}

}}}
