﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
DirectoryObjectTemplate<
    TDirectoryName,
    TFileName
>::DirectoryObjectTemplate() NN_NOEXCEPT
: m_pFileSystem(nullptr)
{
    std::memset(&m_DirectoryInfo, 0, sizeof(m_DirectoryInfo));
    m_DirectoryKey.parentDir = FileSystemObjectTempl::IndexNone;
    std::memset(&m_DirectoryKey.name, 0, sizeof(m_DirectoryKey.name));
    std::memset(&m_Iterator, 0, sizeof(m_Iterator));
}

/**
* @brief        ディレクトリに格納されているファイル数を取得します。
*
* @param[out]   outFileCounts ファイル数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にファイルが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      ディレクトリに格納されているファイル数を取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result DirectoryObjectTemplate<
           TDirectoryName,
           TFileName
       >::CountFile(
           uint32_t* outFileCounts
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);

    uint32_t counts = 0;

    // イテレータを初期化します。
    typename FileSystemObjectTempl::FindIndex fi;
    NN_RESULT_DO(m_pFileSystem->FindOpenWithKey(&fi, m_DirectoryKey));

    // 子ファイル数を計測します。
    FileName name;
    int64_t sizeFile;
    bool isFinished;
    for( ; ; )
    {
        NN_RESULT_DO(m_pFileSystem->FindNextFile(&sizeFile, &name, &isFinished, &fi));
        if( isFinished )
        {
            // ファイル数の計測は完了しました。
            break;
        }
        counts++;
    }
    *outFileCounts = counts;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ディレクトリに格納されているディレクトリ数を取得します。
*
* @param[out]   outDirectoryName ディレクトリ数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にディレクトリ数が取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      ディレクトリに格納されているディレクトリ数を取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result DirectoryObjectTemplate<
           TDirectoryName,
           TFileName
       >::CountDirectory(
           uint32_t* outDirectoryCounts
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);

    uint32_t counts = 0;

    // イテレータを初期化します。
    typename FileSystemObjectTempl::FindIndex fi;
    NN_RESULT_DO(m_pFileSystem->FindOpenWithKey(&fi, m_DirectoryKey));

    // 子ディレクトリ数を計測します。
    DirectoryName name;
    bool isFinished;
    for( ; ; )
    {
        Result result = m_pFileSystem->FindNextDirectory(&name, &isFinished, &fi);
        if( isFinished )
        {
            // ディレクトリ数の計測は完了しました。
            break;
        }
        NN_RESULT_DO(result);
        counts++;
    }

    *outDirectoryCounts = counts;

    NN_RESULT_SUCCESS;
}

/**
* @brief        子ファイル、ディレクトリに対するイテレーションを開始します。
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に処理が終了しました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre         マウントしている。
*
* @details     子ファイル、ディレクトリに対するイテレーションを開始します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result DirectoryObjectTemplate<
           TDirectoryName,
           TFileName
       >::FindOpen() NN_NOEXCEPT
{
    // イテレータを初期化します。
    return m_pFileSystem->FindOpenWithKey(&m_Iterator, m_DirectoryKey);
}

/**
* @brief        子ディレクトリに対するイテレーションを行います。
*
* @param[out]   outDirectoryName        ディレクトリ名
* @param[out]   outFinished             イテレーションが終了したか
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に処理が終了しました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre         マウントしている。
*
* @details      子ディレクトリに対するイテレーションを行います。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result DirectoryObjectTemplate<
           TDirectoryName,
           TFileName
       >::FindNextDirectory(
           TDirectoryName* outDirectoryName,
           bool* outFinished
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    return m_pFileSystem->FindNextDirectory(outDirectoryName, outFinished, &m_Iterator);
}

/**
* @brief        子ファイルに対するイテレーションを行います。
*
* @param[out]   outFileSize ファイルサイズ
* @param[out]   outFileName ファイル名
* @param[out]   outFinished イテレーションが終了したか
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にファイルが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      子ファイルに対するイテレーションを行います。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result DirectoryObjectTemplate<
           TDirectoryName,
           TFileName
       >::FindNextFile(
           int64_t* outFileSize,
           TFileName* outFileName,
           bool* outFinished
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    return m_pFileSystem->FindNextFile(
               outFileSize,
               outFileName,
               outFinished,
               &m_Iterator
           );
}

/**
* @brief        エントリー更新の通知をします。
*
* @param[out]   outFinished             イテレーションが終了したか。
* @param[in]    index                   エントリー ID
* @param[in]    isFile                  ファイルなら true, ディレクトリなら false
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にエントリーが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             iter が範囲外です。
* @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      エントリー更新の通知をします。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
Result DirectoryObjectTemplate<
           TDirectoryName,
           TFileName
       >::Notify(
           bool* outFinished,
           int64_t id,
           bool isFile) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    return m_pFileSystem->Notify(
               outFinished,
               &m_Iterator,
               id,
               isFile
           );
}

/**
* @brief        ディレクトリオブジェクトを初期化します。
*
* @param[in]    pFileSystem         ファイルシステム
* @param[in]    directoryKey        ディレクトリキー
*
* @pre          pFileSystem が NULL ではない。
*
* @details      ディレクトリオブジェクトを初期化します。
*/
template <
    typename TDirectoryName,
    typename TFileName
>
void DirectoryObjectTemplate<
         TDirectoryName,
         TFileName
     >::Initialize(
             FileSystemObjectTempl* pFileSystem,
             typename FileSystemObjectTempl::DirectoryKey& directoryKey
         ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFileSystem);

    m_pFileSystem = pFileSystem;
    m_DirectoryKey = directoryKey;
}

}}}

