﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/fs/detail/fs_MemoryManagementPrivate.h>
#include <nn/fssystem/dbm/fs_AllocationTableStorage.h>

namespace nn { namespace fssystem {
    class IBufferManager;
}}

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief    キャッシュ付きのアロケーションテーブルストレージです。
*/
class BufferedAllocationTableStorage : public AllocationTableStorage
{
    NN_DISALLOW_COPY(BufferedAllocationTableStorage);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    BufferedAllocationTableStorage() NN_NOEXCEPT;

    /**
    * @brief        デストラクタです。
    *
    * @details      デストラクタです。
    */
    virtual ~BufferedAllocationTableStorage() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージをマウントします。
    *
    * @param[in]    pAllocationTable    アロケーションテーブル
    * @param[in]    countManageBlock    管理用ブロックの数
    * @param[in]    sizeBlock           ブロックのサイズ
    * @param[in]    storage             実データ用ストレージ
    * @param[in]    pBufferManager      バッファマネージャ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       処理が正常に終了しました。
    * @retval       上記以外            アロケーションテーブルからのデータ読み込みに失敗しました。
    *
    * @pre          pAllocationTable != nullptr
    * @pre          pBufferManager != nullptr
    * @pre          offsetStorage >= 0
    * @pre          sizeBlock >= MinBlockSize
    *
    * @details      ストレージをマウントします。
    */
    Result InitializeBuffered(
               AllocationTable* pAllocationTable,
               uint32_t countManageBlock,
               uint32_t sizeBlock,
               fs::SubStorage storage,
               IBufferManager* pBufferManager
           ) NN_NOEXCEPT;

    /**
    * @brief        ストレージをアンマウントします。
    *
    * @details      ストレージをアンマウントします。
    */
    void FinalizeBuffered() NN_NOEXCEPT;

    /**
    * @brief        実ストレージの offset から size バイト読み込み buffer にコピーします。
    *
    * @param[in]    offset              オフセット
    * @param[out]   outBuffer           読み込み先バッファ
    * @param[in]    size                読み込むサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   読み込みが終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             offset が範囲外です。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          offset >= 0
    *
    * @details      実ストレージの offset から size バイト読み込み buffer にコピーします。
    */
    virtual Result Read(
                       int64_t offset,
                       void* outBuffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実ストレージの offset 以降に buffer を size バイト分コピーします。
    *
    * @param[in]    offset              オフセット
    * @param[in]    srcBuffer           書き込み元バッファ
    * @param[in]    size                書き込むサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にコピーしました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             offset が範囲外です。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          offset >= 0
    *
    * @details      実ストレージの offset 以降に buffer を size バイト分コピーします。
    */
    virtual Result Write(
                       int64_t offset,
                       const void* srcBuffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        バッファリングを有効化します。
    *
    * @param[in]    cacheCount   キャッシュ数
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - 0 < cacheCount
    *
    * @details      バッファリングを有効化します。
    */
    Result EnableCacheBuffer(int cacheCount) const NN_NOEXCEPT;


    /**
    * @brief        バッファリングを無効化します。
    *
    * @details      バッファリングを無効化します。
    */
    void DisableCacheBuffer() const NN_NOEXCEPT;

private:
    void InvalidateCache(int64_t offset, int64_t size) NN_NOEXCEPT;

private:
    mutable decltype(nn::fs::detail::MakeUnique<int64_t[]>(0)) m_CacheOffsets;
    mutable int m_CacheCount;
    mutable int m_NextCacheIndex;
    mutable char* m_Buffer;
    mutable size_t m_BufferSize;
    mutable int m_EnableCount;
    IBufferManager* m_pBufferManager;
};

}}}

