﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/dbm/fs_AllocationTable.h>

class AllocationTableStorageTest;

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief    アロケーションテーブルに対してストレージインターフェイスを用いてアクセス機能を提供します。
*
* @details  アロケーションテーブルに対してストレージインターフェイスを用いてアクセス機能を提供します。
*
* アロケーションテーブルの実ストレージとこのクラスが提供するストレージを
* 混同しないよう注意してください。
* ストレージ全体をみると、ヘッダ部分と実データ部分があります。
* アロケーションテーブルはヘッダ部分のみに読み書きを行います。
* このクラスはアロケーションテーブルから未使用ストレージ情報を取得し、
* 実データ部分において読み書きを行います。
* ストレージの管理は固定長ブロック単位で行います。
* 1 ブロックのサイズはマウント時に指定します。
*/
class AllocationTableStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(AllocationTableStorage);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    AllocationTableStorage() NN_NOEXCEPT;

    /**
    * @brief        ストレージをマウントします。
    *
    * @param[in]    pAllocationTable    アロケーションテーブル
    * @param[in]    countManageBlock    管理用ブロックの数
    * @param[in]    sizeBlock           ブロックのサイズ
    * @param[in]    storage             実データ用ストレージ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       処理が正常に終了しました。
    * @retval       上記以外            アロケーションテーブルからのデータ読み込みに失敗しました。
    *
    * @pre          pAllocationTable != nullptr
    * @pre          offsetStorage >= 0
    * @pre          sizeBlock >= MinBlockSize
    *
    * @details      ストレージをマウントします。
    */
    Result Initialize(
                AllocationTable* pAllocationTable,
                uint32_t countManageBlock,
                uint32_t sizeBlock,
                fs::SubStorage storage
           ) NN_NOEXCEPT;

    /**
    * @brief        ストレージをアンマウントします。
    *
    * @details      ストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ストレージの総バイトサイズを取得します。
    *
    * @param[out]   outValue        ストレージの総バイト数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          outValue != nullptr
    *
    * @details      ストレージの総バイトサイズを取得します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        引数の個数分のブロックのサイズを取得する。
    *
    * @param[in]    blockCount      ブロック数
    *
    * @return       引数の個数分のブロックのサイズを返します。
    *
    * @pre          マウントしている。
    *
    * @details      引数の個数分のブロックのサイズを取得する。
    */
    int64_t GetBlockSize(uint32_t blockCount) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS_EQUAL(MinBlockSize, 1U << m_BlockSizeShift);
        return static_cast<int64_t>(blockCount) << m_BlockSizeShift;
    }

    /**
    * @brief        実ストレージの offset から size バイト読み込み buffer にコピーします。
    *
    * @param[in]    offset              オフセット
    * @param[out]   outBuffer           読み込み先バッファ
    * @param[in]    size                読み込むサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   読み込みが終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             offset が範囲外です。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          offset >= 0
    *
    * @details      実ストレージの offset から size バイト読み込み buffer にコピーします。
    */
    virtual Result Read(
                       int64_t offset,
                       void* outBuffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実ストレージの offset 以降に buffer を size バイト分コピーします。
    *
    * @param[in]    offset              オフセット
    * @param[in]    srcBuffer           書き込み元バッファ
    * @param[in]    size                書き込むサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にコピーしました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             offset が範囲外です。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          offset >= 0
    *
    * @details      実ストレージの offset 以降に buffer を size バイト分コピーします。
    */
    virtual Result Write(
                       int64_t offset,
                       const void* srcBuffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージに対してフラッシュ要求を出します。
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       処理が正常に終了しました。
    * @retval       上記以外            ストレージでフラッシュに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ストレージに対してフラッシュ要求を出します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Storage.Flush();
    }

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    using IStorage::OperateRange;

    /**
    * @brief        割り当て領域を増やします。
    *
    * @param[in]    counts                      確保するブロックの個数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess               正常に割り当て領域を増やしました。
    * @retval       ResultAllocationTableFull   アロケーションテーブルに空きがありません。
    * @retval       上記以外                    アロケーションテーブルでの読み書きに失敗しました。
    *
    * @details      割り当て領域を増やします。
    */
    Result AllocateBlock(uint32_t counts) NN_NOEXCEPT;

protected:
    /**
    * @brief        offset 位置を含むブロックを取得します。
    *
    * @param[out]   outIndex        ブロックインデックス
    * @param[out]   outBlockCount   該当ブロック数
    * @param[out]   outOffset       ブロック内での位置
    * @param[in]    offset              先頭からの位置
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       offset 位置を含むブロックを取得しました。
    * @retval       ResultInvalidOffset offset 位置が範囲外でした。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @details      offset 位置を含むブロックを取得します。
    */
    Result SeekTo(
               uint32_t* outIndex,
               uint32_t* outBlockCount,
               int64_t* outOffset,
               int64_t offset
           ) const NN_NOEXCEPT;

    /**
    * @brief        シークセッションを先頭に戻します。
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       処理が正常に終了しました。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      シークセッションを先頭に戻します。
    */
    Result SeekToTop() const NN_NOEXCEPT;

private:
    //! アロケートされていないブロックのインデックス
    static const uint32_t UnallocatedIndex = 0xFFFFFFFFU;
    //! 最小ブロックサイズ(バイト数)
    static const uint32_t MinBlockSize = 1 << 1;

private:
    AllocationTable* m_pAllocTable;             //! アロケーションテーブル
    fs::SubStorage m_Storage;                       //! 実際にデータをリードライトするストレージ
    uint32_t m_IndexTop;                        //! 先頭インデックス
    uint32_t m_BlockSizeShift;                  //! ブロック数を求めるためのシフト数
    mutable uint32_t m_SessionCountSeek;        //! (シーク位置のセッション)シークブロック数
    mutable uint32_t m_SessionIndex;            //! (シーク位置のセッション)現在のindex
    mutable uint32_t m_SessionCountBlock;       //! (シーク位置のセッション)連続ブロック数
    mutable uint32_t m_SessionNext;             //! (シーク位置のセッション)次のブロック

private:
    // テスト用に private を公開します。
    friend class ::AllocationTableStorageTest;
};

}}}

