﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SubStorage.h>

class AllocationTableTestSuite;

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief データ可変長 DB クラスです。
*
* @details データ可変長 DB クラスです。
*
* 可変長データ領域を管理するための機構です。
* 固定長ブロックを単位とし、指定したサイズに応じて複数のブロックを確保します。
*
* AllocationTable はストレージ上のブロックを管理するための機構です。
* Format でブロックリストを作成し、Initialize でマウントします。
* Allocate するごとに指定した数だけストレージからブロックを確保します。
*
* ○ セクタとインデックス
*
* AllocationTable の中身は 8 バイトのブロックの配列です。
* Format 時にブロック数を blockCount 個と指定したとき、管理領域を含め
*   全部で SectorReservedCount + blockCount 個のブロックがAllocationTable の対象となります。
* 管理領域を除いていない 1 つめのブロックを 0 番目としたときのブロックの番号をセクタと呼びます
* セクタの最大値は blockCount です。
* 管理領域を除いた 1 つめのブロックを 0 番目としたときのブロックの番号をインデックスと呼びます
* インデックスの最大値は blockCount - SectorReservedCount です
*
* ○ ブロックリストと連続ブロックリストとフリーリスト
*
* AllocationTable 上のすべてのブロックのことをまとめてブロックリストと呼びます。
* 各ブロックには、次のブロックと前のブロックのセクタだけが入っています。
* このブロックによって作成されたリンクリストが連続ブロックリストです。
* 連続ブロックリストの1つ目のブロックを先頭ブロックと呼びます。
* 連続ブロックリストの先頭ブロックの前ブロック情報と、
* 最後のブロックの次ブロック情報は IndexEnd です。
* フリーリストはまだ確保していないブロックで構成された連続ブロックリストです。
* フリーリストの先頭ブロックのセクタは 0 です
*/
class AllocationTable
{
    NN_DISALLOW_COPY(AllocationTable);

public:
    //! ReadNext、ReadPrevious を用いたイテレーションの終端を示します。
    static const uint32_t IndexEnd = 0x80000000U;

public:
    /**
    * @brief        指定したブロック数のアロケーションテーブルに必要なデータサイズを取得します。
    *
    * @param[in]    blockCount  ブロック数
    *
    * @return       指定したブロック数のアロケーションテーブルに必要なデータサイズ。
    *
    * @details      指定したブロック数のアロケーションテーブルに必要なデータサイズを取得します。
    */
    static int64_t QuerySize(uint32_t blockCount) NN_NOEXCEPT;

    /**
    * @brief        アロケーションテーブルをフォーマットします。
    *
    * @param[in]    storage         アロケーションテーブルに割り当てる領域
    * @param[in]    blockCount      ブロック数(管理領域は含まない)
    *
    * @return       関数の処理結果を返します。
    * @return       ResultSuccess   正常に初期化できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @pre          blockCount >= 1
    *
    * @details      アロケーションテーブルをフォーマットします。
    *               初めてマウントする前に 1 度だけ呼び出してください。
    */
    static Result Format(
                      fs::SubStorage storage,
                      uint32_t blockCount
                  ) NN_NOEXCEPT;

    /**
    * @brief        アロケーションテーブル用ストレージを拡張します。
    *
    * @param[in]    storage         アロケーションテーブルに割り当てる領域
    * @param[in]    oldBlockCount   今までのブロック数(要素数)
    * @param[in]    newBlockCount   新しいブロック数(要素数)
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に初期化できました。
    * @retval       上記以外        ストレージでの読み書きに失敗しました。
    *
    * @pre          アンマウントしてから実行する必要があります。
    * @pre          newBlockCount > oldBlockCount
    *
    * @details      アロケーションテーブル用ストレージを拡張します。
    */
    static Result Expand(
                      fs::SubStorage storage,
                      uint32_t oldBlockCount,
                      uint32_t newBlockCount
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    AllocationTable() NN_NOEXCEPT;

    /**
    * @brief        ストレージをマウントします。
    *
    * @param[in]    storage         アロケーションテーブルに割り当てるストレージ
    * @param[in]    blockCount      ブロック数
    *
    * @details      ストレージをマウントします。
    */
    void Initialize(
             fs::SubStorage storage,
             uint32_t blockCount
         ) NN_NOEXCEPT;

    /**
    * @brief        ストレージをアンマウントします。
    *
    * @details      ストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        指定したサイズ分のデータ領域をテーブルから確保します。
    *
    * @param[out]   outIndex                    確保したデータ領域の先頭インデックス
    * @param[in]    blockCount                  確保するブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess               正常にデータを読み込みました。
    * @retval       ResultAllocationTableFull   フリーリストに空きが無い。
    * @retval       上記以外                    ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIndex != nullptr
    * @pre          blockCount > 0
    *
    * @details      指定したサイズ分のデータ領域をテーブルから確保します。
    */
    Result Allocate(uint32_t* outIndex, uint32_t blockCount) NN_NOEXCEPT;

    /**
    * @brief        指定したデータ領域を解放します。
    *
    * @param[in]    index           解放対象ブロックリストに含まれるインデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に解放できました。
    * @retval       上記以外        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      指定したデータ領域を解放します。
    */
    Result Free(uint32_t index) NN_NOEXCEPT;

    /**
    * @brief        指定したデータ領域の次のデータ領域を取得します。
    *
    * @param[out]   outNextIndex        次の連続ブロックチェインの先頭インデックス
    *                                   最後の連続ブロックチェインであれば INDEX_EMPTY
    * @param[out]   outBlockCount       現在の連続ブロックチェインのデータ領域数
    * @param[in]    index               現在の連続ブロックチェインの先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       処理が終了しました。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outNextIndex != nullptr
    * @pre          outBlockCount != nullptr
    * @pre          index が指すセクタが空ではない。
    *
    * @details      連続ブロックチェイン単位でイテレーションします。
    */
    Result ReadNext(
               uint32_t* outNextIndex,
               uint32_t* outBlockCount,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したデータ領域の前データ領域を取得します。
    *
    * @param[out]   outPrevIndex    前の連続ブロックチェインの先頭インデックス
    *                               最初の連続ブロックチェインであれば INDEX_EMPTY
    * @param[out]   outBlockCount   現在の連続ブロックチェインのデータ領域数
    * @param[in]    index           現在の連続ブロックチェインの先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   処理が終了しました。
    * @retval       上記以外        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outPrevIndex != nullptr
    * @pre          outBlockCount != nullptr
    * @pre          index が指すセクタが空ではない。
    *
    * @details      連続ブロックチェイン単位でイテレーションします。
    */
    Result ReadPrevious(
               uint32_t* outPrevIndex,
               uint32_t* outBlockCount,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        総ブロック数を取得します。
    *
    * @param[out]   outBlockCount   総ブロック数
    * @param[in]    index           データ領域の先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に検出できました。
    * @retval       上記以外        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outBlockCount != nullptr
    *
    * @details      総ブロック数を取得します。
    */
    Result CalcTotalBlockCount(
               uint32_t* outBlockCount,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        最後の領域での先頭インデックスを探索します。
    *
    * @param[out]   outTailIndex    最後の領域での先頭インデックス
    * @param[in]    index           先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に検出できました。
    * @retval       上記以外        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outTailIndex != nullptr
    *
    * @details      最後の領域での先頭インデックスを探索します。
    */
    Result LookupTailParentIndex(
               uint32_t* outTailIndex,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        指定した 2 つのデータ領域を連結します。
    *
    * @param[in]    indexBefore         連結後、前側に来るデータ領域
    * @param[in]    indexAfter          連結後、後ろ側に来るデータ領域
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常にデータを読み込みました。
    * @retval       上記以外            ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      指定した 2 つのデータ領域を連結します。
    */
    Result Concat(
               uint32_t indexBefore,
               uint32_t indexAfter
           ) NN_NOEXCEPT;

    /**
    * @brief        データ領域を指定した位置で 2 分割します。
    *
    * @param[out]   outIndexAfter   分割後のデータ領域の先頭インデックス
    * @param[in]    index           分割するデータ領域の先頭インデックス
    * @param[in]    blockCount      分割後に前半部分になるブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess     正常に分割できました。
    * @retval       ResultOutOfRange  blockCount が 0 以下または大きすぎます。
    * @retval       上記以外          ストレージでの読み書きに失敗しました。
    *
    * @pre          outIndexAfter != nullptr
    *
    * @details      データ領域を指定した位置で 2 分割します。
    */
    Result Split(
               uint32_t* outIndexAfter,
               uint32_t index,
               uint32_t blockCount
           ) NN_NOEXCEPT;

    /**
    * @brief        フリーリストの先頭インデックスを取得します。
    *
    * @param[out]   outIndex        フリーリストの先頭インデックス
    *                               フリーリストが空であれば IndexEnd
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIndex != nullptr
    *
    * @details      フリーリストの先頭インデックスを取得します。
    */
    Result ReadFreeListHead(uint32_t* outIndex) NN_NOEXCEPT;

    /**
    * @brief        テーブルの空きブロック数を取得します。
    *
    * @param[out]   outCount        テーブルの空きブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          outCount != nullptr
    *
    * @details      テーブルの空きブロック数を取得します。
    */
    Result CalcFreeListLength(uint32_t* outCount) NN_NOEXCEPT;

    /**
    * @brief        テーブルのブロック数を取得します。
    *
    * @return       テーブルのブロック数。
    *
    * @details      テーブルのブロック数を取得します。
    */
    uint32_t GetBlockCount() const NN_NOEXCEPT
    {
        return m_BlockCount;
    }

private:
    //! アロケーションテーブルのエントリーです。
    struct TableElement
    {
        //! TableElement.next の最上位ビットが立っていればブロックが連続していることを示します。
        static const uint32_t MaskIsSectorChainParent = 0x80000000U;

        //! TableElement.prev の最上位ビットが立っていれば連続領域に含まれていることを示します。
        //! 連続領域に含まれている場合は下位31ビットに連続領域の先頭セクタが設定されます。
        static const uint32_t MaskIsSectorChainChild = 0x80000000U;

        //! 前の先頭エントリーのセクタ
        uint32_t sectorPrevious;

        //! 次の先頭エントリーのセクタ
        uint32_t sectorNext;

        /**
        * @brief        指定したブロックがブロックリストの先頭かどうかを取得します。
        *
        * @return       ブロックリストの先頭かどうか。
        *
        * @details      指定したブロックがブロックリストの先頭かどうかを取得します。
        */
        bool IsListHead() const NN_NOEXCEPT
        {
            return (this->sectorPrevious == SectorMaskPreviousIsListHead);
        }

        /**
        * @brief        指定したブロックをブロックリストの先頭として初期化します。
        *
        * @details      指定したブロックをブロックリストの先頭として初期化します。
        *               作成するアロケーションテーブルのブロック数が 1 以上
        *               (セクタ数が 2 以上) であることを前提としています。
        */
        void SetListFreeHeader() NN_NOEXCEPT
        {
            this->sectorPrevious = SectorFreeListHeader;
            this->sectorNext = SectorListHead;
        }

        /**
        * @brief        指定したブロックをブロックリストの先頭として初期化します。
        *
        * @param[in]    nextSector      次の連続領域ブロックへのセクタ
        * @param[in]    hasChainParent   連続領域の親かどうか
        *
        * @details      指定したブロックをブロックリストの先頭として初期化します。
        */
        void SetListHead(
                 const uint32_t nextSector,
                 const bool hasChainParent
             ) NN_NOEXCEPT
        {
            this->sectorPrevious = SectorMaskPreviousIsListHead;
            this->sectorNext = nextSector;
            if( hasChainParent )
            {
                this->sectorNext |= MaskIsSectorChainParent;
            }
        }

        /**
        * @brief        指定したブロックが最後の連続ブロックかどうかを取得します。
        *
        * @return       最後の連続ブロックの先頭かどうか。
        *
        * @details      指定したブロックが最後の連続ブロックかどうかを取得します。
        */
        bool IsFreeListEmpty() const NN_NOEXCEPT
        {
            return (this->sectorNext == SectorFreeListHeader);
        }

        /**
        * @brief        指定したブロックが最後の連続ブロックの先頭かどうかを取得します。
        *
        * @return       最後の連続ブロックの先頭かどうか。
        *
        * @details      指定したブロックが最後の連続ブロックの先頭かどうかを取得します。
        */
        bool IsTailParent() const NN_NOEXCEPT
        {
            return ((this->sectorNext & SectorMaskValid) == SectorFreeListHeader);
        }

        /**
        * @brief        指定したブロックが連続ブロックの先頭かどうかを取得します。
        *
        * @return       連続ブロックの先頭かどうか。
        *
        * @details      指定したブロックを連続ブロックの先頭かどうかを取得します。
        */
        bool IsChainParent() const NN_NOEXCEPT
        {
            return (this->sectorNext & MaskIsSectorChainParent) != 0;
        }

        /**
        * @brief        指定したブロックを連続ブロックの先頭として初期化します。
        *
        * @param[in]    previousSector  手前の連続ブロックへの先頭セクタ
        * @param[in]    nextSector      次の連続ブロックへの先頭セクタ
        * @param[in]    hasChainParent  子ブロックが存在するかどうか
        *
        * @details      指定したブロックを連続ブロックの先頭として初期化します。
        */
        void SetChainParent(
                 const uint32_t previousSector,
                 const uint32_t nextSector,
                 const bool hasChainParent
             ) NN_NOEXCEPT
        {
            this->sectorPrevious = previousSector;
            this->sectorNext = nextSector;
            if( hasChainParent )
            {
                this->sectorNext |= MaskIsSectorChainParent;
            }
        }

        /**
        * @brief        指定したブロックが単一ブロックの先頭かどうかを取得します。
        *
        * @return       単一ブロックの先頭かどうか。
        *
        * @details      指定したブロックが単一ブロックの先頭かどうかを取得します。
        */
        bool IsSingleParent() const NN_NOEXCEPT
        {
            if( (this->sectorNext & MaskIsSectorChainParent) != 0 )
            {
                return false;
            }
            if( (this->sectorPrevious & MaskIsSectorChainChild) != 0 )
            {
                return false;
            }
            return true;
        }

        /**
        * @brief        指定したブロックが連続ブロックに含まれる子ブロックかどうかを取得します。
        *
        * @return       連続ブロックに含まれる子ブロックかどうか。
        *
        * @details      指定したブロックが連続ブロックに含まれる子ブロックかどうかを取得します。
        */
        bool IsChainChild() const NN_NOEXCEPT
        {
            return ((this->sectorPrevious & MaskIsSectorChainChild) != 0);
        }

        /**
        * @brief        指定したブロックを連続ブロックの子として初期化します。
        *
        * @param[in]    parentSector    連続ブロックの先頭セクタ
        * @param[in]    tailSector      連続ブロックの末尾セクタ
        *
        * @details      指定したブロックを連続ブロックの子として初期化します。
        */
        void SetChainChild(
                 uint32_t parentSector,
                 uint32_t tailSector
             ) NN_NOEXCEPT
        {
            this->sectorPrevious = parentSector | MaskIsSectorChainChild;
            this->sectorNext = tailSector;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<TableElement>::value);

private:
    //! アロケーションテーブルの先頭はフリーリストの先頭として扱います。
    static const uint32_t SectorFreeListHeader = 0x00000000U;

    //! フリーリストの先頭の次は確保した連続ブロックリストの先頭として扱います。
    static const uint32_t SectorListHead = 0x00000001U;

    //! 最上位ビット以外をアドレッシング可能な領域として使用します。
    static const uint32_t SectorMaskValid = 0x7FFFFFFFU;

    //! TableElement.previous がブロックリストの先頭であることを示します。
    static const uint32_t SectorMaskPreviousIsListHead = 0x80000000U;

    //! アロケーションテーブルの予約領域(現在はフリーリスト用に 1 確保)
    static const uint32_t SectorReservedCount = 1;

    //! 無効なセクタを示します。
    static const uint32_t SectorEmpty = 0x80000000U;

private:
    /**
    * @brief        インデックスをセクタ番号に変換します。
    *
    * @param[in]    index   インデックス
    *
    * @return       セクタ番号。
    *
    * @pre          インデックスが範囲外を指していない。
    *
    * @details      インデックスをセクタ番号に変換します。
    */
    uint32_t ConvertIndexToSector(uint32_t index) const NN_NOEXCEPT;

    /**
    * @brief        セクタ番号をインデックスに変換します。
    *
    * @param[in]    sector  セクタ番号
    *
    * @return       インデックス。
    *
    * @pre          セクタ番号が範囲外を指していない。
    *
    * @details      セクタ番号をインデックスに変換します。
    */
    uint32_t ConvertSectorToIndex(uint32_t sector) const NN_NOEXCEPT;

    /**
    * @brief        ブロック情報を読み込む際の個数を取得します。
    *
    * @param[in]    sector  セクタ番号
    *
    * @return       読み込むべきブロックの個数。
    *
    * @details      読み込むブロックが連続ブロックの場合、次ブロックの情報が
    *               必要になるため、基本的に 2 個読み込む。
    *               最終ブロックの場合は次ブロックが無いため 1 個。
    */
    inline uint32_t GetReadBlockCount(uint32_t sector) const NN_NOEXCEPT;

    /**
    * @brief        指定したセクタ以降のブロック複数個ストレージから読み込みます。
    *
    * @param[out]   outDst          読み込んだブロック
    * @param[in]    sector          セクタ
    * @param[in]    blockCount      ブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常にデータを読み込みました。
    * @retval       上記以外        ストレージからのデータ読み込みに失敗しました
    *
    * @pre          マウントしている。
    * @pre          outDst != nullptr
    * @pre          blockCount が 0 ではない。
    *
    * @details      指定したセクタ以降のブロック複数個ストレージから読み込みます。
    */
    Result ReadBlock(
               TableElement* outDst,
               uint32_t sector,
               uint32_t blockCount
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したセクタに書き込みます。
    *
    * @param[in]    pSrcElement     書き込むブロック
    * @param[in]    sector          セクタ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常にデータを書き込めました。
    * @retval       上記以外        ストレージへのデータ書きこみに失敗しました
    *
    * @pre          マウントしている。
    * @pre          pSrcElement != nullptr
    *
    * @details      指定したセクタに書き込みます。
    */
    Result WriteBlock(
               const TableElement* pSrcElement,
               uint32_t sector
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したブロックの次セクタを更新します。
    *
    * @param[in]    sector          更新するセクタ
    * @param[in]    nextSector      次連続領域の先頭セクタ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常にデータを書き込みました。
    * @retval       上記以外        ストレージへのデータ書きこみに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      指定したブロックの次セクタを更新します。
    */
    Result UpdateNext(
               uint32_t sector,
               uint32_t nextSector
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したブロックの前セクタを更新します。
    *
    * @param[in]    sector          更新するセクタ
    * @param[in]    sectorPrevious  前連続領域の先頭セクタ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に処理が終了しました。
    * @retval       上記以外        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          sector != SectorFreeListHeader
    *
    * @details      指定したブロックの前セクタを更新します。
    */
    Result UpdatePrevious(
               uint32_t sector,
               uint32_t previousSector
           ) NN_NOEXCEPT;

    /**
    * @brief        連続領域内のリンク情報を更新します。
    *
    * @param[in]    headSector      ブロックリストの先頭セクタ
    * @param[in]    parentSector    連続ブロックの先頭セクタ
    * @param[in]    lastSector      連続ブロックの末尾セクタ
    * @param[in]    previousSector  前の連続ブロックの先頭セクタ
    * @param[in]    nextSector      次の連続ブロックの先頭セクタ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   リンク情報を正常に更新しました。
    * @retval       上記以外        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      parentSector から始まる連続ブロックを
    *               headSector から始まる連続ブロックにつなげます。
    */
    Result UpdateChain(
               uint32_t headSector,
               uint32_t parentSector,
               uint32_t lastSector,
               uint32_t previousSector,
               uint32_t nextSector
           ) NN_NOEXCEPT;

    /**
    * @brief        フリーリストを取得します。
    *
    * @param[out]   outFreeList     フリーリスト
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   フリーリストが取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outFreeList != nullptr
    *
    * @details      フリーリストを取得します。
    */
    Result ReadFreeList(TableElement* outFreeList) NN_NOEXCEPT;

    /**
    * @brief        フリーリストから空いているデータ領域を取得します。
    *
    * @param[out]   outSector       確保した領域の先頭セクタ
    * @param[out]   outBlockCount   確保できたブロック数
    * @param[in]    pFreeListHeader フリーリスト
    * @param[in]    requiredCount   確保するブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outSector != nullptr
    * @pre          outBlockCount != nullptr
    * @pre          pFreeListHeader != nullptr
    *
    * @details      フリーリストから空いているデータ領域を取得します。
    */
    Result AllocateFreeSector(
               uint32_t* outSector,
               uint32_t* outBlockCount,
               TableElement* pFreeListHeader,
               uint32_t requiredCount
           ) NN_NOEXCEPT;

    /**
    * @brief        更新後のフリーリストを保存します。
    *
    * @param[in]    pFreeListHeader フリーリスト
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常にフリーリストを更新しました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          pFreeListHeader != nullptr
    *
    * @details      更新後のフリーリストを保存します。
    */
    Result UpdateFreeList(const TableElement* pFreeListHeader) NN_NOEXCEPT;

private:
    fs::SubStorage m_Table;                     //! アロケーションテーブル
    uint32_t m_BlockCount;                  //! アロケーションテーブルに割り当てる要素数

private:
    // テスト用に private を公開します。
    friend class AllocationTableTestSuite;
};

}}}

