﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace fssystem {

/*!
* @brief ファイルシステムバッファマネージャインターフェイスクラス
*/
class IBufferManager
{
public:
    //! バッファ属性を表す構造体です。
    class BufferAttribute
    {
    public:
        /*!
        * @brief       デフォルトの属性を作成します。
        */
        BufferAttribute() NN_NOEXCEPT
        : m_Level(0)
        {
        }

        /*!
        * @brief       指定のパラメータを持つ属性を作成します。
        *
        * @param[in]   level   バッファレベル
        */
        explicit BufferAttribute(int level) NN_NOEXCEPT
        : m_Level(level)
        {
        }

        /*!
        * @brief       バッファレベルを取得します
        */
        int GetLevel() const NN_NOEXCEPT
        {
            return m_Level;
        }

    private:
        int m_Level; //!< バッファレベルです
    };

    typedef int64_t CacheHandle; //!< キャッシュハンドルです。

public:
    static const int BufferLevelMin = 0; //!< バッファレベルの最小です。

public:
    /*!
    * @brief       デストラクタです。
    */
    virtual ~IBufferManager() NN_NOEXCEPT
    {
    }

    /*!
    * @brief       指定のサイズと属性を持つバッファを割り当てます。
    *
    * @param[in]   size    割り当てるサイズ
    * @param[in]   bufAttr 割り当てるバッファの属性
    *
    * @return      割り当てたメモリ範囲の先頭アドレスとサイズの組を返します。
    */
    const std::pair<uintptr_t, size_t> AllocateBuffer(
                                           size_t size,
                                           const BufferAttribute& bufAttr
                                       ) NN_NOEXCEPT
    {
        return DoAllocateBuffer(size, bufAttr);
    };

    /*!
    * @brief       指定のサイズとデフォルトの属性を持つバッファを割り当てます。
    *
    * @param[in]   size    割り当てるサイズ
    *
    * @return      割り当てたメモリ範囲の先頭アドレスとサイズの組を返します。
    */
    const std::pair<uintptr_t, size_t> AllocateBuffer(size_t size) NN_NOEXCEPT
    {
        return DoAllocateBuffer(size, BufferAttribute());
    }

    /*!
    * @brief       指定の範囲のバッファを解放します。
    *
    * @param[in]   address 解放するバッファの先頭アドレス
    * @param[in]   size    解放するバッファのサイズ
    */
    void DeallocateBuffer(uintptr_t address, size_t size) NN_NOEXCEPT
    {
        DoDeallocateBuffer(address, size);
    }

    /*!
    * @brief       指定のメモリ範囲をキャッシュとして登録します。
    *
    * @param[in]   address 登録するバッファの先頭アドレス
    * @param[in]   size    登録するバッファのサイズ
    * @param[in]   bufAttr 登録するバッファの属性
    *
    * @return      登録したキャッシュのハンドルを返します。
    *
    * @details     登録したキャッシュは自動的に無効化されることがあります。
    */
    CacheHandle RegisterCache(
                    uintptr_t address,
                    size_t size,
                    const BufferAttribute& bufAttr
                ) NN_NOEXCEPT
    {
        return DoRegisterCache(address, size, bufAttr);
    }

    /*!
    * @brief       指定のハンドルに紐付けられたキャッシュを取得します。
    *
    * @param[in]   handle  取得するキャッシュのハンドル
    *
    * @return      キャッシュのメモリ範囲の先頭アドレスとサイズの組を返します。
    *
    * @details     取得に成功したキャッシュの登録は解除されます。
    *
    *              無効化されたキャッシュのハンドルが指定された場合、
    *              キャッシュの取得に失敗し、返される先頭アドレスとサイズはいずれも 0 になります。
    */
    const std::pair<uintptr_t, size_t> AcquireCache(CacheHandle handle) NN_NOEXCEPT
    {
        return DoAcquireCache(handle);
    }

    /*!
    * @brief       バッファマネージャが管理するメモリ全体のサイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリ全体のサイズを返します。
    */
    size_t GetTotalSize() const NN_NOEXCEPT
    {
        return DoGetTotalSize();
    }

    /*!
    * @brief       バッファマネージャが管理するメモリのうち空き領域のサイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち空き領域のサイズを返します。
    */
    size_t GetFreeSize() const NN_NOEXCEPT
    {
        return DoGetFreeSize();
    }

    /*!
    * @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを返します。
    */
    size_t GetTotalAllocatableSize() const NN_NOEXCEPT
    {
        return DoGetTotalAllocatableSize();
    }

    /*!
    * @brief       バッファマネージャが管理するメモリのうち空き領域サイズのピークを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち空き領域サイズのピークを返します。
    */
    size_t GetFreeSizePeak() const NN_NOEXCEPT
    {
        return DoGetFreeSizePeak();
    }

    /*!
    * @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを返します。
    */
    size_t GetTotalAllocatableSizePeak() const NN_NOEXCEPT
    {
        return DoGetTotalAllocatableSizePeak();
    }

    /*!
    * @brief       バッファマネージャが管理するメモリの確保リトライ回数を取得します。
    *
    * @return      バッファマネージャが管理するメモリの確保リトライ回数を返します。
    */
    size_t GetRetriedCount() const NN_NOEXCEPT
    {
        return DoGetRetriedCount();
    }

    /*!
    * @brief       バッファマネージャが管理するメモリの確保リトライ回数/空き領域サイズのピークをクリアします。
    */
    void ClearPeak() NN_NOEXCEPT
    {
        DoClearPeak();
    }

protected:
    /*!
    * @brief       指定のサイズと属性を持つバッファを割り当てます。
    *
    * @param[in]   size    割り当てるサイズ
    * @param[in]   bufAttr 割り当てるバッファの属性
    *
    * @return      割り当てたメモリ範囲の先頭アドレスとサイズの組を返します。
    */
    virtual const std::pair<uintptr_t, size_t> DoAllocateBuffer(
                                                   size_t size,
                                                   const BufferAttribute& bufAttr
                                               ) NN_NOEXCEPT = 0;

    /*!
    * @brief       指定の範囲のバッファを解放します。
    *
    * @param[in]   address 解放するバッファの先頭アドレス
    * @param[in]   size    解放するバッファのサイズ
    */
    virtual void DoDeallocateBuffer(uintptr_t address, size_t size) NN_NOEXCEPT = 0;

    /*!
    * @brief       指定のメモリ範囲をキャッシュとして登録します。
    *
    * @param[in]   address         登録するバッファの先頭アドレス
    * @param[in]   size            登録するバッファのサイズ
    * @param[in]   bufAttr         登録するバッファの属性
    *
    * @return      登録したキャッシュのハンドルを返します。
    *
    * @details     登録したキャッシュは自動的に無効化されることがあります。
    */
    virtual CacheHandle DoRegisterCache(
                            uintptr_t address,
                            size_t size,
                            const BufferAttribute& bufAttr
                        ) NN_NOEXCEPT = 0;

    /*!
    * @brief       指定のハンドルに紐付けられたキャッシュを取得します。
    *
    * @param[in]   handle  取得するキャッシュのハンドル
    *
    * @return      キャッシュのメモリ範囲の先頭アドレスとサイズの組を返します。
    *
    * @details     取得に成功したキャッシュの登録は解除されます。
    *
    *              無効化されたキャッシュのハンドルが指定された場合、
    *              キャッシュの取得に失敗し、返される先頭アドレスとサイズはいずれも 0 になります。
    */
    virtual const std::pair<uintptr_t, size_t> DoAcquireCache(CacheHandle handle) NN_NOEXCEPT = 0;

    /*!
    * @brief       バッファマネージャが管理するメモリ全体のサイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリ全体のサイズを返します。
    */
    virtual size_t DoGetTotalSize() const NN_NOEXCEPT = 0;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち空き領域のサイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち空き領域のサイズを返します。
    */
    virtual size_t DoGetFreeSize() const NN_NOEXCEPT = 0;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを返します。
    */
    virtual size_t DoGetTotalAllocatableSize() const NN_NOEXCEPT = 0;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち空き領域サイズのピークを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち空き領域サイズのピークを返します。
    */
    virtual size_t DoGetFreeSizePeak() const NN_NOEXCEPT = 0;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを返します。
    */
    virtual size_t DoGetTotalAllocatableSizePeak() const NN_NOEXCEPT = 0;

    /*!
    * @brief       バッファマネージャが管理するメモリの確保リトライ回数を取得します。
    *
    * @return      バッファマネージャが管理するメモリの確保リトライ回数を返します。
    */
    virtual size_t DoGetRetriedCount() const NN_NOEXCEPT = 0;

    /*!
    * @brief       バッファマネージャが管理するメモリの確保リトライ回数/空き領域サイズのピークをクリアします。
    */
    virtual void DoClearPeak() NN_NOEXCEPT = 0;
};

}}
