﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include <nn/lmem/lmem_UnitHeap.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/buffers/fs_IBufferManager.h>
#include <nn/fssystem/buffers/fs_FileSystemBuddyHeap.h>


namespace nn { namespace fssystem {

/*!
* @brief   ファイルシステムバッファマネージャクラス
*/
class FileSystemBufferManager : public IBufferManager
{
    NN_DISALLOW_COPY(FileSystemBufferManager);
    NN_DISALLOW_MOVE(FileSystemBufferManager);

public:
    typedef FileSystemBuddyHeap BuddyHeap;

public:
    /*!
    * @brief       ワークバッファに必要なサイズを計算します。
    *
    * @param[in]   size        ヒープに割り当てる領域のサイズ
    * @param[in]   blockSize   ブロックのサイズ
    *
    * @return      ワークバッファに必要なサイズを返します。
    */
    static size_t QueryWorkBufferSize(int cacheCountMax, int orderMax) NN_NOEXCEPT
    {
        const auto buddyBufferSize = FileSystemBuddyHeap::QueryWorkBufferSize(orderMax);
        const auto tableBufferSize = CacheHandleTable::QueryWorkBufferSize(cacheCountMax);
        return buddyBufferSize + tableBufferSize;
    }

    /*!
    * @brief       コンストラクタです。
    */
    FileSystemBufferManager() NN_NOEXCEPT
    : m_TotalSize(0),
      m_SizeFreePeak(0),
      m_SizeTotalAllocatablePeak(0),
      m_RetriedCount(0),
      m_Locker(true)
    {
    }

    /*!
    * @brief       デストラクタです。
    */
    virtual ~FileSystemBufferManager() NN_NOEXCEPT
    {
    }

    /*!
    * @brief       バッファマネージャの初期化を行います。
    *
    * @param[in]   cacheCountMax   キャッシュ数の最大
    * @param[in]   address         バッファマネージャーに割り当てるメモリブロックの先頭アドレス
    * @param[in]   bufferSize      バッファマネージャーに割り当てるメモリブロックのバイト数
    * @param[in]   blockSize       バディヒープのブロックのバイト数
    *
    * @details     最大オーダーは全領域を格納できるだけの大きさが自動的に設定されます。
    *
    * @pre
    *              - 0 < bufferSize
    */
    nn::Result Initialize(
                   int cacheCountMax,
                   uintptr_t address,
                   size_t bufferSize,
                   size_t blockSize
               ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(bufferSize, 0U);
        NN_RESULT_DO(
            m_CacheHandleTable.Initialize(cacheCountMax)
        );
        NN_RESULT_DO(
            m_Buddy.Initialize(address, bufferSize, blockSize)
        );
        m_TotalSize = m_Buddy.GetTotalFreeSize();
        m_SizeFreePeak = m_TotalSize;
        m_SizeTotalAllocatablePeak = m_TotalSize;
        NN_RESULT_SUCCESS;
    }

    /*!
    * @brief       バッファマネージャの初期化を行います。
    *
    * @param[in]   cacheCountMax   キャッシュ数の最大
    * @param[in]   address         バッファマネージャーに割り当てるメモリブロックの先頭アドレス
    * @param[in]   bufferSize      バッファマネージャーに割り当てるメモリブロックのバイト数
    * @param[in]   blockSize       バディヒープのブロックのバイト数
    * @param[in]   orderMax        バディヒープの最大オーダー数
    *
    * @pre
    *              - 0 < bufferSize
    */
    nn::Result Initialize(
                   int cacheCountMax,
                   uintptr_t address,
                   size_t bufferSize,
                   size_t blockSize,
                   int orderMax
               ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(bufferSize, 0U);
        NN_RESULT_DO(
            m_CacheHandleTable.Initialize(cacheCountMax)
        );
        NN_RESULT_DO(
            m_Buddy.Initialize(address, bufferSize, blockSize, orderMax)
        );
        m_TotalSize = m_Buddy.GetTotalFreeSize();
        m_SizeFreePeak = m_TotalSize;
        m_SizeTotalAllocatablePeak = m_TotalSize;
        NN_RESULT_SUCCESS;
    }

    /*!
    * @brief       バッファマネージャの初期化を行います。
    *
    * @param[in]   cacheCountMax   キャッシュ数の最大
    * @param[in]   address         バッファマネージャーに割り当てるメモリブロックの先頭アドレス
    * @param[in]   bufferSize      バッファマネージャーに割り当てるメモリブロックのバイト数
    * @param[in]   blockSize       バディヒープのブロックのバイト数
    * @param[in]   workBuffer      ワークバッファ
    * @param[in]   workBufferSize  ワークバッファのサイズ
    *
    * @details     ワークバッファを外部から渡します。
    *
    *              最大オーダーは全領域を格納できるだけの大きさが自動的に設定されます。
    *
    * @pre
    *              - 0 < bufferSize
    *              - workBuffer != nullptr
    *              - QueryWorkBufferSize(cacheCountMax, orderMax) <= workBufferSize
    *                where orderMax = FileSystemBuddyHeap::QueryOrderMax(bufferSize, blockSize)
    */
    nn::Result Initialize(
                   int cacheCountMax,
                   uintptr_t address,
                   size_t bufferSize,
                   size_t blockSize,
                   void* workBuffer,
                   size_t workBufferSize
               ) NN_NOEXCEPT
    {
        const auto tableBufferSize = CacheHandleTable::QueryWorkBufferSize(cacheCountMax);
        const auto buddyBufferSize = workBufferSize - tableBufferSize;
        NN_SDK_REQUIRES_GREATER(workBufferSize, tableBufferSize);
        const auto tableBuffer = reinterpret_cast<char*>(workBuffer);
        const auto buddyBuffer = tableBuffer + tableBufferSize;

        NN_RESULT_DO(
            m_CacheHandleTable.Initialize(cacheCountMax, tableBuffer, tableBufferSize)
        );
        NN_RESULT_DO(
            m_Buddy.Initialize(address, bufferSize, blockSize, buddyBuffer, buddyBufferSize)
        );
        m_TotalSize = m_Buddy.GetTotalFreeSize();
        m_SizeFreePeak = m_TotalSize;
        m_SizeTotalAllocatablePeak = m_TotalSize;
        NN_RESULT_SUCCESS;
    }

    /*!
    * @brief       バッファマネージャの初期化を行います。
    *
    * @param[in]   cacheCountMax   キャッシュ数の最大
    * @param[in]   address         バッファマネージャーに割り当てるメモリブロックの先頭アドレス
    * @param[in]   bufferSize      バッファマネージャーに割り当てるメモリブロックのバイト数
    * @param[in]   blockSize       バディヒープのブロックのバイト数
    * @param[in]   orderMax        バディヒープの最大オーダー数
    * @param[in]   workBuffer      ワークバッファ
    * @param[in]   workBufferSize  ワークバッファのサイズ
    *
    * @details     ワークバッファを外部から渡します。
    *
    * @pre
    *              - 0 < bufferSize
    *              - workBuffer != nullptr
    *              - QueryWorkBufferSize(cacheCountMax, orderMax) <= workBufferSize
    */
    nn::Result Initialize(
                   int cacheCountMax,
                   uintptr_t address,
                   size_t bufferSize,
                   size_t blockSize,
                   int orderMax,
                   void* workBuffer,
                   size_t workBufferSize
               ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(bufferSize, 0U);

        const auto tableBufferSize = CacheHandleTable::QueryWorkBufferSize(cacheCountMax);
        const auto buddyBufferSize = workBufferSize - tableBufferSize;
        NN_SDK_REQUIRES_GREATER(workBufferSize, tableBufferSize);
        const auto tableBuffer = reinterpret_cast<char*>(workBuffer);
        const auto buddyBuffer = tableBuffer + tableBufferSize;

        NN_RESULT_DO(
            m_CacheHandleTable.Initialize(cacheCountMax, tableBuffer, tableBufferSize)
        );
        NN_RESULT_DO(
            m_Buddy.Initialize(address, bufferSize, blockSize, orderMax, buddyBuffer, buddyBufferSize)
        );
        m_TotalSize = m_Buddy.GetTotalFreeSize();
        m_SizeFreePeak = m_TotalSize;
        m_SizeTotalAllocatablePeak = m_TotalSize;
        NN_RESULT_SUCCESS;
    }

    /*!
    * @brief       バッファマネージャを破棄します。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_Buddy.Finalize();
        m_CacheHandleTable.Finalize();
    }

private:
    //! キャッシュハンドルテーブルです。
    class CacheHandleTable
    {
        NN_DISALLOW_COPY(CacheHandleTable);
        NN_DISALLOW_MOVE(CacheHandleTable);

    public:
        /*!
        * @brief       ワークバッファに必要なサイズを計算します。
        *
        * @param[in]   orderMax    最大ページ数のオーダー
        *
        * @return      ワークバッファに必要なサイズを返します。
        *
        * @pre
        *              - 0 < cacheCountMax
        */
        static size_t QueryWorkBufferSize(int cacheCountMax) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(cacheCountMax, 0);
            const auto entryHeapSize = sizeof(Entry) * cacheCountMax;
            const auto attrInfoListSize = sizeof(AttrInfo) * 256; // 大雑把に
            return nn::util::align_up(
                       entryHeapSize + attrInfoListSize + NN_ALIGNOF(Entry) + NN_ALIGNOF(AttrInfo),
                       8
                   );
        }

        /*!
        * @brief       コンストラクタです。
        */
        CacheHandleTable() NN_NOEXCEPT;

        /*!
        * @brief       デストラクタです。
        */
        ~CacheHandleTable() NN_NOEXCEPT;

        /*!
        * @brief       初期化します。
        *
        * @param[in]   cacheCountMax   キャッシュの最大数
        *
        * @pre
        *              - 未初期化である。
        */
        nn::Result Initialize(int cacheCountMax) NN_NOEXCEPT;

        /*!
        * @brief       初期化します。
        *
        * @param[in]   cacheCountMax   キャッシュの最大数
        * @param[in]   workBuffer      ワークバッファ
        * @param[in]   workBufferSize  ワークバッファのサイズ
        *
        * @details     ワークバッファを外部から渡します。
        *
        * @pre
        *              - 未初期化である。
        *              - workBuffer != null
        *              - QueryWorkBufferSize(cacheCountMax) <= workBufferSize
        */
        nn::Result Initialize(
                       int cacheCountMax,
                       void* workBuffer,
                       size_t workBufferSize
                   ) NN_NOEXCEPT
        {
            const auto alignedEntryBuffer
                = nn::util::align_up(reinterpret_cast<uintptr_t>(workBuffer), NN_ALIGNOF(Entry));
            m_ExternalEntryBuffer = reinterpret_cast<char*>(alignedEntryBuffer);
            m_EntryBufferSize = sizeof(Entry) * cacheCountMax;

            const auto alignedAttrInfoBuffer
                = nn::util::align_up(
                      reinterpret_cast<uintptr_t>(m_ExternalEntryBuffer + m_EntryBufferSize),
                      NN_ALIGNOF(AttrInfo)
                  );
            const auto workBufferEnd = reinterpret_cast<uintptr_t>(workBuffer) + workBufferSize;
            m_ExternalAttrInfoBuffer = reinterpret_cast<char*>(alignedAttrInfoBuffer);
            m_ExternalAttrInfoCount
                = static_cast<int>((workBufferEnd - alignedAttrInfoBuffer) / sizeof(AttrInfo));

            return Initialize(cacheCountMax);
        }

        /*!
        * @brief       終了処理を行います。
        *
        * @details     デストラクタで自動的に呼び出されるため、明示的に呼び出す必要はありません。
        */
        void Finalize() NN_NOEXCEPT;

        /*!
        * @brief       キャッシュを登録します。
        *
        * @param[out]  pOutHandle  登録したキャッシュのハンドルを格納するバッファ
        * @param[in]   address     登録するメモリ範囲の先頭アドレス
        * @param[in]   size        登録するメモリ範囲のサイズ
        * @param[in]   bufAttr     登録するバッファの属性
        *
        * @return      登録に成功したかどうかを返します。
        * @retval      true        登録に成功しました。
        * @retval      false       登録に失敗しました。
        *
        * @pre
        *              - 初期化済みである。
        *              - pOutHandle != nullptr
        */
        bool Register(
                 CacheHandle* pOutHandle,
                 uintptr_t address,
                 size_t size,
                 const BufferAttribute& bufAttr
             ) NN_NOEXCEPT;

        /*!
        * @brief       キャッシュの登録を解除します。
        *
        * @param[out]  pOutAddress 登録を解除したキャッシュのメモリ範囲の先頭アドレスを格納するバッファ
        * @param[out]  pOutSize    登録を解除したキャッシュのメモリ範囲のサイズを格納するバッファ
        * @param[in]   handle      登録を解除するキャッシュのハンドル
        *
        * @return      登録の解除に成功したかどうかを返します。
        * @retval      true        解除に成功しました。
        * @retval      false       解除に失敗しました。
        *
        * @pre
        *              - 初期化済みである。
        *              - pOutAddress != nullptr
        *              - pOutSize != nullptr
        */
        bool Unregister(uintptr_t* pOutAddress, size_t* pOutSize, CacheHandle handle) NN_NOEXCEPT;

        /*!
        * @brief       最古のキャッシュの登録を解除します。
        *
        * @param[out]  pOutAddress 登録を解除したキャッシュのメモリ範囲の先頭アドレスを格納するバッファ
        * @param[out]  pOutSize    登録を解除したキャッシュのメモリ範囲のサイズを格納するバッファ
        * @param[in]   bufAttr     登録を解除するキャッシュを制限するためのバッファ属性
        *
        * @return      登録の解除に成功したかどうかを返します。
        * @retval      true        解除に成功しました。
        * @retval      false       解除に失敗しました。
        *
        * @pre
        *              - 初期化済みである。
        *              - pOutAddress != nullptr
        *              - pOutSize != nullptr
        */
        bool UnregisterOldest(
                 uintptr_t* pOutAddress,
                 size_t* pOutSize,
                 const BufferAttribute& bufAttr
             ) NN_NOEXCEPT
        {
            return UnregisterOldest(pOutAddress, pOutSize, bufAttr, 0);
        }

        /*!
        * @brief       最古のキャッシュの登録を解除します。
        *
        * @param[out]  pOutAddress     登録を解除したキャッシュのメモリ範囲の先頭アドレスを格納するバッファ
        * @param[out]  pOutSize        登録を解除したキャッシュのメモリ範囲のサイズを格納するバッファ
        * @param[in]   bufAttr         登録を解除するキャッシュを制限するためのバッファ属性
        * @param[in]   requiredSize    必要な空き領域のサイズ
        *
        * @return      登録の解除に成功したかどうかを返します。
        * @retval      true            解除に成功しました。
        * @retval      false           解除に失敗しました。
        *
        * @pre
        *              - 初期化済みである。
        *              - pOutAddress != nullptr
        *              - pOutSize != nullptr
        */
        bool UnregisterOldest(
                 uintptr_t* pOutAddress,
                 size_t* pOutSize,
                 const BufferAttribute& bufAttr,
                 size_t requiredSize
             ) NN_NOEXCEPT;

        /*!
        * @brief       ハンドルを新規作成します。
        *
        * @return      作成したハンドルを返します。
        *
        * @pre
        *              - 初期化済みである。
        */
        CacheHandle PublishCacheHandle() NN_NOEXCEPT;

        /*!
        * @brief       キャッシュサイズの合計を取得します。
        *
        * @return      キャッシュサイズの合計を返します。
        */
        size_t GetTotalCacheSize() const NN_NOEXCEPT;

    private:
        //! テーブルのエントリです。
        class Entry
        {
        public:
            void Initialize(
                     CacheHandle     handle,
                     uintptr_t       address,
                     size_t          size,
                     BufferAttribute bufAttr
                 ) NN_NOEXCEPT
            {
                m_Handle = handle;
                m_Address = address;
                m_Size = size;
                m_BufAttr = bufAttr;
            }
            CacheHandle GetHandle() const NN_NOEXCEPT
            {
                return m_Handle;
            }
            uintptr_t GetAddress() const NN_NOEXCEPT
            {
                return m_Address;
            }
            size_t GetSize() const NN_NOEXCEPT
            {
                return m_Size;
            }
            BufferAttribute GetBufferAttribute() const NN_NOEXCEPT
            {
                return m_BufAttr;
            }
        private:
            CacheHandle     m_Handle;  //!< ハンドル
            uintptr_t       m_Address; //!< 先頭アドレス
            size_t          m_Size;    //!< サイズ
            BufferAttribute m_BufAttr; //!< 属性
        };

        //! 属性の管理情報です。
        class AttrInfo : public nn::util::IntrusiveListBaseNode<AttrInfo>,
                         public nn::fs::detail::Newable
        {
            NN_DISALLOW_COPY(AttrInfo);
        public:
            AttrInfo(int level, int cacheCount, size_t cacheSize) NN_NOEXCEPT
                : m_Level(level),
                  m_CacheCount(cacheCount),
                  m_CacheSize(cacheSize)
            {
            }
            int GetLevel() const NN_NOEXCEPT
            {
                return m_Level;
            }
            int GetCacheCount() const NN_NOEXCEPT
            {
                return m_CacheCount;
            }
            void IncrementCacheCount() NN_NOEXCEPT
            {
                ++m_CacheCount;
            }
            void DecrementCacheCount() NN_NOEXCEPT
            {
                --m_CacheCount;
            }
            size_t GetCacheSize() const NN_NOEXCEPT
            {
                return m_CacheSize;
            }
            void AddCacheSize(size_t differentialSize) NN_NOEXCEPT
            {
                m_CacheSize += differentialSize;
            }
            void SubtractCacheSize(size_t differentialSize) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_GREATER_EQUAL(m_CacheSize, differentialSize);
                m_CacheSize -= differentialSize;
            }

            using Newable::operator new;
            using Newable::operator delete;
            static void* operator new(size_t, void* buf) NN_NOEXCEPT
            {
                return buf;
            }
            static void operator delete(void*, size_t, void*) NN_NOEXCEPT
            {
            }
        private:
            int     m_Level;        //!< レベル
            int     m_CacheCount;   //!< キャッシュ数
            size_t  m_CacheSize;    //!< キャッシュサイズ
        };

        //! 属性リストの型特性です。
        typedef nn::util::IntrusiveListBaseNodeTraits<AttrInfo> AttrListTraits;

    private:
        /*!
        * @brief       キャッシュの登録を解除します。
        *
        * @param[out]  pOutAddress     登録を解除したキャッシュのメモリ範囲の先頭アドレスを格納するバッファ
        * @param[out]  pOutSize        登録を解除したキャッシュのメモリ範囲のサイズを格納するバッファ
        * @param[in]   pEntry          登録を解除するエントリ
        *
        * @pre
        *              - 初期化済みである。
        *              - pOutAddress != nullptr
        *              - pOutSize != nullptr
        *              - pEntry != nullptr
        */
        void UnregisterCore(uintptr_t* pOutAddress, size_t* pOutSize, Entry* pEntry) NN_NOEXCEPT;

        /*!
        * @brief       エントリを確保して初期化します。
        *
        * @param[in]   address エントリに登録するキャッシュのメモリ範囲の先頭アドレス
        * @param[in]   size    エントリに登録するキャッシュのメモリ範囲のサイズ
        * @param[in]   bufAttr エントリに登録するバッファ属性
        *
        * @return      作成したエントリを指すポインタを返します。
        * @retval      nullptr メモリ不足のため新規作成できませんでした。
        *
        * @pre
        *              - 初期化済みである。
        */
        Entry* AcquireEntry(
                   uintptr_t address,
                   size_t size,
                   const BufferAttribute& bufAttr
               ) NN_NOEXCEPT;

        /*!
        * @brief       エントリを解放します。
        *
        * @param[in]   pEntry  解放するエントリ
        *
        * @pre
        *              - 初期化済みである。
        *              - pEntry != nullptr
        */
        void ReleaseEntry(Entry* pEntry) NN_NOEXCEPT;

        /*!
        * @brief       指定の属性の管理情報を取得します。
        *
        * @param[in]   bufAttr 管理情報を取得する属性
        *
        * @return      管理情報をさすポインタを返します。
        * @retval      nullptr 指定の属性の管理情報は登録されていません。
        */
        AttrInfo* FindAttrInfo(const BufferAttribute& bufAttr) NN_NOEXCEPT;

        /*!
        * @brief       指定の属性を持つキャッシュの最低数を取得します。
        *
        * @param[in]   bufAttr バッファ属性
        *
        * @return      バッファの最低数を返します。
        *
        * @details     指定の属性を持つキャッシュの個数は引き剥がしによって本関数の戻り値未満に減らされません。
        */
        int GetCacheCountMin(const BufferAttribute& bufAttr) NN_NOEXCEPT
        {
            NN_UNUSED(bufAttr);
            return m_CacheCountMin;
        }

        /*!
        * @brief       指定の属性を持つキャッシュの最低数を取得します。
        *
        * @param[in]   bufAttr バッファ属性
        *
        * @return      バッファの最低数を返します。
        *
        * @details     指定の属性を持つキャッシュの合計サイズは引き剥がしによって本関数の戻り値未満に減らされません。
        */
        size_t GetCacheSizeMin(const BufferAttribute& bufAttr) NN_NOEXCEPT
        {
            NN_UNUSED(bufAttr);
            return m_CacheSizeMin;
        }

    private:
        std::unique_ptr<char[], nn::fs::detail::Deleter> m_InternalEntryBuffer; //!< エントリ用内部バッファ
        char* m_ExternalEntryBuffer;                                            //!< エントリ用外部バッファ
        size_t m_EntryBufferSize;                                               //!< エントリ用バッファのサイズ
        Entry* m_pEntries;                                                      //!< エントリ
        int m_EntryCount;                                                       //!< エントリ数
        int m_EntryCountMax;                                                    //!< エントリ数の最大
        nn::util::IntrusiveList<AttrInfo, AttrListTraits> m_AttrList;           //!< 属性リスト
        char* m_ExternalAttrInfoBuffer;                                         //!< 属性情報用外部バッファ
        int m_ExternalAttrInfoCount;                                            //!< 属性情報用外部バッファの数
        int m_CacheCountMin;                                                    //!< キャッシュ数の最低保証値
        size_t m_CacheSizeMin;                                                  //!< キャッシュサイズの最低保証値
        size_t m_TotalCacheSize;                                                //!< キャッシュサイズの合計
        CacheHandle m_CurrentHandle;                                            //!< 現在発行しているハンドルの値
    };

private:
    /*!
    * @brief       指定のサイズと属性を持つバッファを割り当てます。
    *
    * @param[in]   size    割り当てるサイズ
    * @param[in]   bufAttr 割り当てるバッファの属性
    *
    * @return      割り当てたメモリ範囲の先頭アドレスとサイズの組を返します。
    *
    * @details     割り当てに失敗した場合、成功するまで短時間待機して割り当てを再試行することを繰り返します。
    */
    virtual const std::pair<uintptr_t, size_t> DoAllocateBuffer(
                                                   size_t size,
                                                   const BufferAttribute& bufAttr
                                               ) NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       指定の範囲のバッファを解放します。
    *
    * @param[in]   address 解放するバッファの先頭アドレス
    * @param[in]   size    解放するバッファのサイズ
    *
    * @pre
    *              - 指定の範囲がこのマネージャによって確保された未解放の領域である
    *              - 解放するメモリ範囲がキャッシュとして登録されていない
    *
    * @details     事前条件に違反していたとしても停止しない可能性があります。
    */
    virtual void DoDeallocateBuffer(uintptr_t address, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       指定のメモリ範囲をキャッシュとして登録します。
    *
    * @param[in]   address         登録するバッファの先頭アドレス
    * @param[in]   size            登録するバッファのサイズ
    * @param[in]   bufAttr         登録するバッファの属性
    *
    * @return      登録したキャッシュのハンドルを返します。
    *
    * @pre
    *              - 登録するメモリ範囲がキャッシュとして登録されていない
    *
    * @details     登録したキャッシュは自動的に無効化されることがあります。
    *
    *              事前条件に違反していたとしても停止しない可能性があります。
    */
    virtual CacheHandle DoRegisterCache(
                            uintptr_t address,
                            size_t size,
                            const BufferAttribute& bufAttr
                        ) NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       指定のハンドルに紐付けられたキャッシュを取得します。
    *
    * @param[in]   handle  取得するキャッシュのハンドル
    *
    * @return      キャッシュのメモリ範囲の先頭アドレスとサイズの組を返します。
    *
    * @details     取得に成功したキャッシュの登録は解除されます。
    *
    *              無効化されたキャッシュのハンドルが指定された場合、
    *              キャッシュの取得に失敗し、返される先頭アドレスとサイズはいずれも 0 になります。
    */
    virtual const std::pair<uintptr_t, size_t> DoAcquireCache(
                                                   CacheHandle handle
                                               ) NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       バッファマネージャが管理するメモリ全体のサイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリ全体のサイズを返します。
    */
    virtual size_t DoGetTotalSize() const NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち空き領域のサイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち空き領域のサイズを返します。
    */
    virtual size_t DoGetFreeSize() const NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズを返します。
    */
    virtual size_t DoGetTotalAllocatableSize() const NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち空き領域サイズのピークを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち空き領域サイズのピークを返します。
    */
    virtual size_t DoGetFreeSizePeak() const NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを取得します。
    *
    * @return      バッファマネージャが管理するメモリのうち割り当て可能な領域の合計サイズのピークを返します。
    */
    virtual size_t DoGetTotalAllocatableSizePeak() const NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       バッファマネージャが管理するメモリの確保リトライ回数を取得します。
    *
    * @return      バッファマネージャが管理するメモリの確保リトライ回数を返します。
    */
    virtual size_t DoGetRetriedCount() const NN_NOEXCEPT NN_OVERRIDE;

    /*!
    * @brief       バッファマネージャが管理するメモリの確保リトライ回数/空き領域サイズのピークをクリアします。
    */
    virtual void DoClearPeak() NN_NOEXCEPT NN_OVERRIDE;
private:
    BuddyHeap m_Buddy;                   //!< バディヒープ
    CacheHandleTable m_CacheHandleTable; //!< キャッシュハンドルテーブル
    size_t m_TotalSize;                  //!< 管理するメモリ全体のサイズ
    size_t m_SizeFreePeak;               //!< 空き領域サイズのピーク
    size_t m_SizeTotalAllocatablePeak;   //!< 割り当て可能な領域の合計サイズのピーク
    size_t m_RetriedCount;               //!< メモリ確保リトライ回数
    mutable nn::os::Mutex m_Locker;      //!< 排他オブジェクト
};

}}
