﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/buffers/fs_IBufferManager.h>

namespace nn { namespace fssystem { namespace buffers {

    namespace detail {

        // バッファが確保できないときにスリープする時間
        static const auto RetryWait = nn::TimeSpan::FromMilliSeconds(10);

        class BufferManagerBufferDeallocator
        {
        public:
            BufferManagerBufferDeallocator(IBufferManager* pBufferManager, size_t size) NN_NOEXCEPT
                : m_pBufferManager(pBufferManager),
                m_Size(size)
            {
                NN_SDK_REQUIRES_NOT_NULL(pBufferManager);
            }

            BufferManagerBufferDeallocator() NN_NOEXCEPT
                : m_pBufferManager(nullptr),
                m_Size(0)
            {
            }

            void operator()(void* ptr) const NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(m_pBufferManager);
                if( ptr != nullptr )
                {
                    m_pBufferManager->DeallocateBuffer(reinterpret_cast<uintptr_t>(ptr), m_Size);
                }
            }

            size_t GetSize() const NN_NOEXCEPT
            {
                return m_Size;
            }

        private:
            IBufferManager* m_pBufferManager;
            size_t m_Size;
        };

    }

    template<typename Action, typename OnBufferAllocationFailed>
    Result DoContinouslyUntilBufferIsAllocated(
        Action action,
        OnBufferAllocationFailed onBufferAllocationFailed,
        const char* pFunctionName
    ) NN_NOEXCEPT
    {
        NN_UNUSED(pFunctionName);

        // キャッシュバッファ割り当てリトライログを出力する失敗回数
        static const auto BufferAllocationRetryLogIntervalCount = 100;
        static const auto BufferAllocationRetryLogCountMax = 10; // オーバーフローは気にしない

        for( auto count = 1; ; ++count )
        {
            auto result = action();
            if( fs::ResultBufferAllocationFailed::Includes(result) )
            {
                if( (1 <= count && count <= BufferAllocationRetryLogCountMax)
                    || (count % BufferAllocationRetryLogIntervalCount) == 0 )
                {
                    NN_SDK_LOG(
                        "[fs] Retried to allocate cache buffer %d times in %s.\n",
                        count,
                        pFunctionName);
                }
                NN_RESULT_DO(onBufferAllocationFailed());
                nn::os::SleepThread(detail::RetryWait);
            }
            else
            {
                NN_RESULT_DO(result);
                NN_RESULT_SUCCESS;
            }
        }
    }

    template<typename Action>
    Result DoContinouslyUntilBufferIsAllocated(
        Action action,
        const char* pFunctionName
    ) NN_NOEXCEPT
    {
        NN_RESULT_DO(DoContinouslyUntilBufferIsAllocated(
            action,
            []() NN_NOEXCEPT
            {
                NN_RESULT_SUCCESS;
            },
            pFunctionName
        ));
        NN_RESULT_SUCCESS;
    }

    template<typename ValueType>
    std::unique_ptr<ValueType, detail::BufferManagerBufferDeallocator> MakeUniqueBufferFromBufferManager(
        IBufferManager* pBufferManager,
        const char* pFunctionName
    ) NN_NOEXCEPT
    {
        NN_UNUSED(pFunctionName);

        NN_SDK_REQUIRES_NOT_NULL(pBufferManager);
        std::pair<uintptr_t, size_t> buffer(0, 0);
        auto result = DoContinouslyUntilBufferIsAllocated(
            [&]() NN_NOEXCEPT -> Result
            {
                buffer = pBufferManager->AllocateBuffer(sizeof(ValueType));
                if( buffer.first == 0 )
                {
                    NN_RESULT_THROW(fs::ResultBufferAllocationFailed());
                }
                NN_RESULT_SUCCESS;
            },
            []() NN_NOEXCEPT
            {
                NN_RESULT_SUCCESS;
            },
            pFunctionName
        );
        NN_UNUSED(result);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_SDK_ASSERT_NOT_EQUAL(buffer.first, static_cast<size_t>(0));
        NN_SDK_ASSERT_ALIGNED(buffer.first, NN_ALIGNOF(ValueType));
        return std::unique_ptr<ValueType, detail::BufferManagerBufferDeallocator>(
            reinterpret_cast<ValueType*>(buffer.first),
            detail::BufferManagerBufferDeallocator(pBufferManager, buffer.second)
            );
    }

    class BufferManagerContext
    {
    public:
        BufferManagerContext() NN_NOEXCEPT
            : m_IsNeedBlocking(false)
        {
        }

    public:
        bool IsNeedBlocking() const NN_NOEXCEPT
        {
            return m_IsNeedBlocking;
        }

        void SetNeedBlocking(bool isNeedBlocking) NN_NOEXCEPT
        {
            m_IsNeedBlocking = isNeedBlocking;
        }

    private:
        bool m_IsNeedBlocking;
    };

    void RegisterBufferManagerContext(const BufferManagerContext* pContext) NN_NOEXCEPT;

    void EnableBlockingBufferManagerAllocation() NN_NOEXCEPT;

    BufferManagerContext* GetBufferManagerContext() NN_NOEXCEPT;

    class ScopedBufferManagerContextRegistration
    {
    public:
        explicit ScopedBufferManagerContextRegistration() NN_NOEXCEPT
        {
            m_pOldContext = GetBufferManagerContext();
            if( m_pOldContext != nullptr )
            {
                m_CurrentContext = *m_pOldContext;
            }
            RegisterBufferManagerContext(&m_CurrentContext);
        }

        ~ScopedBufferManagerContextRegistration() NN_NOEXCEPT
        {
            RegisterBufferManagerContext(m_pOldContext);
        }

    private:
        BufferManagerContext m_CurrentContext;
        const BufferManagerContext* m_pOldContext;
    };

    template<typename IsValidBuffer>
    Result AllocateBufferUsingBufferManagerContext(
        std::pair<uintptr_t, size_t>* outBuffer,
        fssystem::IBufferManager* pBufferManager,
        size_t size,
        const IBufferManager::BufferAttribute attribute,
        IsValidBuffer funcIsValidBuffer,
        const char* pFunctionName
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outBuffer);
        NN_SDK_ASSERT_NOT_NULL(pBufferManager);
        NN_SDK_ASSERT_NOT_NULL(pFunctionName);

        NN_UNUSED(pFunctionName);

        *outBuffer = std::pair<uintptr_t, size_t>(0, 0);
        auto pContext = GetBufferManagerContext();

        if( pContext == nullptr || !pContext->IsNeedBlocking() )
        {
            auto buffer = pBufferManager->AllocateBuffer(size, attribute);
            if( !funcIsValidBuffer(buffer) )
            {
                if( buffer.first != 0 )
                {
                    pBufferManager->DeallocateBuffer(buffer.first, buffer.second);
                }
                NN_RESULT_THROW(fs::ResultBufferAllocationFailed());
            }
            *outBuffer = buffer;
            NN_SDK_ASSERT_NOT_EQUAL(static_cast<uintptr_t>(0), outBuffer->first);
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_RESULT_DO(
                DoContinouslyUntilBufferIsAllocated(
                    [=]() NN_NOEXCEPT -> Result
                    {
                        auto buffer = pBufferManager->AllocateBuffer(size, attribute);
                        if( !funcIsValidBuffer(buffer) )
                        {
                            if( buffer.first != 0 )
                            {
                                pBufferManager->DeallocateBuffer(buffer.first, buffer.second);
                            }
                            NN_RESULT_THROW(fs::ResultBufferAllocationFailed());
                        }
                        *outBuffer = buffer;
                        NN_RESULT_SUCCESS;
                    },
                    [=]() NN_NOEXCEPT -> Result
                    {
                        NN_RESULT_SUCCESS;
                    },
                    pFunctionName
                )
            );
            NN_SDK_ASSERT_NOT_EQUAL(static_cast<uintptr_t>(0), outBuffer->first);
            NN_RESULT_SUCCESS;
        }
    }

}}}
