﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Allocator.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/kvdb/kvdb_AutoBuffer.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/kvdb/kvdb_KeyValueArchive.h>
#include <nn/kvdb/kvdb_Result.h>

namespace nn { namespace fssrv { namespace kvdb {

    template<class Key>
    class FlatMapKeyValueStore
    {
        NN_DISALLOW_COPY(FlatMapKeyValueStore);
        NN_DISALLOW_MOVE(FlatMapKeyValueStore);

    public:
        struct KeyValue
        {
            Key key;
            void* value;
            size_t size;

            static KeyValue Make(const Key& key, void* value, size_t size) NN_NOEXCEPT
            {
                KeyValue keyValue = { key, value, size };
                return keyValue;
            }
        };

        class Iterator
        {
        public:
            Iterator() NN_NOEXCEPT {}
            Iterator(KeyValue* begin, KeyValue* end) NN_NOEXCEPT : m_Begin(begin), m_End(end) {}

            const Key& Get() const
            {
                return m_Begin->key;
            }

            void* GetPointer()
            {
                return m_Begin->value;
            }

            size_t GetSize() const
            {
                return m_Begin->size;
            }

            void Next()
            {
                ++m_Begin;
            }

            bool IsEnd() const
            {
                return m_Begin == m_End;
            }

            void Fix(KeyValue* keyValue, KeyValue* end)
            {
                if (end > m_End)
                {
                    /* 挿入による位置ずれ修正 */
                    NN_SDK_ASSERT(end == m_End + 1);
                    if (IsEnd())
                    {
                        /* 既に終端に来ていたなら、終端のままとする */
                        m_Begin = end;
                    }
                    else if (keyValue < m_Begin)
                    {
                        ++m_Begin;
                    }
                    m_End = end;
                }
                else if (end < m_End)
                {
                    /* 削除による位置ずれ修正 */
                    NN_SDK_ASSERT(end == m_End - 1);
                    if (keyValue < m_Begin)
                    {
                        --m_Begin;
                    }
                    m_End = end;
                }
            }

        private:
            KeyValue* m_Begin;
            KeyValue* m_End;
        };

        class ConstIterator
        {
        public:
            ConstIterator() NN_NOEXCEPT {}
            ConstIterator(const KeyValue* begin, const KeyValue* end) NN_NOEXCEPT : m_Begin(begin), m_End(end) {}

            const Key& Get() const
            {
                return m_Begin->key;
            }

            const void* GetPointer() const
            {
                return m_Begin->value;
            }

            size_t GetSize() const
            {
                return m_Begin->size;
            }

            void Next()
            {
                ++m_Begin;
            }

            bool IsEnd() const
            {
                return m_Begin == m_End;
            }

        private:
            const KeyValue* m_Begin;
            const KeyValue* m_End;
        };

    private:
        typedef nn::kvdb::BoundedString<768> Path;

        class Index
        {
        public:
            Index() NN_NOEXCEPT : m_Count(), m_Table() {}

            ~Index() NN_NOEXCEPT
            {
                if (m_Table)
                {
                    Clear();
                    m_MemoryResource->deallocate(m_Table, sizeof(KeyValue) * m_MaxCount);
                    m_Table = nullptr;
                }
            }

            Result Initialize(int maxCount, MemoryResource* memoryResource) NN_NOEXCEPT
            {
                m_Table = reinterpret_cast<KeyValue*>(memoryResource->allocate(sizeof(KeyValue) * maxCount));
                NN_RESULT_THROW_UNLESS(m_Table, nn::kvdb::ResultAllocationMemoryFailed());
                m_MaxCount = maxCount;
                m_MemoryResource = memoryResource;

                NN_RESULT_SUCCESS;
            }

            int Count() const NN_NOEXCEPT
            {
                return m_Count;
            }

            Result Put(const Key& key, const void* value, size_t size) NN_NOEXCEPT
            {
                auto it = GetLowerBoundImpl(key);
                if (it != GetEndImpl() && it->key == key)
                {
                    m_MemoryResource->deallocate(it->value, it->size);
                }
                else
                {
                    NN_RESULT_THROW_UNLESS(m_Count < m_MaxCount, nn::kvdb::ResultOutOfMaxKeySize());
                    std::memmove(it + 1, it, (GetEndImpl() - it) * sizeof(KeyValue));
                    m_Count++;
                }

                auto allocated = m_MemoryResource->allocate(size);
                NN_RESULT_THROW_UNLESS(allocated, nn::kvdb::ResultAllocationMemoryFailed());
                std::memcpy(allocated, value, size);

                *it = KeyValue::Make(key, allocated, size);

                NN_RESULT_SUCCESS;
            }

            bool Delete(const Key& key) NN_NOEXCEPT
            {
                auto it = GetLowerBoundImpl(key);
                if (it != GetEndImpl() && it->key == key)
                {
                    m_MemoryResource->deallocate(it->value, it->size);
                    std::memmove(it, it + 1, (GetEndImpl() - (it + 1)) * sizeof(KeyValue));

                    m_Count--;
                    return true;
                }

                return false;
            }

            Result PushBackRaw(const Key& key, void* value, size_t size) NN_NOEXCEPT
            {
                NN_RESULT_THROW_UNLESS(m_Count < m_MaxCount, nn::kvdb::ResultOutOfMaxKeySize());

                m_Table[m_Count] = KeyValue::Make(key, value, size);
                m_Count++;

                NN_RESULT_SUCCESS;
            }

            void Clear() NN_NOEXCEPT
            {
                for (int i = 0; i < m_Count; i++)
                {
                    m_MemoryResource->deallocate(m_Table[i].value, m_Table[i].size);
                }

                m_Count = 0;
            }

            void FixIterator(Iterator* iterator, const SaveDataIndexerKey& key)  NN_NOEXCEPT
            {
                NN_SDK_ASSERT(iterator != nullptr);
                auto it = GetLowerBoundImpl(key);
                iterator->Fix(it, GetEndImpl());
            }

            Iterator GetBeginIterator() NN_NOEXCEPT
            {
                return Iterator(GetBeginImpl(), GetEndImpl());
            }

            Iterator GetLowerBoundIterator(const Key& key) NN_NOEXCEPT
            {
                return Iterator(GetLowerBoundImpl(key), GetEndImpl());
            }

            ConstIterator GetBeginConstIterator() const NN_NOEXCEPT
            {
                return ConstIterator(GetBeginImpl(), GetEndImpl());
            }

            ConstIterator GetLowerBoundConstIterator(const Key& key) const NN_NOEXCEPT
            {
                return ConstIterator(GetLowerBoundImpl(key), GetEndImpl());
            }


        private:
            KeyValue* GetLowerBoundImpl(const Key& key) NN_NOEXCEPT
            {
                auto compare = [](const KeyValue& keyValue, const Key& key)->bool
                {
                    return keyValue.key < key;
                };
                return std::lower_bound(GetBeginImpl(), GetEndImpl(), key, compare);
            }

            const KeyValue* GetLowerBoundImpl(const Key& key) const NN_NOEXCEPT
            {
                auto compare = [](const KeyValue& keyValue, const Key& key)->bool
                {
                    return keyValue.key < key;
                };
                return std::lower_bound(GetBeginImpl(), GetEndImpl(), key, compare);
            }

            KeyValue* GetBeginImpl() NN_NOEXCEPT
            {
                return m_Table;
            }

            const KeyValue* GetBeginImpl() const NN_NOEXCEPT
            {
                return m_Table;
            }

            KeyValue* GetEndImpl() NN_NOEXCEPT
            {
                return m_Table + m_Count;
            }

            const KeyValue* GetEndImpl() const NN_NOEXCEPT
            {
                return m_Table + m_Count;
            }

            int m_Count;
            int m_MaxCount;
            KeyValue* m_Table;
            MemoryResource* m_MemoryResource;
        };

    public:
        FlatMapKeyValueStore() NN_NOEXCEPT {}

        Result Initialize(int maxCount, MemoryResource* memoryResource) NN_NOEXCEPT
        {
            m_ArchivePath.Assign("");
            m_ArchiveTemporaryPath.Assign("");

            NN_RESULT_DO(m_Index.Initialize(maxCount, memoryResource));
            m_MemoryResource = memoryResource;

            NN_RESULT_SUCCESS;
        }

        Result Initialize(const char directoryPath[], int maxCount, MemoryResource* memoryResource) NN_NOEXCEPT
        {
            fs::DirectoryEntryType entryType;
            NN_RESULT_DO(fs::GetEntryType(&entryType, directoryPath));
            NN_RESULT_THROW_UNLESS(entryType == fs::DirectoryEntryType_Directory, fs::ResultPathNotFound());

            m_ArchivePath.AssignFormat("%s%s", directoryPath, "/imkvdb.arc");
            m_ArchiveTemporaryPath.AssignFormat("%s%s", directoryPath, "/imkvdb.tmp");

            NN_RESULT_DO(m_Index.Initialize(maxCount, memoryResource));
            m_MemoryResource = memoryResource;

            NN_RESULT_SUCCESS;
        }

        Result Load() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(strlen(m_ArchivePath) > 0);

            m_Index.Clear();

            nn::kvdb::AutoBuffer buffer;
            NN_RESULT_TRY(ReadArchive(&buffer))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            NN_RESULT_DO(LoadFrom(buffer.Get(), buffer.GetSize()));

            NN_RESULT_SUCCESS;
        }

        /**
        * @brief    ファイルシステムにキーバリューを永続化します。
        * @details  ファイルシステムがトランザクション機能を有している場合、destructive を true にすることで更新に必要な空き容量を小さくできます。
        * @params[in] destructive 前回永続化されていたデータを破壊的に更新します。
        */
        Result Save() NN_NOEXCEPT
        {
            return Save(false);
        }

        Result Save(bool destructive) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(strlen(m_ArchivePath) > 0);

            auto archiveSize = CalculateArchiveSize();
            nn::kvdb::AutoBuffer buffer;
            NN_RESULT_DO(buffer.Initialize(archiveSize));

            nn::kvdb::KeyValueArchiveBufferWriter writer(buffer.Get(), buffer.GetSize());
            SaveTo(&writer);

            NN_RESULT_DO(CommitArchive(buffer.Get(), buffer.GetSize(), destructive));

            NN_RESULT_SUCCESS;
        }

        Result Put(const Key& key, const void* data, size_t size) NN_NOEXCEPT
        {
            return m_Index.Put(key, data, size);
        }

        Result Get(size_t* outValue, const Key& key, void* buffer, size_t size) const NN_NOEXCEPT
        {
            auto iter = m_Index.GetLowerBoundConstIterator(key);
            NN_RESULT_THROW_UNLESS(!iter.IsEnd(), nn::kvdb::ResultKeyNotFound());
            NN_RESULT_THROW_UNLESS(iter.Get() == key, nn::kvdb::ResultKeyNotFound());

            auto readSize = std::min(size, iter.GetSize());
            std::memcpy(buffer, iter.GetPointer(), readSize);

            *outValue = readSize;
            NN_RESULT_SUCCESS;
        }

        Result GetPointer(void** outValue, const Key& key) NN_NOEXCEPT
        {
            auto iter = m_Index.GetLowerBoundIterator(key);
            NN_RESULT_THROW_UNLESS(!iter.IsEnd(), nn::kvdb::ResultKeyNotFound());
            NN_RESULT_THROW_UNLESS(iter.Get() == key, nn::kvdb::ResultKeyNotFound());

            *outValue = iter.GetPointer();
            NN_RESULT_SUCCESS;
        }

        Result GetPointer(const void** outValue, const Key& key) const NN_NOEXCEPT
        {
            auto iter = m_Index.GetLowerBoundConstIterator(key);
            NN_RESULT_THROW_UNLESS(!iter.IsEnd(), nn::kvdb::ResultKeyNotFound());
            NN_RESULT_THROW_UNLESS(iter.Get() == key, nn::kvdb::ResultKeyNotFound());

            *outValue = iter.GetPointer();
            NN_RESULT_SUCCESS;
        }

        Result GetSize(size_t* outValue, const Key& key) const NN_NOEXCEPT
        {
            auto iter = m_Index.GetLowerBoundConstIterator(key);
            NN_RESULT_THROW_UNLESS(!iter.IsEnd(), nn::kvdb::ResultKeyNotFound());
            NN_RESULT_THROW_UNLESS(iter.Get() == key, nn::kvdb::ResultKeyNotFound());

            *outValue = iter.GetSize();
            NN_RESULT_SUCCESS;
        }

        Result Delete(const Key& key) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(m_Index.Delete(key), nn::kvdb::ResultKeyNotFound());

            NN_RESULT_SUCCESS;
        }

        Iterator GetBeginIterator() NN_NOEXCEPT
        {
            return m_Index.GetBeginIterator();
        }

        Iterator GetLowerBoundIterator(const Key& key) NN_NOEXCEPT
        {
            return m_Index.GetLowerBoundIterator(key);
        }

        ConstIterator GetBeginConstIterator() const NN_NOEXCEPT
        {
            return m_Index.GetBeginConstIterator();
        }

        ConstIterator GetLowerBoundConstIterator(const Key& key) const NN_NOEXCEPT
        {
            return m_Index.GetLowerBoundConstIterator(key);
        }

        int Count() const NN_NOEXCEPT
        {
            return m_Index.Count();
        }

        void FixIterator(Iterator* iterator, const SaveDataIndexerKey& key)  NN_NOEXCEPT
        {
            m_Index.FixIterator(iterator, key);
        }

    private:
        Index m_Index;
        Path m_ArchivePath;
        Path m_ArchiveTemporaryPath;
        MemoryResource* m_MemoryResource;

        size_t CalculateArchiveSize() const NN_NOEXCEPT
        {
            nn::kvdb::KeyValueArchiveSizeCalculator calculator;

            for (auto i = m_Index.GetBeginConstIterator(); !i.IsEnd(); i.Next())
            {
                calculator.AddEntry(sizeof(Key), i.GetSize());
            }

            return calculator.GetSize();
        }

        Result LoadFrom(const char* buffer, size_t size) NN_NOEXCEPT
        {
            nn::kvdb::KeyValueArchiveBufferReader reader(buffer, size);

            uint32_t entryCount;
            NN_RESULT_DO(reader.ReadEntryCount(&entryCount));

            for (uint32_t i = 0; i < entryCount; i++)
            {
                nn::kvdb::KeyValueArchiveBufferReader::KeyValueSize keyValueSize;
                NN_RESULT_DO(reader.GetKeyValueSize(&keyValueSize));

                auto valueSize = keyValueSize.valueSize;
                auto value = m_MemoryResource->allocate(valueSize);
                NN_RESULT_THROW_UNLESS(value, nn::kvdb::ResultAllocationMemoryFailed());

                bool isSuccess = false;
                NN_UTIL_SCOPE_EXIT
                {
                    if (!isSuccess)
                    {
                        m_MemoryResource->deallocate(value, valueSize);
                    };
                };

                Key key;
                NN_RESULT_DO(reader.ReadKeyValue(&key, sizeof(key), value, valueSize));
                NN_RESULT_DO(m_Index.PushBackRaw(key, value, valueSize));

                isSuccess = true;
            }

            NN_RESULT_SUCCESS;
        }

        Result ReadArchive(nn::kvdb::AutoBuffer* buffer) const NN_NOEXCEPT
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, m_ArchivePath, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            int64_t archiveSize;
            NN_RESULT_DO(fs::GetFileSize(&archiveSize, file));

            NN_RESULT_DO(buffer->Initialize(static_cast<size_t>(archiveSize)));
            NN_RESULT_DO(fs::ReadFile(file, 0, buffer->Get(), buffer->GetSize()));

            NN_RESULT_SUCCESS;
        }

        void SaveTo(nn::kvdb::KeyValueArchiveBufferWriter* writer) const NN_NOEXCEPT
        {
            writer->WriteHeader(static_cast<uint32_t>(m_Index.Count()));
            for (auto i = m_Index.GetBeginConstIterator(); !i.IsEnd(); i.Next())
            {
                auto& key = i.Get();
                writer->WriteEntry(&key, sizeof(key), i.GetPointer(), i.GetSize());
            }
        }

        Result CommitArchive(const void* buffer, size_t size, bool destructive) NN_NOEXCEPT
        {
            if (destructive)
            {
                fs::DeleteFile(m_ArchivePath);
                NN_RESULT_DO(fs::CreateFile(m_ArchivePath, size));
                {
                    fs::FileHandle file;
                    NN_RESULT_DO(fs::OpenFile(&file, m_ArchivePath, fs::OpenMode_Write));
                    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
                    NN_RESULT_DO(fs::WriteFile(file, 0, buffer, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
                }
            }
            else
            {
                // TODO: fs が強制 Create を実装したら対応する
                fs::DeleteFile(m_ArchiveTemporaryPath);
                NN_RESULT_DO(fs::CreateFile(m_ArchiveTemporaryPath, size));

                {
                    fs::FileHandle file;
                    NN_RESULT_DO(fs::OpenFile(&file, m_ArchiveTemporaryPath, fs::OpenMode_Write));
                    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
                    NN_RESULT_DO(fs::WriteFile(file, 0, buffer, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
                }

                // TODO: fs が atomic rename を実装したら対応する
                fs::DeleteFile(m_ArchivePath);
                NN_RESULT_DO(fs::RenameFile(m_ArchiveTemporaryPath, m_ArchivePath));
            }

            NN_RESULT_SUCCESS;
        }
    };

}}}
