﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <memory>
#include <nn/nn_Allocator.h>
#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fssrv/fssrv_SaveDataIndexerTypes.h>
#include <nn/os/os_Mutex.h>
#include <nn/fssrv/kvdb/fssrv_FlatMapKeyValueStore.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace fssrv {

    typedef nn::fssrv::kvdb::FlatMapKeyValueStore<SaveDataIndexerKey> SaveDataIndexDatabaseImpl;
    typedef nn::fssrv::kvdb::FlatMapKeyValueStore<SaveDataIndexerKey>::Iterator SaveDataIndexDatabaseIterator;

    class SaveDataInfoReaderImpl;
    class Reader;

    const size_t SaveDataIndexerInternalPathLengthMax = 48;

    class ISaveDataIndexer
    {
    public:
        virtual ~ISaveDataIndexer() NN_NOEXCEPT {};

        virtual Result Commit() NN_NOEXCEPT = 0;
        virtual Result Rollback() NN_NOEXCEPT = 0;
        virtual Result Reset() NN_NOEXCEPT = 0;

        virtual Result Publish(nn::fs::SaveDataId* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT = 0;
        virtual Result Get(SaveDataIndexerValue* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT = 0;
        virtual Result PutStaticSaveDataIdIndex(const SaveDataIndexerKey& key) NN_NOEXCEPT = 0;
        virtual bool IsRemainedReservedOnly() NN_NOEXCEPT = 0;

        virtual Result Delete(nn::fs::SaveDataId id) NN_NOEXCEPT = 0;
        virtual Result SetSpaceId(nn::fs::SaveDataId id, nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT = 0;
        virtual Result SetSize(nn::fs::SaveDataId id, int64_t size) NN_NOEXCEPT = 0;
        virtual Result SetState(nn::fs::SaveDataId id, SaveDataState state) NN_NOEXCEPT = 0;
        virtual Result GetKey(SaveDataIndexerKey* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT = 0;
        virtual Result GetValue(SaveDataIndexerValue* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT = 0;
        virtual Result SetValue(const SaveDataIndexerKey& key, const SaveDataIndexerValue& value) NN_NOEXCEPT = 0;
        virtual int GetIndexCount() NN_NOEXCEPT = 0;

        virtual Result OpenSaveDataInfoReader(std::shared_ptr<SaveDataInfoReaderImpl>* outValue) NN_NOEXCEPT = 0;
    };


    class SaveDataIndexer : public ISaveDataIndexer
    {
    private:
        class ReaderAccessor : public nn::util::IntrusiveListBaseNode<ReaderAccessor>, public nn::fs::detail::Newable
        {
        public:
            explicit ReaderAccessor(std::shared_ptr<Reader> accessor)
                : m_Accessor(accessor)
            {
            }
            std::shared_ptr<Reader> Lock() NN_NOEXCEPT
            {
                return m_Accessor.lock();
            }
            bool IsExpired() NN_NOEXCEPT
            {
                return m_Accessor.expired();
            }

        private:
            const std::weak_ptr<Reader> m_Accessor;
        };

    public:
        SaveDataIndexer(const char* systemSaveDataMountName, nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId systemSaveDataId, MemoryResource* memoryResource) NN_NOEXCEPT;
        ~SaveDataIndexer() NN_NOEXCEPT;

        virtual Result Commit() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Rollback() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Reset() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Publish(nn::fs::SaveDataId* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Get(SaveDataIndexerValue* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result PutStaticSaveDataIdIndex(const SaveDataIndexerKey& key) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsRemainedReservedOnly() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Delete(nn::fs::SaveDataId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetSpaceId(nn::fs::SaveDataId id, nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetSize(nn::fs::SaveDataId id, int64_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetState(nn::fs::SaveDataId id, SaveDataState state) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetKey(SaveDataIndexerKey* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetValue(SaveDataIndexerValue* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetValue(const SaveDataIndexerKey& key, const SaveDataIndexerValue& value) NN_NOEXCEPT NN_OVERRIDE;
        virtual int GetIndexCount() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result OpenSaveDataInfoReader(std::shared_ptr<SaveDataInfoReaderImpl>* outValue) NN_NOEXCEPT NN_OVERRIDE;

        SaveDataIndexDatabaseIterator GetBeginIterator() NN_NOEXCEPT;
        int GetHandle() NN_NOEXCEPT;
        std::unique_lock<os::Mutex> GetScopedLock() NN_NOEXCEPT;
        void FixIterator(SaveDataIndexDatabaseIterator* iterator, const SaveDataIndexerKey& key) NN_NOEXCEPT;
        void UnregisterReader() NN_NOEXCEPT;

    private:
        Result TryInitializeDatabase() NN_NOEXCEPT;
        Result TryLoadDatabase(bool isForce) NN_NOEXCEPT;
        Result UpdateValueBySaveDataId(nn::fs::SaveDataId id,  void* context, std::function<void(SaveDataIndexerValue* pValue, void* context)> delegate) NN_NOEXCEPT;
        void UpdateHandle() NN_NOEXCEPT;
        Result RegisterReader(std::shared_ptr<Reader> reader) NN_NOEXCEPT;
        Result FixReader(const SaveDataIndexerKey& key) NN_NOEXCEPT;

    private:

        char m_SystemSaveDataMountName[SaveDataIndexerInternalPathLengthMax];
        const nn::fs::SystemSaveDataId m_SystemSaveDataId;
        const nn::fs::SaveDataSpaceId m_SpaceId;
        MemoryResource* m_MemoryResource;

        SaveDataIndexDatabaseImpl m_Database;
        nn::os::Mutex m_Mutex;

        bool m_IsInitialized;
        bool m_IsLoaded;

        nn::fs::SaveDataId m_LastPublishedId;
        int m_CurrentHandleValue;

        nn::util::IntrusiveList<ReaderAccessor, util::IntrusiveListBaseNodeTraits<ReaderAccessor>> m_ReaderList;
    };

    // Key/Value を 1 エントリのみ保持し、永続化しないインデクサ
    class SaveDataIndexerLite : public ISaveDataIndexer
    {
    public:
        SaveDataIndexerLite() NN_NOEXCEPT;

        virtual Result Commit() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Rollback() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Reset() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Publish(nn::fs::SaveDataId* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result Get(SaveDataIndexerValue* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result PutStaticSaveDataIdIndex(const SaveDataIndexerKey& key) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsRemainedReservedOnly() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Delete(nn::fs::SaveDataId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetSpaceId(nn::fs::SaveDataId id, nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetSize(nn::fs::SaveDataId id, int64_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetState(nn::fs::SaveDataId id, SaveDataState state) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetKey(SaveDataIndexerKey* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetValue(SaveDataIndexerValue* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetValue(const SaveDataIndexerKey& key, const SaveDataIndexerValue& value) NN_NOEXCEPT NN_OVERRIDE;
        virtual int GetIndexCount() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result OpenSaveDataInfoReader(std::shared_ptr<SaveDataInfoReaderImpl>* outValue) NN_NOEXCEPT NN_OVERRIDE;

    private:
        nn::os::Mutex m_Mutex;
        nn::fs::SaveDataId m_LastPublishedId;
        nn::util::optional<SaveDataIndexerKey> m_Key;
        SaveDataIndexerValue m_Value;
    };

    void GenerateSaveDataInfo(fs::SaveDataInfo* pOutValue, const SaveDataIndexerKey& key, const SaveDataIndexerValue& value);
}}
