﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_PriorityPrivate.h>

namespace nn { namespace fssrv {
class DeferredProcessQueueForPriority;
}}

namespace nn { namespace fssrv {

enum class FileSystemProxyServerSessionType : int
{
    Any,
    Realtime,
    Background,
    Other
};

const int FileSystemProxyServerActiveSessionCount = 5;

class FileSystemProxyServerSessionResourceManager
{
    NN_DISALLOW_COPY(FileSystemProxyServerSessionResourceManager);
    NN_DISALLOW_MOVE(FileSystemProxyServerSessionResourceManager);

public:
    static const int RealtimeSessionCount = 1;
    static const int BackgroundSessionCount = 1;
    static const int OtherSessionCount = 3;
    static const int SessionCount = RealtimeSessionCount + BackgroundSessionCount + OtherSessionCount;

public:
    explicit FileSystemProxyServerSessionResourceManager(DeferredProcessQueueForPriority* pDeferredProcessQueueForPriority) NN_NOEXCEPT
        : m_SessionUsedByRealtimeCount(0),
          m_SessionUsedByBackgroundCount(0),
          m_SessionUsedByOtherCount(0),
          m_pDeferredProcessQueueForPriority(pDeferredProcessQueueForPriority)
    {
        NN_SDK_REQUIRES_NOT_NULL(pDeferredProcessQueueForPriority);
    }

public:
    Result Acquire(fs::PriorityRaw priority) NN_NOEXCEPT;

    bool Acquire(FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT;

    void Release() NN_NOEXCEPT;

    void GetSessionCountsForDebug(int* outRealtimeCount, int* outBackgroundCount, int* outOtherCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outRealtimeCount);
        NN_SDK_REQUIRES_NOT_NULL(outBackgroundCount);
        NN_SDK_REQUIRES_NOT_NULL(outOtherCount);

        *outRealtimeCount = m_SessionUsedByRealtimeCount;
        *outBackgroundCount = m_SessionUsedByBackgroundCount;
        *outOtherCount = m_SessionUsedByOtherCount;
    }

private:
    bool OnAcquire(FileSystemProxyServerSessionType* outValue, fs::PriorityRaw priority) NN_NOEXCEPT;

    bool OnAcquire(FileSystemProxyServerSessionType* outValue, FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT;

    void OnRelease(FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT;

private:
    std::atomic_int m_SessionUsedByRealtimeCount;
    std::atomic_int m_SessionUsedByBackgroundCount;
    std::atomic_int m_SessionUsedByOtherCount;
    DeferredProcessQueueForPriority* m_pDeferredProcessQueueForPriority;
};

}}
