﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssrv/fssrv_FileSystemProxyServerSessionResourceManager.h>
#include <nn/os/os_ConditionVariable.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace os {
struct MultiWaitHolderType;
}}

namespace nn { namespace fssrv {

    struct DeferredProcessEntryForDeviceError : public util::IntrusiveListBaseNode<DeferredProcessEntryForDeviceError>, public fs::detail::Newable
    {
        os::MultiWaitHolderType* pProcessHolder;
        Bit64 processId;
    };

    class DeferredProcessQueueForDeviceError
    {
        NN_DISALLOW_COPY(DeferredProcessQueueForDeviceError);
        NN_DISALLOW_MOVE(DeferredProcessQueueForDeviceError);
    public:
        DeferredProcessQueueForDeviceError() NN_NOEXCEPT;
        ~DeferredProcessQueueForDeviceError() NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        Result Push(const DeferredProcessEntryForDeviceError& entry) NN_NOEXCEPT;

        void Pop() NN_NOEXCEPT;
        void Pop(Bit64 processId) NN_NOEXCEPT;

        const DeferredProcessEntryForDeviceError& Front() NN_NOEXCEPT;
        const DeferredProcessEntryForDeviceError& Front(Bit64 processId) NN_NOEXCEPT;

        bool IsEmpty() NN_NOEXCEPT;
        bool IsEmpty(Bit64 processId) NN_NOEXCEPT;

        size_t Count() NN_NOEXCEPT;
        size_t Count(Bit64 processId) NN_NOEXCEPT;

    private:
        typedef util::IntrusiveList<DeferredProcessEntryForDeviceError, util::IntrusiveListBaseNodeTraits<DeferredProcessEntryForDeviceError>> DeferredProcessEntryQueueType;
        DeferredProcessEntryQueueType m_List;
        os::Mutex m_Mutex;
    };

    class DeferredProcessEntryForPriority : public util::IntrusiveListBaseNode<DeferredProcessEntryForPriority>, public fs::detail::Newable
    {
    public:
        DeferredProcessEntryForPriority(os::MultiWaitHolderType* pProcessHolder, FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT
            : m_pProcessHolder(pProcessHolder),
              m_SessionType(sessionType)
        {
            NN_SDK_REQUIRES_NOT_NULL(pProcessHolder);
            NN_SDK_REQUIRES(
                sessionType == FileSystemProxyServerSessionType::Realtime
                || sessionType == FileSystemProxyServerSessionType::Other
                || sessionType == FileSystemProxyServerSessionType::Background
            );
        }

    public:
        os::MultiWaitHolderType* GetProcessHolder() const NN_NOEXCEPT
        {
            return m_pProcessHolder;
        }

        FileSystemProxyServerSessionType GetSessionType() const NN_NOEXCEPT
        {
            return m_SessionType;
        }

    private:
        os::MultiWaitHolderType* m_pProcessHolder;
        FileSystemProxyServerSessionType m_SessionType;
    };

    class DeferredProcessQueueForPriority
    {
        NN_DISALLOW_COPY(DeferredProcessQueueForPriority);
        NN_DISALLOW_MOVE(DeferredProcessQueueForPriority);

    public:
        // WaitConditionCount + スレッド数がキューの個数の上限となる
        static const size_t WaitConditionCount = 10;

    public:
        DeferredProcessQueueForPriority() NN_NOEXCEPT;
        ~DeferredProcessQueueForPriority() NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        Result Push(os::MultiWaitHolderType* pProcessHolder, FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT;

        bool Pop(os::MultiWaitHolderType** outValue, FileSystemProxyServerSessionResourceManager* pSessionResourceManager) NN_NOEXCEPT;

        bool HasAny(FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT;

        size_t Count() NN_NOEXCEPT;

        void Wait() NN_NOEXCEPT;

    private:
        typedef util::IntrusiveList<DeferredProcessEntryForPriority, util::IntrusiveListBaseNodeTraits<DeferredProcessEntryForPriority>> DeferredProcessEntryQueueType;

    private:
        DeferredProcessEntryQueueType m_List;
        os::Mutex m_Mutex;
        os::ConditionVariable m_ConditionalVariable;
    };

}}
