﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/fssrv/fssrv_DeferredProcessQueue.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/sf/sf_Result.h>

namespace nn { namespace fssrv {

    typedef void (*NotifyProcessDeferredFunction)(Bit64 processId);

    template<typename FileSystemProxyServer, NotifyProcessDeferredFunction NotifyProcessDeferred>
    class DeferredProcessManager
    {
        NN_DISALLOW_COPY(DeferredProcessManager);
        NN_DISALLOW_MOVE(DeferredProcessManager);

    public:
        static const uintptr_t InvokeTag = 3;

    public:
        DeferredProcessManager() NN_NOEXCEPT
            : m_InvokeDeferredProcessMutexForDeviceError(false),
              m_IsInvokeDeferredProcessEventLinked(false),
              m_IsInitialized(false)
        {
        }

        ~DeferredProcessManager() NN_NOEXCEPT
        {
            Finalize();
        }

    public:
        void Initialize() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(!m_IsInitialized);
            os::InitializeEvent(&m_InvokeDeferredProcessEvent, false, os::EventClearMode_ManualClear);
            os::InitializeMultiWaitHolder(&m_InvokeDeferredProcessEventHolder, &m_InvokeDeferredProcessEvent);
            os::SetMultiWaitHolderUserData(&m_InvokeDeferredProcessEventHolder, InvokeTag);
            m_IsInitialized = true;
        }

        void Finalize() NN_NOEXCEPT
        {
            if( m_IsInitialized )
            {
                m_DeferredProcessQueueForDeviceError.Finalize();
                m_DeferredProcessQueueForPriority.Finalize();
                os::FinalizeMultiWaitHolder(&m_InvokeDeferredProcessEventHolder);
                os::FinalizeEvent(&m_InvokeDeferredProcessEvent);
                m_IsInitialized = false;
            }
        }

        void QueueDeferredProcess(FileSystemProxyServer* pFileSystemProxyServerManager, os::MultiWaitHolderType* pDeferredProcess) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            NN_SDK_REQUIRES_NOT_NULL(pFileSystemProxyServerManager);
            NN_SDK_REQUIRES_NOT_NULL(pDeferredProcess);

            // 遅延されたオブジェクトをキューイング
            bool isDeferredByDeviceError = QueueDeferredProcessForDeviceError(pDeferredProcess);
            bool isDeferredByPriority = QueueDeferredProcessForPriority(pFileSystemProxyServerManager, pDeferredProcess);
            NN_ABORT_UNLESS(isDeferredByDeviceError || isDeferredByPriority);
        }

        void InvokeDeferredProcess(FileSystemProxyServer* pFileSystemProxyServerManager, FileSystemProxyServerSessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            NN_SDK_REQUIRES_NOT_NULL(pFileSystemProxyServerManager);
            NN_SDK_REQUIRES_NOT_NULL(pSessionResourceManager);

            for( ; ; )
            {
                Bit64 processIdToInvoke = fssystem::ServiceContext::InvalidProcessId;
                bool isRequestedToInvokeDeferredForDeviceError = fssystem::GetServiceContext()->GetProcessIdRequestedToInvokeDeferred(&processIdToInvoke);
                bool isAnyInvoked = false;

                fssystem::GetServiceContext()->ResetDeferredProcessContext(false);

                if( isRequestedToInvokeDeferredForDeviceError )
                {
                    // 遅延されたオブジェクトの解除要求が来ていたら同じスレッドで引き続き処理する
                    InvokeDeferredProcessForDeviceError(pFileSystemProxyServerManager, pSessionResourceManager, processIdToInvoke);
                    isAnyInvoked = true;
                }

                if( InvokeDeferredProcessForPriority(pFileSystemProxyServerManager, pSessionResourceManager) )
                {
                    isAnyInvoked = true;
                }

                if( !isAnyInvoked )
                {
                    break;
                }
            }

            os::SignalEvent(&m_InvokeDeferredProcessEvent);
        }

        void ClearInvokeDeferredProcessEvent() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            m_IsInvokeDeferredProcessEventLinked = false;
            os::ClearEvent(&m_InvokeDeferredProcessEvent);
        }

        DeferredProcessQueueForDeviceError* GetDeferredProcessQueueForDeviceError() NN_NOEXCEPT
        {
            return &m_DeferredProcessQueueForDeviceError;
        }

        DeferredProcessQueueForPriority* GetDeferredProcessQueueForPriority() NN_NOEXCEPT
        {
            return &m_DeferredProcessQueueForPriority;
        }

    private:
        bool QueueDeferredProcessForDeviceError(os::MultiWaitHolderType* pDeferredProcess) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            NN_SDK_REQUIRES_NOT_NULL(pDeferredProcess);

            Bit64 processIdToBeQueued = fssystem::ServiceContext::InvalidProcessId;
            bool isDeferred = fssystem::GetServiceContext()->IsProcessDeferredByDeviceError(&processIdToBeQueued);

            if( isDeferred )
            {
                DeferredProcessEntryForDeviceError entry;
                {
                    entry.processId = processIdToBeQueued;
                    entry.pProcessHolder = pDeferredProcess;
                }
                std::lock_guard<os::Mutex> scopedLock(m_InvokeDeferredProcessMutexForDeviceError);
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_DeferredProcessQueueForDeviceError.Push(entry));
                NotifyProcessDeferred(processIdToBeQueued);
            }

            return isDeferred;
        }

        void InvokeDeferredProcessForDeviceError(FileSystemProxyServer* pFileSystemProxyServerManager, FileSystemProxyServerSessionResourceManager* pSessionResourceManager, Bit64 processId) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            NN_SDK_REQUIRES_NOT_NULL(pFileSystemProxyServerManager);
            NN_SDK_REQUIRES_NOT_NULL(pSessionResourceManager);

            // TORIAEZU: 遅延された処理の再試行は 1 スレッドずつ行う
            std::lock_guard<os::Mutex> scopedLock(m_InvokeDeferredProcessMutexForDeviceError);
            while( !m_DeferredProcessQueueForDeviceError.IsEmpty(processId) )
            {
                fssystem::GetServiceContext()->ResetDeferredProcessContext(false);
                pSessionResourceManager->Acquire(FileSystemProxyServerSessionType::Any);

                auto& entry = m_DeferredProcessQueueForDeviceError.Front(processId);
                auto result = pFileSystemProxyServerManager->ProcessInvokeRequestWithDefer(entry.pProcessHolder);
                if( nn::sf::ResultProcessDeferred::Includes(result) )
                {
                    NotifyProcessDeferred(processId);
                    return;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                m_DeferredProcessQueueForDeviceError.Pop(processId);
            }
        }

        bool QueueDeferredProcessForPriority(FileSystemProxyServer* pFileSystemProxyServerManager, os::MultiWaitHolderType* pDeferredProcess) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            NN_SDK_REQUIRES_NOT_NULL(pFileSystemProxyServerManager);
            NN_SDK_REQUIRES_NOT_NULL(pDeferredProcess);

            int proritySessionTypeToBeQueued = fssystem::ServiceContext::InvalidPrioritySessionType;
            bool isDeferred = fssystem::GetServiceContext()->IsProcessDeferredByPriority(&proritySessionTypeToBeQueued);

            if( isDeferred )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_DeferredProcessQueueForPriority.Push(
                    pDeferredProcess,
                    static_cast<FileSystemProxyServerSessionType>(proritySessionTypeToBeQueued)
                ));

                if( !m_IsInvokeDeferredProcessEventLinked.exchange(true) )
                {
                    pFileSystemProxyServerManager->AddUserWaitHolder(&m_InvokeDeferredProcessEventHolder);
                }

                m_DeferredProcessQueueForPriority.Wait();
            }

            return isDeferred;
        }

        bool InvokeDeferredProcessForPriority(FileSystemProxyServer* pFileSystemProxyServerManager, FileSystemProxyServerSessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            NN_SDK_REQUIRES_NOT_NULL(pFileSystemProxyServerManager);
            NN_SDK_REQUIRES_NOT_NULL(pSessionResourceManager);

            os::MultiWaitHolderType* pDeferred;
            if( m_DeferredProcessQueueForPriority.Pop(&pDeferred, pSessionResourceManager) )
            {
                fssystem::GetServiceContext()->ResetDeferredProcessContext(true);

                auto result = pFileSystemProxyServerManager->ProcessInvokeRequestWithDefer(pDeferred);
                if( nn::sf::ResultProcessDeferred::Includes(result) )
                {
                    // Pop で優先度用セッションを確保している。このため、ここにくるのは DeviceError のものだけ。
                    // キューを移し、次の処理へ。
                    bool isDeferred = QueueDeferredProcessForDeviceError(pDeferred);
                    NN_ABORT_UNLESS(isDeferred);
                }

                return true;
            }
            else
            {
                return false;
            }
        }

    private:
        DeferredProcessQueueForDeviceError m_DeferredProcessQueueForDeviceError;

        os::Mutex m_InvokeDeferredProcessMutexForDeviceError;
        DeferredProcessQueueForPriority m_DeferredProcessQueueForPriority;

        std::atomic_bool m_IsInvokeDeferredProcessEventLinked;
        os::EventType m_InvokeDeferredProcessEvent;
        os::MultiWaitHolderType m_InvokeDeferredProcessEventHolder;

        bool m_IsInitialized;
    };

}}
