﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dd.h>
#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/nn_Allocator.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>

namespace nn { namespace fssrv { namespace fscreator {

    class BuiltInStorageFileSystemCreator : public IBuiltInStorageFileSystemCreator
    {
        NN_DISALLOW_COPY(BuiltInStorageFileSystemCreator);
        NN_DISALLOW_MOVE(BuiltInStorageFileSystemCreator);

    public:
        BuiltInStorageFileSystemCreator(
            IBuiltInStorageCreator* pBisCreator,
            IMemoryStorageCreator* pMemoryStorageCreator,
            IFatFileSystemCreator* pFatFsCreator,
            ISdCardProxyFileSystemCreator* pSdCardProxyFileSystemCreator,
            detail::FatFileSystemCacheManager* pFatFileSystemCacheManager
            ) NN_NOEXCEPT;

        virtual Result Create(std::shared_ptr<fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath, nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;

        virtual bool IsSignedSystemPartitionOnSdCardValid() NN_NOEXCEPT NN_OVERRIDE;
        virtual void SetSdCardPortReady() NN_NOEXCEPT NN_OVERRIDE;
        virtual void Initialize(fs::FileSystemProxyErrorInfo* pErrorInfo, os::Mutex* pErrorInfoMutex) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result SetBisRoot(nn::fs::BisPartitionId id, const char* rootPath) NN_NOEXCEPT NN_OVERRIDE;

    protected:
        // fat on memory を生成し、 pStorage を fat として開いた内容をコピーして返す
        Result CreateAndLoadFatFileSystemOnMemory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, std::shared_ptr<nn::fs::IStorage> pStorage, fs::BisPartitionId bisId, fscreator::IMemoryStorageCreator::MemoryStorageId memoryStorageId, Result resultForInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT;

        Result CreateFatFileSystemOnMemory(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, fscreator::IMemoryStorageCreator::MemoryStorageId id) NN_NOEXCEPT;
        Result CreateFatFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, std::shared_ptr<nn::fs::IStorage> pStorage, fs::BisPartitionId id, Result resultForInvalidFatFormat, Result resultUsableSpaceNotEnough) NN_NOEXCEPT;
        virtual Result CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT;

    protected:
        IBuiltInStorageCreator* const m_pBisCreator;
        IMemoryStorageCreator* const m_pMemoryStorageCreator;
        IFatFileSystemCreator* const m_pFatFsCreator;
        ISdCardProxyFileSystemCreator* const m_pSdCardProxyFileSystemCreator;
        detail::FatFileSystemCacheManager* const m_pFatFileSystemCacheManager;

        bool m_IsSignedSystemPartitionOnSdCardValid = false;
        bool m_IsSdCardPortReady = false;

        fs::FileSystemProxyErrorInfo* m_pErrorInfo;
        os::Mutex* m_pErrorInfoMutex;
    };


    class BuiltInStorageFileSystemCreatorSspOnBis : public BuiltInStorageFileSystemCreator
    {
    public:
        BuiltInStorageFileSystemCreatorSspOnBis(
            IBuiltInStorageCreator* pBisCreator,
            IMemoryStorageCreator* pMemoryStorageCreator,
            IFatFileSystemCreator* pFatFsCreator,
            ISdCardProxyFileSystemCreator* pSdCardProxyFileSystemCreator,
            detail::FatFileSystemCacheManager* pFatFileSystemCacheManager
        ) NN_NOEXCEPT
            : BuiltInStorageFileSystemCreator(pBisCreator, pMemoryStorageCreator, pFatFsCreator, pSdCardProxyFileSystemCreator, pFatFileSystemCacheManager)
        {}

    private:
        virtual Result CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
    };


    class BuiltInStorageFileSystemCreatorSspOnSd : public BuiltInStorageFileSystemCreator
    {
    public:
        BuiltInStorageFileSystemCreatorSspOnSd(
            IBuiltInStorageCreator* pBisCreator,
            IMemoryStorageCreator* pMemoryStorageCreator,
            IFatFileSystemCreator* pFatFsCreator,
            ISdCardProxyFileSystemCreator* pSdCardProxyFileSystemCreator,
            detail::FatFileSystemCacheManager* pFatFileSystemCacheManager
        ) NN_NOEXCEPT
            : BuiltInStorageFileSystemCreator(pBisCreator, pMemoryStorageCreator, pFatFsCreator, pSdCardProxyFileSystemCreator, pFatFileSystemCacheManager)
        {}

    private:
        virtual Result CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
    };


    class BuiltInStorageFileSystemCreatorSystemPartitionOnMemoryBase : public BuiltInStorageFileSystemCreator
    {
    public:
        BuiltInStorageFileSystemCreatorSystemPartitionOnMemoryBase(
            IBuiltInStorageCreator* pBisCreator,
            IMemoryStorageCreator* pMemoryStorageCreator,
            IFatFileSystemCreator* pFatFsCreator,
            ISdCardProxyFileSystemCreator* pSdCardProxyFileSystemCreator,
            detail::FatFileSystemCacheManager* pFatFileSystemCacheManager,
            nn::dd::PhysicalAddress systemPartitionOnMemoryPhysicalAddress,
            size_t systemPartitionOnMemorySize
        ) NN_NOEXCEPT
            : BuiltInStorageFileSystemCreator(pBisCreator, pMemoryStorageCreator, pFatFsCreator, pSdCardProxyFileSystemCreator, pFatFileSystemCacheManager)
            , m_SystemPartitionOnMemoryPhysicalAddress(systemPartitionOnMemoryPhysicalAddress)
            , m_SystemPartitionOnMemorySize(systemPartitionOnMemorySize)
        {}

    private:
        virtual Result CreateCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutValue, nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;

        const nn::dd::PhysicalAddress m_SystemPartitionOnMemoryPhysicalAddress;
        const size_t                  m_SystemPartitionOnMemorySize;
    };

    template <uint64_t Address, int Size>
    class BuiltInStorageFileSystemCreatorSystemPartitionOnMemory : public BuiltInStorageFileSystemCreatorSystemPartitionOnMemoryBase
    {
    public:
        BuiltInStorageFileSystemCreatorSystemPartitionOnMemory(
            IBuiltInStorageCreator* pBisCreator,
            IMemoryStorageCreator* pMemoryStorageCreator,
            IFatFileSystemCreator* pFatFsCreator,
            ISdCardProxyFileSystemCreator* pSdCardProxyFileSystemCreator,
            detail::FatFileSystemCacheManager* pFatFileSystemCacheManager
        ) NN_NOEXCEPT
            : BuiltInStorageFileSystemCreatorSystemPartitionOnMemoryBase(pBisCreator, pMemoryStorageCreator, pFatFsCreator, pSdCardProxyFileSystemCreator, pFatFileSystemCacheManager, Address, Size)
        {}
    };


}}}
