﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/nn_Allocator.h>
#include <nn/os/os_Mutex.h>
#include <nn/crypto/crypto_Sha256Generator.h>

namespace nn { namespace fssrv { namespace fscreator {

    enum class BisPartitionEncryptionKeyId
    {
        Calibration = 0,
        SafeMode,
        UserSystemProperEncryption,
        UserSystem,

        Count,
    };
    const int BisPartitionEncryptionKeyIdCount = static_cast<int>(BisPartitionEncryptionKeyId::Count);


    class BuiltInStorageCreator : public IBuiltInStorageCreator
    {
        NN_DISALLOW_COPY(BuiltInStorageCreator);
        NN_DISALLOW_MOVE(BuiltInStorageCreator);
    private:
        static const int BisStorageInstanceCount = 16;

    public:

        struct Key
        {
            char value[16];
        };
        NN_STATIC_ASSERT(std::is_pod<Key>::value);

        typedef void (*GetEncryptionKeyFunction)(Key* pOutValueKey, int keyCount, BisPartitionEncryptionKeyId id, int keyGeneration);

        struct Configuration
        {
            bool enablePackage2HashVerification;
            GetEncryptionKeyFunction getKeyFunc;
            IMemoryStorageCreator* pMemoryStorageCreator;
            const unsigned char* pSignedSystemPartitionSignKeyPublicModulus;
            char package2Hash[crypto::Sha256Generator::HashSize];

        public:
            static const int SignedSystemPartitionSignKeyPublicModulusSize = 256;
        };
        NN_STATIC_ASSERT(std::is_pod<Configuration>::value);

    public:
        explicit BuiltInStorageCreator(const Configuration& config) NN_NOEXCEPT;
        virtual Result Create(std::shared_ptr<fs::IStorage>* outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual Result InvalidateCache() NN_NOEXCEPT NN_OVERRIDE;
        virtual ~BuiltInStorageCreator() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result LoadAndVerifySignedSystemPartition(std::unique_ptr<nn::fs::IStorage>* outValue, std::unique_ptr<nn::fs::IStorage>&& baseStorage) NN_NOEXCEPT NN_OVERRIDE;

    protected:
        virtual nn::fs::BisPartitionId RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT;
        virtual bool IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT;

    protected:
        nn::fs::IStorage** GetCachedPartitionStorage(nn::fs::BisPartitionId id) NN_NOEXCEPT;
        void InvalidateCachePartition(nn::fs::BisPartitionId id) NN_NOEXCEPT;
        Result OpenAndCacheBuiltInStoragePartitionInternal(nn::fs::IStorage** outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT;
        Result OpenAndCacheBuiltInStoragePartition(nn::fs::IStorage** outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT;
        Result AddEncryptionLayer(std::unique_ptr<nn::fs::IStorage>* outValue, std::unique_ptr<nn::fs::IStorage>&& pBaseStorage, nn::fs::BisPartitionId id) NN_NOEXCEPT;

    private:
        Configuration m_Config;
        nn::os::Mutex m_CacheMutex;
        nn::fs::IStorage* m_CachedBisPartitions[BisStorageInstanceCount];


        friend class BuiltInStorageServiceTestBase;
    };


    class BuiltInStorageCreatorSspOnSd : public BuiltInStorageCreator
    {
    public:
        explicit BuiltInStorageCreatorSspOnSd(const Configuration& config) NN_NOEXCEPT
            : BuiltInStorageCreator(config)
        {}

    private:
        virtual nn::fs::BisPartitionId RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
    };


    class BuiltInStorageCreatorRedirectSafe : public BuiltInStorageCreator
    {
    public:
        explicit BuiltInStorageCreatorRedirectSafe(const Configuration& config) NN_NOEXCEPT
            : BuiltInStorageCreator(config)
        {}

    private:
        virtual nn::fs::BisPartitionId RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
    };


    class BuiltInStorageCreatorSspOnSystem : public BuiltInStorageCreator
    {
    public:
        explicit BuiltInStorageCreatorSspOnSystem(const Configuration& config) NN_NOEXCEPT
            : BuiltInStorageCreator(config)
        {}

    private:
        virtual nn::fs::BisPartitionId RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
    };


    class BuiltInStorageCreatorSspOnSafe : public BuiltInStorageCreator
    {
    public:
        explicit BuiltInStorageCreatorSspOnSafe(const Configuration& config) NN_NOEXCEPT
            : BuiltInStorageCreator(config)
        {}

    private:
        virtual nn::fs::BisPartitionId RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT NN_OVERRIDE;
    };


}}}
