﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs { namespace fsa {

class IFileSystem;

class ICommonMountNameGenerator
{
public:
    virtual Result GenerateCommonMountName(char* name, size_t nameSize) = 0;
    virtual ~ICommonMountNameGenerator()
    {
    };
};

/**
* @brief    IFileSystem を FS ライブラリに登録します。
*
* @details  指定したマウント名で IFileSystem を実装したクラスのインスタンスを FS ライブラリに登録します。
*           以後、FS ライブラリの関数で当該マウント名を持つパスが指定されると、登録されたインスタンスの
*           関数が実体パスを引数に呼び出されます。登録されたインスタンスの所有権は FS ライブラリに委譲されます。
*           FS ライブラリがインスタンスを破棄する際に delete が呼び出されます。
*/
Result Register(const char* name, std::unique_ptr<IFileSystem>&& fileSystem) NN_NOEXCEPT;


/**
* @brief    IFileSystem を FS ライブラリに登録します。
*
* @details  ICommonMountNameGenerator と共に IFileSystem を FS ライブラリに登録します。
*           登録した ICommonMountNameGenerator は ConvertToFsCommonPath() API 呼出し時に利用されます。
*/
Result Register(const char* name, std::unique_ptr<IFileSystem>&& fileSystem, std::unique_ptr<ICommonMountNameGenerator>&& mountNameGenerator) NN_NOEXCEPT;

/**
* @brief    IFileSystem を FS ライブラリに登録します。
*
* @details  ICommonMountNameGenerator と共に IFileSystem を FS ライブラリに登録します。
*           登録した ICommonMountNameGenerator は ConvertToFsCommonPath() API 呼出し時に利用されます。
*           makeFileDataCacheAvailable == true の場合、この IFileSystem でファイルデータキャッシュが利用可能となります。
*           makePathBasedFileDataCacheAvailable == true の場合、この IFileSystem で個別ファイルデータキャッシュが利用可能となります。
*           isMultiCommitSupported == true の場合、複数コミットをサポートしたファイルシステムとして登録します。
*/
Result Register(const char* name, std::unique_ptr<IFileSystem>&& fileSystem, std::unique_ptr<ICommonMountNameGenerator>&& mountNameGenerator, bool makeFileDataCacheAvailable, bool makePathBasedFileDataCacheAvailable, bool isMultiCommitSupported) NN_NOEXCEPT;

/**
* @brief    指定した名前の IFileSystem 登録を FS ライブラリから解除します。
*
* @details  登録が解除されると delete が呼び出されてインスタンスが破棄されます。
*           nn::fs::Unmount が呼び出された時も同様の挙動になります。
*/
void Unregister(const char* name) NN_NOEXCEPT;

}}}
